/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.statistics;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.Logger;
import org.esa.snap.core.datamodel.GeoCoding;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.statistics.BandConfiguration;
import org.esa.snap.statistics.ProductValidator;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

public class ProductValidatorTest {
    private final int oneSecond = 1000;
    private ProductData.UTC startDate;
    private ProductData.UTC endDate;
    private List<BandConfiguration> bandConfigurations;
    private ProductValidator productValidator;
    private Logger loggerMock;
    private Product product;

    @Before
    public void setUp() throws Exception {
        this.loggerMock = (Logger)Mockito.mock(Logger.class);
        this.bandConfigurations = new ArrayList<BandConfiguration>();
        this.startDate = ProductData.UTC.parse((String)"2012-05-21 00:00:00", (String)"yyyy-MM-dd HH:mm:ss");
        this.endDate = ProductData.UTC.parse((String)"2012-11-08 00:00:00", (String)"yyyy-MM-dd HH:mm:ss");
        this.productValidator = new ProductValidator(this.bandConfigurations, this.startDate, this.endDate, this.loggerMock);
        this.product = (Product)Mockito.mock(Product.class);
        Mockito.when((Object)this.product.getSceneGeoCoding()).thenReturn(Mockito.mock(GeoCoding.class));
        Mockito.when((Object)this.product.getStartTime()).thenReturn((Object)this.startDate);
        Mockito.when((Object)this.product.getEndTime()).thenReturn((Object)this.endDate);
    }

    @Test
    public void testIsValid_IfIsEntirelyInTimeRange() {
        boolean valid = this.productValidator.isValid(this.product);
        Assert.assertEquals((Object)true, (Object)valid);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.loggerMock});
    }

    @Test
    public void testIsInvalid_IfProductDoesNotContainAGeoCoding() {
        Mockito.when((Object)this.product.getSceneGeoCoding()).thenReturn(null);
        Mockito.when((Object)this.product.getName()).thenReturn((Object)"No Geocoding");
        boolean valid = this.productValidator.isValid(this.product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this.loggerMock)).warning("Product skipped. The product 'No Geocoding' does not contain a geo coding.");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.loggerMock});
    }

    @Test
    public void testIsInvalid_IfIsNotEntirelyInTimeRange_beforeTimeRange() {
        Mockito.when((Object)this.product.getStartTime()).thenReturn((Object)this.before(this.startDate));
        Mockito.when((Object)this.product.getEndTime()).thenReturn((Object)this.endDate);
        Mockito.when((Object)this.product.getName()).thenReturn((Object)"OutOfDateRange_before");
        boolean valid = this.productValidator.isValid(this.product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this.loggerMock)).warning("Product skipped. The product 'OutOfDateRange_before' is not inside the date range from 21-MAY-2012 00:00:00.000000 to 08-NOV-2012 00:00:00.000000");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.loggerMock});
    }

    @Test
    public void testIsInvalid_IfIsNotEntirelyInTimeRange_afterTimeRange() {
        Mockito.when((Object)this.product.getStartTime()).thenReturn((Object)this.startDate);
        Mockito.when((Object)this.product.getEndTime()).thenReturn((Object)this.after(this.endDate));
        Mockito.when((Object)this.product.getName()).thenReturn((Object)"OutOfDateRange_after");
        boolean valid = this.productValidator.isValid(this.product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this.loggerMock)).warning("Product skipped. The product 'OutOfDateRange_after' is not inside the date range from 21-MAY-2012 00:00:00.000000 to 08-NOV-2012 00:00:00.000000");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.loggerMock});
    }

    @Test
    public void testProductValidatorThatHasOnlyStartTime() throws Exception {
        this.productValidator = new ProductValidator(this.bandConfigurations, this.startDate, null, this.loggerMock);
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.startDate, this.after(this.endDate))));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.startDate, this.before(this.endDate))));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.startDate, this.endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.startDate, null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(null, null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(null, this.endDate)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(this.before(this.startDate), null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(this.before(this.startDate), this.endDate)));
    }

    @Test
    public void testProductValidatorThatHasOnlyEndTime() throws Exception {
        this.productValidator = new ProductValidator(this.bandConfigurations, null, this.endDate, this.loggerMock);
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(null, this.endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.after(this.startDate), this.endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.before(this.startDate), this.endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.startDate, this.endDate)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(null, null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(this.startDate, null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(null, this.after(this.endDate))));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(this.startDate, this.after(this.endDate))));
    }

    @Test
    public void testProductValidatorThatHasNoTimes() throws Exception {
        this.productValidator = new ProductValidator(this.bandConfigurations, null, null, this.loggerMock);
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(null, this.endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.after(this.startDate), this.endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.before(this.startDate), this.endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.startDate, this.endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(null, null)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.startDate, null)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(null, this.after(this.endDate))));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.startDate, this.after(this.endDate))));
    }

    private boolean isValid(Product product) {
        return this.productValidator.isValid(product);
    }

    private Product configureProductTimes(ProductData.UTC startDate, ProductData.UTC endDate) {
        Mockito.when((Object)this.product.getStartTime()).thenReturn((Object)startDate);
        Mockito.when((Object)this.product.getEndTime()).thenReturn((Object)endDate);
        return this.product;
    }

    @Test
    public void testIsValid_IfProductHasAllBandsNeededInBandConfigurations() {
        BandConfiguration bandConfiguration = new BandConfiguration();
        bandConfiguration.sourceBandName = "band-1";
        this.bandConfigurations.add(bandConfiguration);
        Mockito.when((Object)this.product.containsBand("band-1")).thenReturn((Object)true);
        boolean valid = this.productValidator.isValid(this.product);
        Assert.assertEquals((Object)true, (Object)valid);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.loggerMock});
    }

    @Test
    public void testIsInvalid_IfProductDoesNotContainAllBandsNeededInBandConfigurations() {
        BandConfiguration bandConfiguration = new BandConfiguration();
        bandConfiguration.sourceBandName = "band-1";
        this.bandConfigurations.add(bandConfiguration);
        Mockito.when((Object)this.product.containsBand("band-1")).thenReturn((Object)false);
        Mockito.when((Object)this.product.getName()).thenReturn((Object)"InvalidProduct");
        boolean valid = this.productValidator.isValid(this.product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this.loggerMock)).warning("Product skipped. The product 'InvalidProduct' does not contain the band 'band-1'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.loggerMock});
    }

    @Test
    public void testIsInvalid_IfProductCanNotResolveTheExpressionNeededInBandConfigurations() {
        BandConfiguration bandConfiguration = new BandConfiguration();
        bandConfiguration.expression = "band_1 + 4";
        this.bandConfigurations.add(bandConfiguration);
        Mockito.when((Object)this.product.isCompatibleBandArithmeticExpression("band_1 + 4")).thenReturn((Object)false);
        Mockito.when((Object)this.product.getName()).thenReturn((Object)"InvalidProduct");
        boolean valid = this.productValidator.isValid(this.product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this.loggerMock)).warning("Product skipped. The product 'InvalidProduct' can not resolve the band arithmetic expression 'band_1 + 4'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.loggerMock});
    }

    @Test
    public void testIsInvalid_IfProductAlreadyContainsBandWithExpressionName() {
        BandConfiguration bandConfiguration = new BandConfiguration();
        bandConfiguration.expression = "band_1 + 4";
        this.bandConfigurations.add(bandConfiguration);
        Mockito.when((Object)this.product.isCompatibleBandArithmeticExpression("band_1 + 4")).thenReturn((Object)true);
        Mockito.when((Object)this.product.getName()).thenReturn((Object)"InvalidProduct");
        Mockito.when((Object)this.product.containsBand("band_1_+_4")).thenReturn((Object)true);
        boolean valid = this.productValidator.isValid(this.product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this.loggerMock)).warning("Product skipped. The product 'InvalidProduct' already contains a band 'band_1_+_4'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.loggerMock});
    }

    private ProductData.UTC before(ProductData.UTC date) {
        long time = date.getAsDate().getTime() - 1000L;
        return ProductData.UTC.create((Date)new Date(time), (long)0L);
    }

    private ProductData.UTC after(ProductData.UTC date) {
        long time = date.getAsDate().getTime() + 1000L;
        return ProductData.UTC.create((Date)new Date(time), (long)0L);
    }
}

