/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.statistics;

import java.io.File;
import java.io.IOException;
import java.util.logging.Logger;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.core.gpf.OperatorException;
import org.esa.snap.statistics.ProductLoader;
import org.esa.snap.statistics.ProductLoop;
import org.esa.snap.statistics.ProductValidator;
import org.esa.snap.statistics.StatisticComputer;
import org.hamcrest.Matcher;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InOrder;
import org.mockito.Mockito;
import org.mockito.internal.matchers.EndsWith;
import org.mockito.verification.VerificationMode;

public class ProductLoopTest {
    private StatisticComputer statisticComputerMock;
    private ProductLoop productLoop;
    private ProductLoader productLoaderMock;
    private Logger loggerMock;
    private ProductData.UTC startDate;
    private ProductData.UTC endDate;
    private Product validProductMock1;
    private Product validProductMock2;
    private ProductValidator validatorMock;

    @Before
    public void setUp() throws Exception {
        this.statisticComputerMock = (StatisticComputer)Mockito.mock(StatisticComputer.class);
        this.productLoaderMock = (ProductLoader)Mockito.mock(ProductLoader.class);
        this.loggerMock = (Logger)Mockito.mock(Logger.class);
        this.startDate = ProductData.UTC.parse((String)"22-MAR-2008 00:00:00");
        this.endDate = ProductData.UTC.parse((String)"15-SEP-2008 00:00:00");
        this.validProductMock1 = this.createTimeValidProductMock(4, 3);
        this.validProductMock2 = this.createTimeValidProductMock(7, 2);
        this.validatorMock = (ProductValidator)Mockito.mock(ProductValidator.class);
        Mockito.when((Object)this.validatorMock.isValid((Product)Mockito.any(Product.class))).thenReturn((Object)true);
        this.productLoop = new ProductLoop(this.productLoaderMock, this.validatorMock, this.statisticComputerMock, this.loggerMock);
    }

    @Test
    public void testThatAlreadyLoadedProductsShouldNotBeDisposedAfterStatisticComputation() {
        Product[] loadedProducts = new Product[]{this.validProductMock1, this.validProductMock2};
        this.productLoop.loop(loadedProducts, new File[0]);
        InOrder inOrder = Mockito.inOrder((Object[])new Object[]{this.statisticComputerMock});
        ((StatisticComputer)inOrder.verify((Object)this.statisticComputerMock)).computeStatistic((Product)Mockito.same((Object)this.validProductMock1));
        ((StatisticComputer)inOrder.verify((Object)this.statisticComputerMock)).computeStatistic((Product)Mockito.same((Object)this.validProductMock2));
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.never())).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock2, (VerificationMode)Mockito.never())).dispose();
    }

    @Test
    public void testThatLoadedProductsShouldBeDisposedAfterStatisticComputation() throws IOException {
        File file1 = new File("1");
        File file2 = new File("2");
        Mockito.when((Object)this.productLoaderMock.loadProduct(file1)).thenReturn((Object)this.validProductMock1);
        Mockito.when((Object)this.productLoaderMock.loadProduct(file2)).thenReturn((Object)this.validProductMock2);
        File[] productFilesToLoad = new File[]{file1, file2};
        this.productLoop.loop(new Product[0], productFilesToLoad);
        InOrder inOrder = Mockito.inOrder((Object[])new Object[]{this.statisticComputerMock});
        ((StatisticComputer)inOrder.verify((Object)this.statisticComputerMock)).computeStatistic((Product)Mockito.same((Object)this.validProductMock1));
        ((StatisticComputer)inOrder.verify((Object)this.statisticComputerMock)).computeStatistic((Product)Mockito.same((Object)this.validProductMock2));
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.times((int)1))).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock2, (VerificationMode)Mockito.times((int)1))).dispose();
    }

    @Test
    public void testThatFilesWhichPointToAnAlreadyLoadedProductShouldNotBeOpenedTwice() throws IOException {
        File file1 = this.validProductMock1.getFileLocation();
        File file2 = this.validProductMock2.getFileLocation();
        Mockito.when((Object)this.productLoaderMock.loadProduct(file1)).thenReturn((Object)this.validProductMock1);
        Product[] alreadyLoadedProducts = new Product[]{this.validProductMock2};
        File[] productFilesToLoad = new File[]{file1, file2};
        this.productLoop.loop(alreadyLoadedProducts, productFilesToLoad);
        InOrder inOrder = Mockito.inOrder((Object[])new Object[]{this.statisticComputerMock});
        ((StatisticComputer)inOrder.verify((Object)this.statisticComputerMock)).computeStatistic((Product)Mockito.same((Object)this.validProductMock2));
        ((StatisticComputer)inOrder.verify((Object)this.statisticComputerMock)).computeStatistic((Product)Mockito.same((Object)this.validProductMock1));
        ((ProductLoader)Mockito.verify((Object)this.productLoaderMock)).loadProduct((File)Mockito.eq((Object)file1));
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.times((int)1))).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock2, (VerificationMode)Mockito.never())).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock2, (VerificationMode)Mockito.atLeastOnce())).getFileLocation();
    }

    @Test
    public void testThatNoOperatorExceptionOccursIfNoProductsAreComputed() {
        try {
            this.productLoop.loop(new Product[0], new File[0]);
            Assert.assertEquals((long)0L, (long)this.productLoop.getProductNames().length);
        }
        catch (OperatorException expected) {
            Assert.assertEquals((Object)"No input products found.", (Object)expected.getMessage());
        }
    }

    @Test
    public void testThatFilesWhichCanNotBeOpenedAreLogged() throws IOException {
        File file1 = this.validProductMock1.getFileLocation();
        File file2 = new File("No reader available");
        Mockito.when((Object)this.productLoaderMock.loadProduct(file1)).thenReturn((Object)this.validProductMock1);
        Mockito.when((Object)this.productLoaderMock.loadProduct(file2)).thenReturn(null);
        Product[] alreadyLoadedProducts = new Product[]{};
        File[] productFilesToLoad = new File[]{file1, file2};
        this.productLoop.loop(alreadyLoadedProducts, productFilesToLoad);
        ((Logger)Mockito.verify((Object)this.loggerMock)).severe("Failed to read from 'No reader available' (not a data product or reader missing)");
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock, (VerificationMode)Mockito.times((int)1))).computeStatistic(this.validProductMock1);
        ((ProductLoader)Mockito.verify((Object)this.productLoaderMock, (VerificationMode)Mockito.times((int)2))).loadProduct((File)Mockito.any(File.class));
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.statisticComputerMock, this.productLoaderMock});
    }

    @Test
    public void testThatIOExceptionsAreLogged() throws IOException {
        File file1 = this.validProductMock1.getFileLocation();
        File file2 = new File("Causes IO Exception");
        Mockito.when((Object)this.productLoaderMock.loadProduct(file1)).thenReturn((Object)this.validProductMock1);
        Mockito.when((Object)this.productLoaderMock.loadProduct(file2)).thenThrow(new Throwable[]{new IOException()});
        Product[] alreadyLoadedProducts = new Product[]{};
        File[] productFilesToLoad = new File[]{file1, file2};
        this.productLoop.loop(alreadyLoadedProducts, productFilesToLoad);
        ((Logger)Mockito.verify((Object)this.loggerMock)).severe("Failed to read from 'Causes IO Exception' (not a data product or reader missing)");
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock, (VerificationMode)Mockito.times((int)1))).computeStatistic(this.validProductMock1);
        ((ProductLoader)Mockito.verify((Object)this.productLoaderMock, (VerificationMode)Mockito.times((int)2))).loadProduct((File)Mockito.any(File.class));
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.statisticComputerMock, this.productLoaderMock});
    }

    @Test
    public void testThatAlreadyLoadedProductsAreSkippedIfOutOfDateRange() {
        Product productMockBefore = this.createProductMock(4, 2, true, false);
        Product productMockAfter = this.createProductMock(4, 2, false, true);
        Product[] alreadyLoadedProducts = new Product[]{productMockBefore, productMockAfter, this.validProductMock1};
        Mockito.when((Object)this.validatorMock.isValid(productMockBefore)).thenReturn((Object)false);
        Mockito.when((Object)this.validatorMock.isValid(productMockAfter)).thenReturn((Object)false);
        this.productLoop.loop(alreadyLoadedProducts, new File[0]);
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock1);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.statisticComputerMock});
        ((Product)Mockito.verify((Object)productMockBefore, (VerificationMode)Mockito.never())).dispose();
        ((Product)Mockito.verify((Object)productMockAfter, (VerificationMode)Mockito.never())).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.never())).dispose();
    }

    @Test
    public void testThatProductsToBeLoadedAreSkippedIfOutOfDateRange() throws IOException {
        File file1 = new File("before");
        File file2 = new File("after");
        File file3 = this.validProductMock1.getFileLocation();
        Product productMockBefore = this.createProductMock(4, 2, true, false);
        Product productMockAfter = this.createProductMock(4, 2, false, true);
        Mockito.when((Object)this.productLoaderMock.loadProduct(file1)).thenReturn((Object)productMockBefore);
        Mockito.when((Object)this.productLoaderMock.loadProduct(file2)).thenReturn((Object)productMockAfter);
        Mockito.when((Object)this.productLoaderMock.loadProduct(file3)).thenReturn((Object)this.validProductMock1);
        Product[] alreadyLoadedProducts = new Product[]{};
        File[] productFilesToLoad = new File[]{file1, file2, file3};
        Mockito.when((Object)this.validatorMock.isValid(productMockBefore)).thenReturn((Object)false);
        Mockito.when((Object)this.validatorMock.isValid(productMockAfter)).thenReturn((Object)false);
        this.productLoop.loop(alreadyLoadedProducts, productFilesToLoad);
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock1);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.statisticComputerMock});
        ((Product)Mockito.verify((Object)productMockBefore, (VerificationMode)Mockito.times((int)1))).dispose();
        ((Product)Mockito.verify((Object)productMockAfter, (VerificationMode)Mockito.times((int)1))).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.times((int)1))).dispose();
    }

    @Test
    public void testThatLoopWorksIfAlreadyLoadedProductsIsNull() throws IOException {
        File file = this.validProductMock1.getFileLocation();
        Mockito.when((Object)this.productLoaderMock.loadProduct(file)).thenReturn((Object)this.validProductMock1);
        this.productLoop.loop(null, new File[]{file});
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock1);
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.times((int)1))).dispose();
    }

    @Test
    public void testThatLoopWorksIfAlreadyLoadedProductsContainsNullValues() throws IOException {
        Product[] alreadyLoadedProducts = new Product[]{this.validProductMock1, null, this.validProductMock2};
        this.productLoop.loop(alreadyLoadedProducts, new File[0]);
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock1);
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock2);
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.never())).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock2, (VerificationMode)Mockito.never())).dispose();
        String[] productNames = this.productLoop.getProductNames();
        Assert.assertEquals((long)2L, (long)productNames.length);
        Assert.assertThat((Object)productNames[0], (Matcher)this.endsWith(this.validProductMock1.getFileLocation().getName()));
        Assert.assertThat((Object)productNames[1], (Matcher)this.endsWith(this.validProductMock2.getFileLocation().getName()));
    }

    @Test
    public void testThatLoopWorksIfProductFilesToLoadContainsNullValues() throws IOException {
        File file1 = this.validProductMock1.getFileLocation();
        File file2 = this.validProductMock2.getFileLocation();
        File[] productFilesToLoad = new File[]{file1, file2};
        Mockito.when((Object)this.productLoaderMock.loadProduct(file1)).thenReturn((Object)this.validProductMock1);
        Mockito.when((Object)this.productLoaderMock.loadProduct(file2)).thenReturn((Object)this.validProductMock2);
        this.productLoop.loop(null, productFilesToLoad);
        ((ProductLoader)Mockito.verify((Object)this.productLoaderMock)).loadProduct(file1);
        ((ProductLoader)Mockito.verify((Object)this.productLoaderMock)).loadProduct(file2);
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock1);
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock2);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.statisticComputerMock, this.productLoaderMock});
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.times((int)1))).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock2, (VerificationMode)Mockito.times((int)1))).dispose();
        String[] productNames = this.productLoop.getProductNames();
        Assert.assertEquals((long)2L, (long)productNames.length);
        Assert.assertThat((Object)productNames[0], (Matcher)this.endsWith(file1.getName()));
        Assert.assertThat((Object)productNames[1], (Matcher)this.endsWith(file2.getName()));
    }

    @Test
    public void testThatComputationIsPerformedWhenStartAndEndTimeOfProductAreNotSet() {
        Product[] alreadyLoadedProducts = new Product[]{this.validProductMock1, this.validProductMock2};
        Mockito.when((Object)this.validProductMock1.getStartTime()).thenReturn(null);
        Mockito.when((Object)this.validProductMock2.getStartTime()).thenReturn(null);
        Mockito.when((Object)this.validProductMock1.getEndTime()).thenReturn(null);
        Mockito.when((Object)this.validProductMock2.getEndTime()).thenReturn(null);
        this.productLoop.loop(alreadyLoadedProducts, new File[0]);
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock1);
        ((StatisticComputer)Mockito.verify((Object)this.statisticComputerMock)).computeStatistic(this.validProductMock2);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.statisticComputerMock});
        ((Product)Mockito.verify((Object)this.validProductMock1, (VerificationMode)Mockito.never())).dispose();
        ((Product)Mockito.verify((Object)this.validProductMock2, (VerificationMode)Mockito.never())).dispose();
        String[] productNames = this.productLoop.getProductNames();
        Assert.assertEquals((long)2L, (long)productNames.length);
        Assert.assertThat((Object)productNames[0], (Matcher)this.endsWith(this.validProductMock1.getFileLocation().getName()));
        Assert.assertThat((Object)productNames[1], (Matcher)this.endsWith(this.validProductMock2.getFileLocation().getName()));
    }

    private Product createTimeValidProductMock(int startOffset, int numObservationDays) {
        return this.createProductMock(startOffset, numObservationDays, false, false);
    }

    private Product createProductMock(int startOffset, int numObservationDays, boolean before_startDate, boolean after_endDate) {
        ProductData.UTC startTime = this.getStartTime(startOffset, numObservationDays, before_startDate, after_endDate);
        ProductData.UTC endTime = new ProductData.UTC(startTime.getMJD() + (double)numObservationDays);
        File fileLocation = this.getFileLocation(startOffset, numObservationDays, before_startDate, after_endDate);
        Product mock = (Product)Mockito.mock(Product.class);
        Mockito.when((Object)mock.getStartTime()).thenReturn((Object)startTime);
        Mockito.when((Object)mock.getEndTime()).thenReturn((Object)endTime);
        Mockito.when((Object)mock.getFileLocation()).thenReturn((Object)fileLocation);
        Mockito.when((Object)mock.getName()).thenReturn((Object)fileLocation.getName());
        return mock;
    }

    private ProductData.UTC getStartTime(int startOffset, int numObservationDays, boolean before_startDate, boolean after_endDate) {
        ProductData.UTC startTime = before_startDate ? new ProductData.UTC(this.startDate.getMJD() - (double)startOffset - (double)numObservationDays) : (after_endDate ? new ProductData.UTC(this.endDate.getMJD() + (double)startOffset) : new ProductData.UTC(this.startDate.getMJD() + (double)startOffset));
        return startTime;
    }

    private File getFileLocation(int startOffset, int numObservationDays, boolean before, boolean after) {
        StringBuilder location = new StringBuilder();
        location.append("mock_loc_");
        if (before) {
            location.append("before_");
        } else if (after) {
            location.append("after_");
        }
        location.append(startOffset).append("_");
        location.append(numObservationDays);
        return new File(location.toString());
    }

    private EndsWith endsWith(String suffix) {
        return new EndsWith(suffix);
    }
}

