/*
 * Copyright (C) 2015 CS SI
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, see http://www.gnu.org/licenses/
 */
package org.esa.snap.smart.configurator.ui;

import org.esa.snap.ui.ModalDialog;

import javax.swing.JScrollPane;
import java.awt.Window;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 *
 * Dialog to edit a line with multiple separators in a text area showing one line per peace of string.
 * It manages double quotes, meaning for the string "A B \"C D\"" with space as separator it shows in the dialog:
 * A
 * B
 * C D
 *
 * @author Nicolas Ducoin
 */
public class LineSplitTextEditDialog extends ModalDialog {

    private javax.swing.JTextArea textArea;

    private String separator;
    private String textWithSeparators;


    
    /**
     * Creates new form LineSplitTextEditDialog
     *
     * @param parent the parent window
     * @param textWithSeparators the text to be edited or displayed
     * @param separator the separator for this text, the text will get split using this
     * @param title the title of the dialog
     */
    public LineSplitTextEditDialog(
            Window parent,
            String textWithSeparators,
            String separator,
            String title) {
        this(parent, textWithSeparators, separator, title, true);
    }


    /**
     * Creates new form LineSplitTextEditDialog
     *
     * @param parent the parent window
     * @param textWithSeparators the text to be edited or displayed
     * @param separator the separator for this text, the text will get split using this
     * @param title the title of the dialog
     * @param canEdit if false, the text won't be editable
     */
    public LineSplitTextEditDialog(
            Window parent,
            String textWithSeparators,
            String separator,
            String title,
            boolean canEdit) {
        super(parent, title, ID_OK_CANCEL, null);

        this.textWithSeparators = textWithSeparators;
        this.separator = separator;

        initComponents();

        String multiLineText = toMultiLine(textWithSeparators, separator);
        textArea.setText(multiLineText);

        if(!canEdit) {
            textArea.setEditable(false);
        }
    }


    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    private void initComponents() {

        JScrollPane textAreaScrollPane = new javax.swing.JScrollPane();
        textArea = new javax.swing.JTextArea();

        textArea.setColumns(20);
        textArea.setRows(10);
        textAreaScrollPane.setViewportView(textArea);

        setContent(textAreaScrollPane);
    }

    @Override
    protected void onOK() {
        String multiLineString = textArea.getText();
        textWithSeparators = toMonoLine(multiLineString);
        super.onOK();
    }

    /**
     *
     * @return the text with separator, it can be the text passed in the contructor (if cancel was pressed)
     * or a modified text, comming from the multi-line editing window
     */
    public String getTextWithSeparators() {
        return textWithSeparators;
    }

    static String toMultiLine(String monoLineString, String separator) {
        StringBuilder stringBuilder = new StringBuilder();
        Pattern regex = Pattern.compile("[^" + separator + "\"']+|\"[^\"]*\"|'[^']*'");
        Matcher regexMatcher = regex.matcher(monoLineString);
        while (regexMatcher.find()) {
            stringBuilder.append(regexMatcher.group());
            stringBuilder.append(System.lineSeparator());
        }
        return stringBuilder.toString();
    }

    String toMonoLine(String multiLineString) {
        StringBuilder builder = new StringBuilder();

        String[] linesOfText = multiLineString.split(System.lineSeparator());

        if(linesOfText.length > 0) {
            builder.append(linesOfText[0]);

            for (int i=1 ; i<linesOfText.length ; i++) {
                builder.append(separator);
                builder.append(linesOfText[i]);
            }
        }

        return builder.toString();
    }
}
