/*
 * Decompiled with CFR 0.152.
 */
package org.esa.smos;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import org.esa.smos.SmosUtils;
import org.junit.Assert;
import org.junit.Test;

public class SmosUtilsTest {
    @Test
    public void testGetSensingTimesFromFilename() {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
        df.setTimeZone(TimeZone.getTimeZone("UTC"));
        Date startTime = SmosUtils.getSensingStartTimeFromFilename((String)"SM_OPER_MIR_SMUDP2_20101019T050111_20101019T055510_309_002_1.zip");
        Date stopTime = SmosUtils.getSensingStopTimeFromFilename((String)"SM_OPER_MIR_SMUDP2_20101019T050111_20101019T055510_309_002_1.zip");
        Assert.assertEquals((Object)"2010-10-19T05:01:11", (Object)df.format(startTime));
        Assert.assertEquals((Object)"2010-10-19T05:55:10", (Object)df.format(stopTime));
    }

    @Test
    public void testGetSensingTimesFromFilename_InvalidDateFormat() {
        Assert.assertNull((Object)SmosUtils.getSensingStartTimeFromFilename((String)"SM_OPER_MIR_SMUDP2_20101MART050111_20101019T055510_309_002_1.zip"));
        Assert.assertNull((Object)SmosUtils.getSensingStopTimeFromFilename((String)"SM_OPER_MIR_SMUDP2_20101019T050111_20101SEPT055510_309_002_1.zip"));
    }

    @Test
    public void testGetSensingTimesFromTooShortFilename() {
        Assert.assertNull((Object)SmosUtils.getSensingStartTimeFromFilename((String)"SM_OPER_MIR_SMUDP2_20101019T0501"));
        Assert.assertNull((Object)SmosUtils.getSensingStopTimeFromFilename((String)"SM_OPER_MIR_SMUDP2_20101019T050111_20101019T0555"));
    }

    @Test
    public void testGetStartDateFromFilename() {
        Calendar cal = GregorianCalendar.getInstance();
        cal.set(2007, 0, 1);
        Assert.assertTrue((boolean)SmosUtilsTest.isSameDay(cal.getTime(), SmosUtils.getSensingStartDayFromFilename((String)"SM_TEST_AUX_PATT99_20070101T000000_20781231T235959_00000006.EEF")));
        Assert.assertTrue((boolean)SmosUtilsTest.isSameDay(cal.getTime(), SmosUtils.getSensingStartDayFromFilename((String)"SM_TEST_AUX_BWGHT__20070101T000000_20781231T235959_00000003.EEF")));
        Assert.assertTrue((boolean)SmosUtilsTest.isSameDay(cal.getTime(), SmosUtils.getSensingStartDayFromFilename((String)"SM_TEST_AUX_RFI____20070101T000000_20781231T235959_00000001.EEF")));
        Assert.assertTrue((boolean)SmosUtilsTest.isSameDay(cal.getTime(), SmosUtils.getSensingStartDayFromFilename((String)"SM_TEST_AUX_RFI____20070101T235959_20781231T235959_00000001.EEF")));
        cal.set(2007, 1, 23);
        Assert.assertTrue((boolean)SmosUtilsTest.isSameDay(cal.getTime(), SmosUtils.getSensingStartDayFromFilename((String)"SM_TEST_AUX_VTEC___20070223T000000_20070224T000000_00000001.EEF")));
        cal.set(2012, 10, 18);
        Assert.assertTrue((boolean)SmosUtilsTest.isSameDay(cal.getTime(), SmosUtils.getSensingStartDayFromFilename((String)"SM_OPER_MIR_BWLD1C_20121118T002733_20121118T012104_116_001_1.zip")));
        cal.set(2012, 10, 20);
        Assert.assertTrue((boolean)SmosUtilsTest.isSameDay(cal.getTime(), SmosUtils.getSensingStartDayFromFilename((String)"SM_OPER_MIR_UNCU1A_20121120T010256_20121120T014313_116_001_1.zip")));
        cal.set(2005, 0, 1);
        Assert.assertTrue((boolean)SmosUtilsTest.isSameDay(cal.getTime(), SmosUtils.getSensingStartDayFromFilename((String)"SM_TEST_AUX_MOONT__20050101T000000_20500101T000000_001_001_4.zip")));
    }

    @Test
    public void testGetProductType() {
        Assert.assertEquals((Object)"AUX_DATA", (Object)SmosUtils.getProductType((String)"SM_TEST_AUX_LSMASK_20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertEquals((Object)"AUX_DATA", (Object)SmosUtils.getProductType((String)"SM_TEST_AUX_BFP____20070101T000000_20781231T235959_00000004.EEF"));
        Assert.assertEquals((Object)"AUX_DATA", (Object)SmosUtils.getProductType((String)"SM_TEST_AUX_SUNT___20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertEquals((Object)"MIR_GMATD_", (Object)SmosUtils.getProductType((String)"SM_TEST_MIR_GMATD__20121117T020130_20781231T235959_115_001_3.tgz"));
        Assert.assertEquals((Object)"AUX_DATA", (Object)SmosUtils.getProductType((String)"SM_TEST_AUX_CNFL1P_20120101T000000_20500101T000000_101_001_3.EEF"));
        Assert.assertEquals((Object)"MIR_CRSD1A", (Object)SmosUtils.getProductType((String)"SM_OPER_MIR_CRSD1a_20121117T025249_20121117T043300_116_001_1.zip"));
        Assert.assertEquals((Object)"RQC_RQD", (Object)SmosUtils.getProductType((String)"SM_OPER_RQC_VTEC_C_20090512T230000_20090514T010000_311_001_1.EEF"));
        Assert.assertEquals((Object)"RQC_RQD", (Object)SmosUtils.getProductType((String)"SM_TEST_RQD_SCND1C_20121117T120515_20121117T122512_311_001_1.EEF"));
    }

    @Test
    public void testIsDBLFileName() {
        Assert.assertFalse((boolean)SmosUtils.isDblFileName((String)"plupsi.txt"));
        Assert.assertFalse((boolean)SmosUtils.isDblFileName((String)"SM_OPER_MIR_BWLD1C_20121118T002733_20121118T012104_116_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isDblFileName((String)"noExtensionFile"));
        Assert.assertTrue((boolean)SmosUtils.isDblFileName((String)"SM_xxxx_MIR_CORN0__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isDblFileName((String)"SM_BLAH_MIR_TARD1A_20070223T112710_20070223T121514_001_001_0.DBL"));
    }

    @Test
    public void testIsHDRFileName() {
        Assert.assertFalse((boolean)SmosUtils.isHdrFileName((String)"plupsi.txt"));
        Assert.assertFalse((boolean)SmosUtils.isHdrFileName((String)"SM_OPER_MIR_BWLD1C_20121118T002733_20121118T012104_116_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isHdrFileName((String)"noExtensionFile"));
        Assert.assertTrue((boolean)SmosUtils.isHdrFileName((String)"SM_xxxx_MIR_CORN0__20070223T061024_20070223T062500_001_001_0.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isHdrFileName((String)"SM_BLAH_MIR_TARD1A_20070223T112710_20070223T121514_001_001_0.hdr"));
    }

    @Test
    public void testIsL0FileType() {
        Assert.assertTrue((boolean)SmosUtils.isL0Type((String)"SM_HELP_MIR_SC_F0__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isL0Type((String)"SM_xxxx_MIR_UNCU0__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isL0Type((String)"SM_xxxx_MIR_CORN0__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isL0Type((String)"SM_TEST_MIR_SC_F0__20070223T112710_20070223T121514_001_001_0.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isL0Type((String)"SM_TEST_AUX_IGRF___20080102T010000_20080102T025959_105_001_0.zip"));
        Assert.assertFalse((boolean)SmosUtils.isL0Type((String)"SM_TEST_AUX_PATT99_20070101T000000_20781231T235959_00000006.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isL0Type((String)"SM_TEST_AUX_IGRF___20080102T010000_20080102T025959_105_001_0.zip"));
    }

    @Test
    public void testIsL1aFileType() {
        Assert.assertTrue((boolean)SmosUtils.isL1aType((String)"SM_OPER_MIR_TARD1A_20121117T102025_20121117T105144_203_001_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isL1aType((String)"SM_OPER_MIR_CRSD1A_20121117T025249_20121117T043300_116_001_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL1aType((String)"SM_TEST_MIR_AFWU1A_20121117T022130_20121117T024030_203_001_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isL1aType((String)"SM_TEST_MIR_ANIR1A_20121119T213500_20121119T231156_306_001_3.zip"));
        Assert.assertFalse((boolean)SmosUtils.isL1aType((String)"SM_TEST_MIR_GMATD__20121117T020130_20781231T235959_115_001_3.tgz"));
        Assert.assertFalse((boolean)SmosUtils.isL1aType((String)"SM_OPER_RQD_BWSF1C_20121120T015956_20121120T023316_203_001_1.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isL1aType((String)"SM_OPER_AUX_ORBRES_20121118T000000_20121125T000000_240_028_1.EEF"));
    }

    @Test
    public void testIsL1bFileType() {
        Assert.assertTrue((boolean)SmosUtils.isL1bType((String)"SM_OPER_MIR_SC_D1B_20121118T002704_20121118T012104_116_001_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL1bType((String)"SM_OPER_MIR_TARD1B_20121117T102025_20121117T105144_116_001_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL1bType((String)"SM_TEST_MIR_GMATU__20121117T022130_20781231T235959_115_001_3.tgz"));
        Assert.assertTrue((boolean)SmosUtils.isL1bType((String)"SM_TEST_MIR_FTTD___20091228T124638_20091228T125235_307_002_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL1bType((String)"SM_TEST_MIR_FTTF___20121120T005052_20121120T005310_308_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isL1bType((String)"M_OPER_MIR_CRSD1A_20121117T025249_20121117T043300_116_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isL1bType((String)"SM_OPER_RQD_BWSF1C_20121120T015956_20121120T023316_203_001_1.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isL1bType((String)"SM_OPER_AUX_ORBRES_20121118T000000_20121125T000000_240_028_1.EEF"));
    }

    @Test
    public void testIsL1cFileType() {
        Assert.assertTrue((boolean)SmosUtils.isL1cType((String)"SM_TEST_MIR_SCLD1C_20070223T061024_20070223T070437_141_000_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isL1cType((String)"SM_TEST_MIR_BWLD1C_20070223T061024_20070223T070437_141_000_0.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isL1cType((String)"SM_TEST_MIR_BWLF1C_20070223T112729_20070223T121644_141_000_0.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isL1cType((String)"SM_TEST_AUX_RFI____20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isL1cType((String)"SM_TEST_AUX_RFI____20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isL1cType((String)"SM_BLAH_MIR_TARD1A_20070223T112710_20070223T121514_001_001_0.DBL"));
    }

    @Test
    public void testIsL2FileType() {
        Assert.assertTrue((boolean)SmosUtils.isL2Type((String)"SO_GNAT_MIR_TSM_2__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isL2Type((String)"SO_GNOT_MIR_TOS_2__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isL2Type((String)"SO_GNAT_MIR_SM__2__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isL2Type((String)"SO_GNOT_MIR_OS__2__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertTrue((boolean)SmosUtils.isL2Type((String)"SM_TEST_MIR_SMUDP2_20121118T143742_20121118T153047_303_002_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL2Type((String)"SM_TEST_MIR_SMDAP2_20121118T135052_20121118T144140_303_007_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL2Type((String)"SM_TEST_MIR_OSUDP2_20121118T143742_20121118T153047_306_002_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL2Type((String)"SM_TEST_MIR_OSDAP2_20121118T143742_20121118T153047_306_002_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isL2Type((String)"SM_TEST_AUX_IGRF___20080102T010000_20080102T025959_105_001_0.zip"));
        Assert.assertFalse((boolean)SmosUtils.isL2Type((String)"SM_TEST_AUX_PATT99_20070101T000000_20781231T235959_00000006.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isL2Type((String)"SM_TEST_MIR_SC_F0__20070223T112710_20070223T121514_001_001_0.DBL"));
    }

    @Test
    public void testIsL2Format() {
        Assert.assertTrue((boolean)SmosUtils.isL2Format((String)"SM_TEST_MIR_SMUDP2_20121118T143742_20121118T153047_303_002_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL2Format((String)"SM_TEST_MIR_SMDAP2_20121118T135052_20121118T144140_303_007_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL2Format((String)"SM_TEST_MIR_OSUDP2_20121118T143742_20121118T153047_306_002_1.zip"));
        Assert.assertTrue((boolean)SmosUtils.isL2Format((String)"SM_TEST_MIR_OSDAP2_20121118T143742_20121118T153047_306_002_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isL2Format((String)"SM_TEST_AUX_IGRF___20080102T010000_20080102T025959_105_001_0.zip"));
        Assert.assertFalse((boolean)SmosUtils.isL2Format((String)"SM_TEST_AUX_PATT99_20070101T000000_20781231T235959_00000006.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isL2Format((String)"SM_TEST_MIR_SC_F0__20070223T112710_20070223T121514_001_001_0.DBL"));
    }

    @Test
    public void testIsLightBufrSpecification() throws Exception {
        Assert.assertTrue((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMM_20131028030552_20131028003256_20131028020943_bufr_v505.bin"));
        Assert.assertTrue((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMM_20131028030552_20131028003256_20131028020943_bufr_v505.bin.bz2"));
        Assert.assertTrue((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMM_20131028033037_20131028002942_20131028003302_bufr_v505.bin"));
        Assert.assertTrue((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMM_20131028033037_20131028002942_20131028003302_bufr_v505.bin.bz2"));
        Assert.assertTrue((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMM_20131028044206_20131028020942_20131028034943_bufr_v505.bin"));
        Assert.assertTrue((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMM_20131028044206_20131028020942_20131028034943_bufr_v505.bin.bz2"));
        Assert.assertTrue((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMM_20131028044206_20131028020942_20131028034943_bufr_v647.bin"));
        Assert.assertFalse((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_D_LEMM_20131028044206_20131028020942_20131028034943_bufr_v505.bin"));
        Assert.assertFalse((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMN_20131028044206_20131028020942_20131028034943_bufr_v505.bin"));
        Assert.assertFalse((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMN_20131028044206_20131028020942_20131028034943_bufr_v505.zip"));
        Assert.assertFalse((boolean)SmosUtils.isLightBufrType((String)"W_ES-ESA-ESAC,SMOS,N256_C_LEMN_20131028044206_20131028020942_20131028034943_bufr_v505.bz2"));
    }

    @Test
    public void testIsBufrSpecification() throws Exception {
        Assert.assertTrue((boolean)SmosUtils.isBufrType((String)"miras_20131028_002942_20131028_003302_smos_20947_t_20131028_033058_l1c.bufr"));
        Assert.assertTrue((boolean)SmosUtils.isBufrType((String)"miras_20131028_003256_20131028_020943_smos_20947_o_20131028_031005_l1c.bufr"));
    }

    @Test
    public void testIsAuxFileType() {
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_ORBPRE_20121117T000000_20121124T000000_240_034_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_APOD01_20070101T000000_20781231T235959_00000002.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_BFP____20070101T000000_20781231T235959_00000003.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_BSCAT__20070101T000000_20781231T235959_00000002.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_BWGHT__20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_CNFL1P_20120101T000000_20500101T000000_101_001_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_DGG____20050101T000000_20500101T000000_001_002_4.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_DGG____20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_FAIL___20070101T000000_20781231T235959_00000002.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_FLATT__20070101T000000_20781231T235959_00000014.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_GALAXY_20050101T000000_20500101T000000_001_001_4.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_GLXY___20070101T000000_20781231T235959_000000007.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_GALNIR_20050101T000000_20500101T000000_300_001_5.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_LCF____20070101T000000_20781231T235959_00000004.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_MASK___20050101T000000_20500101T000000_001_001_4.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_MASK___20070101T000000_20781231T235959_00000006.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_MOONT__20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_NIR____20070101T000000_20781231T235959_00000006.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_PLM____20070101T000000_20781231T235959_0000000012.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_PMS____20070101T000000_20781231T235959_00000005.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_VTEC___20080102T010000_20080102T025959_105_001_0.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_RFI____20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_IGRF___20080102T010000_20080102T025959_105_001_0.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_SPAR___20070101T000000_20781231T235959_00000006.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_SUNT___20050101T000000_20500101T000000_001_001_4.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_SUNT___20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_SGLINT_20050101T000000_20500101T000000_001_001_4.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_TIME___20050101T000000_20500101T000000_001_001_4.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_VTEC_C_20080102T010000_20080102T025959_105_001_0.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_VTEC_P_20080414T010000_20080414T025959_105_001_0.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_PATT99_20070101T000000_20781231T235959_00000006.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_PATT12_20070101T000000_20781231T235959_00000004.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_ORBPRE_20121117T000000_20121124T000000_240_034_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_APDS00_20050101T000000_20500101T000000_001_001_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_LSMASK_20070101T000000_20781231T235959_00000001.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_ECMWF__20090408T005820_20090408T020050_301_001_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_DFFFRA_20050101T000000_20500101T000000_001_001_9.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_DFFXYZ_20050101T000000_20500101T000000_001_001_9.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_DFFLAI_20090509T000000_20090608T000000_302_001_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_DFFLMX_20050101T000000_20500101T000000_001_001_9.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_DGGXYZ_20050101T000000_20500101T000000_001_001_9.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_DGGTLV_20121129T010000_20121130T030000_301_001_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_DGGTFO_20121129T010000_20121130T030000_301_001_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_DGGROU_20121129T010000_20121130T030000_301_001_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_DGGRFI_20121129T010000_20121130T030000_301_001_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_DGGFLO_20121129T010000_20121130T030000_301_001_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_WEF____20050101T000000_20500101T000000_001_001_9.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_MN_WEF_20050101T000000_20500101T000000_001_001_9.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_SOIL_P_20050101T000000_20500101T000000_001_001_9.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_GAL_SM_20050101T000000_20500101T000000_001_001_9.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_LANDCL_20050101T000000_20500101T000000_001_001_9.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_CNFSMD_20050101T000000_20500101T000000_001_001_9.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_CNFSMF_20050101T000000_20500101T000000_001_001_9.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_FLTSEA_20050101T000000_20500101T000000_001_010_8.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_RGHNS1_20050101T000000_20500101T000000_001_011_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_RGHNS2_20050101T000000_20500101T000000_001_010_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_RGHNS3_20050101T000000_20500101T000000_001_010_8.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_GAL_OS_20050101T000000_20500101T000000_001_004_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_GAL2OS_20050101T000000_20500101T000000_001_010_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_FOAM___20050101T000000_20500101T000000_001_010_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_ATMOS__20050101T000000_20500101T000000_001_010_8.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_DISTAN_20050101T000000_20500101T000000_001_010_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_SSS____20050101T000000_20500101T000000_001_010_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_CNFOSD_20050101T000000_20500101T000000_001_010_8.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_CNFOSF_20050101T000000_20500101T000000_001_010_8.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_AGDPT__20050101T000000_20500101T000000_001_003_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_ECOLAI_20050101T000000_20500101T000000_001_003_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_BNDLST_20050101T000000_20500101T000000_001_003_8.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_MISP___20050101T000000_20500101T000000_300_001_5.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_APDL___20050101T000000_20500101T000000_300_001_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_AUX_APDS___20050101T000000_20500101T000000_300_001_5.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_ORBRES_20091105T000000_20091106T000000_280_009_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_CNFFAR_20050101T000000_20500101T000000_100_002_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_OTT1D__20050101T000000_20500101T000000_001_002_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_OTT1F__20050101T000000_20500101T000000_001_002_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_OTT2D__20050101T000000_20500101T000000_001_002_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_OTT3F__20050101T000000_20500101T000000_001_002_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_OTT2F__20050101T000000_20500101T000000_001_002_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_OTT3D__20050101T000000_20500101T000000_001_004_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_BULL_B_20101202T000000_20500101T000000_100_002_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_VTEC_R_20091117T230000_20091119T010000_306_001_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_PLM____20050101T000000_20500101T000000_300_006_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_BFP____20050101T000000_20500101T000000_300_001_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_BSCAT__20050101T000000_20500101T000000_300_001_4.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_BWGHT__20050101T000000_20500101T000000_300_003_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_FAIL___20050101T000000_20500101T000000_300_001_4.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_LCF____20050101T000000_20500101T000000_300_005_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_MOONT__20050101T000000_20500101T000000_300_001_4.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_NIR____20050101T000000_20500101T000000_300_002_4.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_PMS____20050101T000000_20500101T000000_300_005_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_RFI____20050101T000000_20500101T000000_300_002_3.zip"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_SPAR___20050101T000000_20500101T000000_320_001_3.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_ORBRES_20091119T000000_20091120T000000_280_023_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_AUX_CNFL0P_20050101T000000_20500101T000000_001_002_3.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isAuxFileType((String)"SM_OPER_MIR_BWLD1C_20121118T002733_20121118T012104_116_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isAuxFileType((String)"SM_TEST_MIR_GMATD__20121117T020130_20781231T235959_115_001_3.tgz"));
    }

    @Test
    public void testIsQualityControlType() {
        Assert.assertTrue((boolean)SmosUtils.isQualityControlType((String)"SM_OPER_RQC_ECMWF__20090405T011500_20090405T021820_301_001_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isQualityControlType((String)"SM_TEST_RQD_OSDAP2_20121117T065617_20121117T075017_311_001_1.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isQualityControlType((String)"SM_OPER_MIR_BWLD1C_20121118T002733_20121118T012104_116_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isQualityControlType((String)"SM_xxxx_MIR_CORN0__20070223T061024_20070223T062500_001_001_0.DBL"));
    }

    @Test
    public void testIsMirasPlanType() {
        Assert.assertTrue((boolean)SmosUtils.isMirasPlanType((String)"SM_TEST_MPL_ORBSCT_20070223T060002_20781231T235959_00000001.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isMirasPlanType((String)"SM_OPER_MPL_PROTEV_20091120T101808_20091221T101153_280_023_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isMirasPlanType((String)"SM_OPER_MPL_XBDOWN_20091123T000000_20091124T000000_331_001_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isMirasPlanType((String)"SM_OPER_MPL_APIDPL_20091120T000000_20091123T000000_331_001_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isMirasPlanType((String)"SM_OPER_MPL_HLPLAN_20091120T000000_20091123T000000_331_001_1.EEF"));
        Assert.assertTrue((boolean)SmosUtils.isMirasPlanType((String)"SM_OPER_MPL_XBDPRE_20091123T000000_20091130T000000_110_001_7.EEF"));
        Assert.assertFalse((boolean)SmosUtils.isMirasPlanType((String)"SM_OPER_MIR_BWLD1C_20121118T002733_20121118T012104_116_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isMirasPlanType((String)"SM_TEST_AUX_IGRF___20080102T010000_20080102T025959_105_001_0.zip"));
        Assert.assertFalse((boolean)SmosUtils.isMirasPlanType((String)"TheNewSensoir.txt"));
    }

    @Test
    public void testIsAuxECMWFType() {
        Assert.assertTrue((boolean)SmosUtils.isAuxECMWFType((String)"SM_OPER_AUX_ECMWF__20091113T030500_20091113T040730_306_001_3.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isAuxECMWFType((String)"SM_OPER_AUX_ECMWF__20110227T182140_20110227T192410_310_001_3.zip"));
        Assert.assertFalse((boolean)SmosUtils.isAuxECMWFType((String)"SM_xxxx_MIR_CORN0__20070223T061024_20070223T062500_001_001_0.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isAuxECMWFType((String)"SM_TEST_MIR_OSUDP2_20121118T143742_20121118T153047_306_002_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isAuxECMWFType((String)"SM_OPER_MIR_BWLD1C_20121118T002733_20121118T012104_116_001_1.zip"));
    }

    @Test
    public void testIsSmUserFormat() {
        Assert.assertTrue((boolean)SmosUtils.isSmUserFormat((String)"SM_TEST_MIR_SMUDP2_20070225T041815_20070225T050750_306_001_8.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isSmUserFormat((String)"SM_OPER_MIR_SCSF1C_20100315T144805_20100315T154207_330_001_1"));
    }

    @Test
    public void testIsSmAnalysisFormat() {
        Assert.assertTrue((boolean)SmosUtils.isSmAnalysisFormat((String)"SM_TEST_MIR_SMDAP2_20121117T183648_20121117T193048_304_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isSmAnalysisFormat((String)"SM_OPER_MIR_SCSF1C_20100315T144805_20100315T154207_330_001_1"));
    }

    @Test
    public void testIsOsUserFormat() {
        Assert.assertTrue((boolean)SmosUtils.isOsUserFormat((String)"SM_TEST_MIR_OSUDP2_20070225T041815_20070225T050750_306_001_8.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isOsUserFormat((String)"SM_TEST_MIR_BWSF1C_20070223T112729_20070223T121644_141_000_0.zip"));
    }

    @Test
    public void testIsOsAnalysisFormat() {
        Assert.assertTrue((boolean)SmosUtils.isOsAnalysisFormat((String)"SM_TEST_MIR_OSDAP2_20070225T041815_20070225T050750_306_001_8.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isOsAnalysisFormat((String)"SM_OPER_MIR_OSUDP2_20091204T001853_20091204T011255_310_001_1.zip"));
    }

    @Test
    public void testIsDualPolBrowseFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDualPolBrowseFormat((String)"SM_OPER_MIR_BWLD1C_20100405T143038_20100405T152439_330_001_1.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isDualPolBrowseFormat((String)"SM_OPER_MIR_BWSD1C_20100201T134256_20100201T140057_324_001_1.HDR"));
        Assert.assertFalse((boolean)SmosUtils.isDualPolBrowseFormat((String)"SM_OPER_MIR_OSUDP2_20091204T001853_20091204T011255_310_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isDualPolBrowseFormat((String)"SM_OPER_MIR_SMDAP2_20111130T141947_20111130T151305_500_001_1.DBL"));
    }

    @Test
    public void testIsFullPolBrowseFormat() {
        Assert.assertTrue((boolean)SmosUtils.isFullPolBrowseFormat((String)"SM_OPER_MIR_BWLF1C_20100405T143038_20100405T152439_330_001_1.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isFullPolBrowseFormat((String)"SM_OPER_MIR_BWSF1C_20100201T134256_20100201T140057_324_001_1.HDR"));
        Assert.assertFalse((boolean)SmosUtils.isDualPolBrowseFormat((String)"SM_OPER_MIR_OSUDP2_20091204T001853_20091204T011255_310_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isDualPolBrowseFormat((String)"SM_OPER_MIR_SMDAP2_20111130T141947_20111130T151305_500_001_1.DBL"));
    }

    @Test
    public void testIsBrowseFormat() {
        Assert.assertTrue((boolean)SmosUtils.isBrowseFormat((String)"SM_OPER_MIR_BWLD1C_20100405T143038_20100405T152439_330_001_1.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isBrowseFormat((String)"SM_OPER_MIR_BWSD1C_20100201T134256_20100201T140057_324_001_1.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isBrowseFormat((String)"SM_OPER_MIR_BWLF1C_20100201T134256_20100201T140057_324_001_1.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isBrowseFormat((String)"SM_OPER_MIR_BWSF1C_20100201T134256_20100201T140057_324_001_1.HDR"));
        Assert.assertFalse((boolean)SmosUtils.isBrowseFormat((String)"SM_TEST_MIR_OSDAP2_20070225T041815_20070225T050750_306_001_8.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isBrowseFormat((String)"SM_OPER_MIR_OSUDP2_20091204T001853_20091204T011255_310_001_1.zip"));
    }

    @Test
    public void testIsDualPolScienceFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDualPolScienceFormat((String)"SM_OPER_MIR_SCSD1C_20100405T143038_20100405T152439_330_001_1.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isDualPolScienceFormat((String)"SM_OPER_MIR_SCLD1C_20100201T134256_20100201T140057_324_001_1.HDR"));
        Assert.assertFalse((boolean)SmosUtils.isDualPolScienceFormat((String)"SM_OPER_MIR_OSUDP2_20091204T001853_20091204T011255_310_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isDualPolScienceFormat((String)"SM_OPER_MIR_SMDAP2_20111130T141947_20111130T151305_500_001_1.DBL"));
    }

    @Test
    public void testIsFullPolScienceFormat() {
        Assert.assertTrue((boolean)SmosUtils.isFullPolScienceFormat((String)"SM_OPER_MIR_SCSF1C_20100405T143038_20100405T152439_330_001_1.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isFullPolScienceFormat((String)"SM_OPER_MIR_SCLF1C_20100201T134256_20100201T140057_324_001_1.HDR"));
        Assert.assertFalse((boolean)SmosUtils.isFullPolScienceFormat((String)"SM_OPER_MIR_OSUDP2_20091204T001853_20091204T011255_310_001_1.zip"));
        Assert.assertFalse((boolean)SmosUtils.isFullPolScienceFormat((String)"SM_OPER_MIR_SMDAP2_20111130T141947_20111130T151305_500_001_1.DBL"));
    }

    @Test
    public void testIsCompressedFile() {
        File fileWithoutExtension = new File("a_file_without_extension");
        Assert.assertFalse((boolean)SmosUtils.isCompressedFile((File)fileWithoutExtension));
        File textFile = new File("bla.txt");
        Assert.assertFalse((boolean)SmosUtils.isCompressedFile((File)textFile));
        File zipFile = new File("gna.zip");
        Assert.assertTrue((boolean)SmosUtils.isCompressedFile((File)zipFile));
        File zipFileWithCapitalExtension = new File("firlefanz.ZIP");
        Assert.assertTrue((boolean)SmosUtils.isCompressedFile((File)zipFileWithCapitalExtension));
    }

    @Test
    public void testIsDggTlvFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDggTlvFormat((String)"SM_OPER_AUX_DGGTLV_20110605T000000_20110608T000000_307_001_3.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isDggTlvFormat((String)"SM_OPER_MIR_BWLF1C_20100201T134256_20100201T140057_324_001_1.DBL"));
    }

    @Test
    public void testIsDggTfoFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDggTfoFormat((String)"SM_OPER_AUX_DGGTFO_20140411T001616_20500101T000000_400_001_1.HDR"));
        Assert.assertFalse((boolean)SmosUtils.isDggTfoFormat((String)"SM_OPER_MIR_OSUDP2_20091204T001853_20091204T011255_310_001_1.DBL"));
    }

    @Test
    public void testIsDggRouFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDggRouFormat((String)"SM_OPER_AUX_DGGROU_20110605T000000_20110608T000000_307_001_3.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isDggRouFormat((String)"SM_OPER_MIR_OSUDP2_20091204T001853_20091204T011255_310_001_1.DBL"));
    }

    @Test
    public void testIsDggRfiFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDggRfiFormat((String)"SM_OPER_AUX_DGGRFI_20140413T003825_20500101T000000_400_001_1.HDR"));
        Assert.assertFalse((boolean)SmosUtils.isDggRfiFormat((String)"SM_TEST_MIR_OSUDP2_20121118T143742_20121118T153047_306_002_1.DBL"));
    }

    @Test
    public void testIsDggFloFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDggFloFormat((String)"SM_OPER_AUX_DGGFLO_20140413T003825_20500101T000000_400_001_1.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isDggFloFormat((String)"SM_TEST_AUX_RFI____20070101T235959_20781231T235959_00000001.DBL"));
    }

    @Test
    public void testIsLsMaskFormat() {
        Assert.assertTrue((boolean)SmosUtils.isLsMaskFormat((String)"SM_OPER_AUX_LSMASK_20050101T000000_20500101T000000_300_003_3.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isLsMaskFormat((String)"SM_TEST_MIR_GMATD__20121117T020130_20781231T235959_115_001_3.DBL"));
    }

    @Test
    public void testIsVTecFormat() {
        Assert.assertTrue((boolean)SmosUtils.isVTecFormat((String)"SM_OPER_AUX_VTEC_C_20100118T230000_20100120T010000_306_001_3.HDR"));
        Assert.assertTrue((boolean)SmosUtils.isVTecFormat((String)"SM_OPER_AUX_VTEC_P_20100207T230000_20100209T010000_306_001_3.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isVTecFormat((String)"SO_GNOT_MIR_OS__2__20070223T061024_20070223T062500_001_001_0.DBL"));
    }

    @Test
    public void testIsDffLaiFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDffLaiFormat((String)"SM_OPER_AUX_DFFLAI_20100117T000000_20100216T000000_306_001_3.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isDffLaiFormat((String)"SM_TEST_MIR_SMDAP2_20121118T135052_20121118T144140_303_007_1.DBL"));
    }

    @Test
    public void testIsDffSnoFormat() {
        Assert.assertTrue((boolean)SmosUtils.isDffSnoFormat((String)"SM_OPER_AUX_DFFSNO_20170226T000000_20170228T140000_102_001_3.DBL"));
        Assert.assertFalse((boolean)SmosUtils.isDffSnoFormat((String)"SM_TEST_MIR_SMDAP2_20121118T135052_20121118T144140_303_007_1.DBL"));
    }

    static boolean isSameDay(Date date_1, Date date_2) {
        if (date_1 == null || date_2 == null) {
            return false;
        }
        Calendar cal_1 = GregorianCalendar.getInstance();
        cal_1.setTime(date_1);
        Calendar cal_2 = GregorianCalendar.getInstance();
        cal_2.setTime(date_2);
        return cal_1.get(1) == cal_2.get(1) && cal_1.get(2) == cal_2.get(2) && cal_1.get(5) == cal_2.get(5);
    }
}

