/*
 * Decompiled with CFR 0.152.
 */
package org.esa.s2tbx.dataio.openjp2;

import com.sun.jna.Pointer;
import com.sun.jna.ptr.PointerByReference;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.DataBufferInt;
import java.awt.image.DataBufferShort;
import java.awt.image.DataBufferUShort;
import java.awt.image.PixelInterleavedSampleModel;
import java.awt.image.Raster;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.esa.s2tbx.dataio.openjp2.RasterUtils;
import org.esa.s2tbx.dataio.openjp2.TileImageDescriptor;
import org.esa.s2tbx.dataio.openjp2.library.Callbacks;
import org.esa.s2tbx.dataio.openjp2.library.OpenJp2;
import org.esa.s2tbx.dataio.openjp2.struct.DecompressParams;
import org.esa.s2tbx.dataio.openjp2.struct.DecompressionCodec;
import org.esa.s2tbx.dataio.openjp2.struct.Image;
import org.esa.s2tbx.dataio.openjp2.struct.ImageComponent;
import org.esa.snap.core.util.SystemUtils;
import sun.awt.image.SunWritableRaster;

public class OpenJP2Decoder
implements AutoCloseable {
    private static final ExecutorService executor = Executors.newFixedThreadPool(Math.min(Runtime.getRuntime().availableProcessors() / 2, 4));
    private PointerByReference pStream;
    private DecompressParams parameters;
    private DecompressionCodec pCodec;
    private PointerByReference pImage;
    private int width;
    private int height;
    private final Path tileFile;
    private int resolution;
    private int layer;
    private int dataType;
    private int tileIndex;
    private int bandIndex;
    private Logger logger = SystemUtils.LOG;
    private final Set<Path> pendingWrites;
    private Function<Path, Void> writeCompletedCallback;

    public OpenJP2Decoder(Path cacheDir, Path file, int bandIndex, int dataType, int resolution, int layer, int tileIndex) {
        this.dataType = dataType;
        this.resolution = resolution;
        this.layer = layer;
        this.tileIndex = tileIndex;
        this.bandIndex = bandIndex == -1 ? 0 : bandIndex;
        this.tileFile = cacheDir.resolve(file.getFileName().toString().replace(".", "_").toLowerCase() + "_" + String.valueOf(tileIndex) + "_" + String.valueOf(resolution) + "_" + String.valueOf(this.bandIndex) + ".raw");
        this.pStream = OpenJp2.opj_stream_create_default_file_stream(file.toAbsolutePath().toString(), 1);
        if (this.pStream == null || this.pStream.getValue() == null) {
            throw new RuntimeException("Failed to create the stream from the file");
        }
        this.parameters = this.initDecodeParams(file);
        this.pCodec = this.setupDecoder(this.parameters);
        this.pImage = new PointerByReference();
        OpenJp2.opj_read_header(this.pStream, this.pCodec, this.pImage);
        Image jImage = RasterUtils.dereference(Image.class, this.pImage.getValue());
        ImageComponent component = ((ImageComponent[])jImage.comps.toArray(jImage.numcomps))[this.bandIndex];
        this.width = component.w;
        this.height = component.h;
        this.pendingWrites = Collections.synchronizedSet(new HashSet());
        this.writeCompletedCallback = value -> {
            if (value != null) {
                this.pendingWrites.remove(value);
            }
            return null;
        };
    }

    public Dimension getImageDimensions() throws IOException {
        return new Dimension(this.width, this.height);
    }

    public Raster read() throws IOException {
        return this.decompress(null);
    }

    public Raster read(Rectangle rectangle) throws IOException {
        return this.decompress(rectangle);
    }

    @Override
    public void close() {
        try {
            if (this.pImage != null && this.pImage.getValue() != null) {
                OpenJp2.opj_image_destroy(this.pImage.getValue());
            }
            if (this.pCodec != null) {
                OpenJp2.opj_destroy_codec(this.pCodec.getPointer());
            }
            if (this.pStream != null && this.pStream.getValue() != null) {
                OpenJp2.opj_stream_destroy(this.pStream);
            }
        }
        catch (Exception ex) {
            this.logger.warning(ex.getMessage());
        }
    }

    private ImageComponent[] decode() {
        Image jImage = RasterUtils.dereference(Image.class, this.pImage.getValue());
        if (this.parameters.nb_tile_to_decode == 0) {
            if (OpenJp2.opj_set_decode_area(this.pCodec, jImage, this.parameters.DA_x0, this.parameters.DA_y0, this.parameters.DA_x1, this.parameters.DA_y1) == 0) {
                throw new RuntimeException("Failed to set the decoded area");
            }
            if (OpenJp2.opj_decode(this.pCodec, this.pStream, jImage) == 0 && OpenJp2.opj_end_decompress(this.pCodec, this.pStream) != 0) {
                throw new RuntimeException("Failed to decode image");
            }
        } else if (OpenJp2.opj_get_decoded_tile(this.pCodec, this.pStream, jImage, this.parameters.tile_index) == 0) {
            throw new RuntimeException("Failed to decode tile");
        }
        jImage = RasterUtils.dereference(Image.class, this.pImage.getValue());
        ImageComponent[] comps = (ImageComponent[])jImage.comps.toArray(jImage.numcomps);
        if (jImage.color_space != 3 && jImage.numcomps == 3 && comps[0].dx == comps[0].dy && comps[1].dx != 1) {
            jImage.color_space = 3;
        } else if (jImage.numcomps <= 2) {
            jImage.color_space = 2;
        }
        return comps;
    }

    private DecompressParams initDecodeParams(Path inputFile) {
        DecompressParams params = new DecompressParams();
        params.decod_format = -1;
        params.cod_format = -1;
        OpenJp2.opj_set_default_decoder_parameters(params.core);
        params.decod_format = RasterUtils.getFileFormat(inputFile);
        params.cod_format = RasterUtils.getFormat("jp2");
        params.core.cp_reduce = this.resolution;
        params.core.cp_layer = this.layer;
        params.tile_index = this.tileIndex;
        params.nb_tile_to_decode = params.tile_index >= 0 ? 1 : 0;
        return params;
    }

    private DecompressionCodec setupDecoder(DecompressParams params) {
        DecompressionCodec.ByReference codec;
        switch (params.decod_format) {
            case 0: {
                codec = OpenJp2.opj_create_decompress(0);
                break;
            }
            case 1: {
                codec = OpenJp2.opj_create_decompress(2);
                break;
            }
            case 2: {
                codec = OpenJp2.opj_create_decompress(1);
                break;
            }
            default: {
                throw new RuntimeException("File is not coded with JPEG-2000");
            }
        }
        if (SystemUtils.LOG.getLevel().intValue() <= Level.FINE.intValue()) {
            OpenJp2.opj_set_info_handler(codec, new Callbacks.MessageFunction(){

                @Override
                public void invoke(Pointer msg, Pointer client_data) {
                    OpenJP2Decoder.this.logger.info(msg.getString(0L));
                }

                @Override
                public Pointer invoke(Pointer p_codec) {
                    return p_codec;
                }
            }, null);
            OpenJp2.opj_set_warning_handler(codec, new Callbacks.MessageFunction(){

                @Override
                public void invoke(Pointer msg, Pointer client_data) {
                    OpenJP2Decoder.this.logger.warning(msg.getString(0L));
                }

                @Override
                public Pointer invoke(Pointer p_codec) {
                    return p_codec;
                }
            }, null);
        }
        OpenJp2.opj_set_error_handler(codec, new Callbacks.MessageFunction(){

            @Override
            public void invoke(Pointer msg, Pointer client_data) {
                OpenJP2Decoder.this.logger.severe(msg.getString(0L));
            }

            @Override
            public Pointer invoke(Pointer p_codec) {
                return p_codec;
            }
        }, null);
        int setupDecoder = OpenJp2.opj_setup_decoder(codec, params.core);
        if (setupDecoder == 0) {
            throw new RuntimeException("Failed to setup decoder");
        }
        return codec;
    }

    private Raster decompress(Rectangle roi) throws IOException {
        DataBuffer buffer;
        int height;
        int width;
        while (this.pendingWrites.contains(this.tileFile)) {
            Thread.yield();
        }
        int[] bandOffsets = new int[]{0};
        if (!Files.exists(this.tileFile, new LinkOption[0])) {
            ImageComponent[] components = this.decode();
            ImageComponent component = components[this.bandIndex];
            width = component.w;
            height = component.h;
            int[] pixels = component.data.getPointer().getIntArray(0L, component.w * component.h);
            executor.submit(() -> {
                try {
                    this.pendingWrites.add(this.tileFile);
                    RasterUtils.write(component.w, component.h, pixels, this.dataType, this.tileFile, this.writeCompletedCallback);
                }
                catch (Exception ex) {
                    this.logger.warning(ex.getMessage());
                }
            });
            if (components.length > 1) {
                for (int i = 0; i < components.length; ++i) {
                    int index = i;
                    if (index == this.bandIndex) continue;
                    executor.submit(() -> {
                        try {
                            String fName = this.tileFile.getFileName().toString();
                            fName = fName.substring(0, fName.lastIndexOf("_")) + "_" + String.valueOf(index) + ".raw";
                            Path otherBandFile = Paths.get(fName, new String[0]);
                            this.pendingWrites.add(otherBandFile);
                            RasterUtils.write(components[index].w, components[index].h, components[index].data.getPointer().getIntArray(0L, components[index].w * components[index].h), this.dataType, otherBandFile, this.writeCompletedCallback);
                        }
                        catch (Exception ex) {
                            this.logger.warning(ex.getMessage());
                        }
                    });
                }
            }
            switch (this.dataType) {
                case 0: {
                    buffer = RasterUtils.extractROIAsByteBuffer(pixels, width, height, roi);
                    break;
                }
                case 1: {
                    buffer = RasterUtils.extractROIAsUShortBuffer(pixels, width, height, roi);
                    break;
                }
                case 2: {
                    buffer = RasterUtils.extractROIAsShortBuffer(pixels, width, height, roi);
                    break;
                }
                case 3: {
                    buffer = RasterUtils.extractROI(pixels, width, height, roi);
                    break;
                }
                default: {
                    throw new UnsupportedOperationException("Source buffer type not supported");
                }
            }
            if (roi != null) {
                width = roi.width;
                height = roi.height;
            }
        } else {
            switch (this.dataType) {
                case 0: {
                    TileImageDescriptor<byte[]> fileDescriptor = RasterUtils.readAsByteArray(this.tileFile, roi);
                    width = fileDescriptor.getWidth();
                    height = fileDescriptor.getHeight();
                    buffer = new DataBufferByte(fileDescriptor.getDataArray(), width * height);
                    break;
                }
                case 1: {
                    TileImageDescriptor<byte[]> fileDescriptor = RasterUtils.readAsShortArray(this.tileFile, roi);
                    width = fileDescriptor.getWidth();
                    height = fileDescriptor.getHeight();
                    buffer = new DataBufferUShort(fileDescriptor.getDataArray(), width * height);
                    break;
                }
                case 2: {
                    TileImageDescriptor<byte[]> fileDescriptor = RasterUtils.readAsShortArray(this.tileFile, roi);
                    width = fileDescriptor.getWidth();
                    height = fileDescriptor.getHeight();
                    buffer = new DataBufferShort(fileDescriptor.getDataArray(), width * height);
                    break;
                }
                case 3: {
                    TileImageDescriptor<byte[]> fileDescriptor = RasterUtils.readAsIntArray(this.tileFile, roi);
                    width = fileDescriptor.getWidth();
                    height = fileDescriptor.getHeight();
                    buffer = new DataBufferInt(fileDescriptor.getDataArray(), width * height);
                    break;
                }
                default: {
                    throw new UnsupportedOperationException("Source buffer type not supported");
                }
            }
        }
        while (this.pendingWrites.contains(this.tileFile)) {
            Thread.yield();
        }
        PixelInterleavedSampleModel sampleModel = new PixelInterleavedSampleModel(this.dataType, width, height, 1, width, bandOffsets);
        SunWritableRaster raster = null;
        try {
            raster = new SunWritableRaster(sampleModel, buffer, new Point(0, 0));
        }
        catch (Exception e) {
            this.logger.severe(e.getMessage());
        }
        return raster;
    }
}

