/*
 * Decompiled with CFR 0.152.
 */
package org.esa.beam.gpf.operators.standard.reproject;

import com.bc.ceres.glevel.MultiLevelImage;
import com.bc.ceres.glevel.MultiLevelModel;
import com.bc.ceres.glevel.MultiLevelSource;
import com.bc.ceres.glevel.support.AbstractMultiLevelSource;
import com.bc.ceres.glevel.support.DefaultMultiLevelImage;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import javax.media.jai.ImageLayout;
import javax.media.jai.Interpolation;
import javax.media.jai.JAI;
import org.esa.beam.framework.datamodel.Band;
import org.esa.beam.framework.datamodel.ColorPaletteDef;
import org.esa.beam.framework.datamodel.CrsGeoCoding;
import org.esa.beam.framework.datamodel.FlagCoding;
import org.esa.beam.framework.datamodel.GeoCoding;
import org.esa.beam.framework.datamodel.GeoPos;
import org.esa.beam.framework.datamodel.ImageGeometry;
import org.esa.beam.framework.datamodel.ImageInfo;
import org.esa.beam.framework.datamodel.IndexCoding;
import org.esa.beam.framework.datamodel.Product;
import org.esa.beam.framework.datamodel.ProductData;
import org.esa.beam.framework.datamodel.ProductNode;
import org.esa.beam.framework.datamodel.ProductNodeGroup;
import org.esa.beam.framework.datamodel.RasterDataNode;
import org.esa.beam.framework.datamodel.SampleCoding;
import org.esa.beam.framework.dataop.dem.ElevationModel;
import org.esa.beam.framework.dataop.dem.ElevationModelDescriptor;
import org.esa.beam.framework.dataop.dem.ElevationModelRegistry;
import org.esa.beam.framework.dataop.dem.Orthorectifier;
import org.esa.beam.framework.dataop.dem.Orthorectifier2;
import org.esa.beam.framework.dataop.resamp.Resampling;
import org.esa.beam.framework.gpf.Operator;
import org.esa.beam.framework.gpf.OperatorException;
import org.esa.beam.framework.gpf.OperatorSpi;
import org.esa.beam.framework.gpf.annotations.OperatorMetadata;
import org.esa.beam.framework.gpf.annotations.Parameter;
import org.esa.beam.framework.gpf.annotations.SourceProduct;
import org.esa.beam.framework.gpf.annotations.TargetProduct;
import org.esa.beam.gpf.operators.standard.reproject.Log10OpImage;
import org.esa.beam.gpf.operators.standard.reproject.ReplaceNaNOpImage;
import org.esa.beam.gpf.operators.standard.reproject.Reproject;
import org.esa.beam.jai.ImageManager;
import org.esa.beam.jai.ResolutionLevel;
import org.esa.beam.util.Debug;
import org.esa.beam.util.ProductUtils;
import org.esa.beam.util.io.FileUtils;
import org.geotools.factory.Hints;
import org.geotools.referencing.CRS;
import org.opengis.referencing.FactoryException;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.cs.AxisDirection;
import org.opengis.referencing.operation.TransformException;

@OperatorMetadata(alias="Reproject", category="Geometric Operations", version="1.0", authors="Marco Z\u00fchlke, Marco Peters, Ralf Quast, Norman Fomferra", copyright="(c) 2009 by Brockmann Consult", description="Reprojection of a source product to a target Coordinate Reference System.", internal=false)
public class ReprojectionOp
extends Operator {
    @SourceProduct(alias="source", description="The product which will be reprojected.")
    private Product sourceProduct;
    @SourceProduct(alias="collocateWith", optional=true, label="Collocation product", description="The source product will be collocated with this product.")
    private Product collocationProduct;
    @TargetProduct
    private Product targetProduct;
    @Parameter(description="A file which contains the target Coordinate Reference System in WKT format.")
    private File wktFile;
    @Parameter(description="A text specifying the target Coordinate Reference System, either in WKT or as an authority code. For appropriate EPSG authority codes see (www.epsg-registry.org). AUTO authority can be used with code 42001 (UTM), and 42002 (Transverse Mercator) where the scene center is used as reference. Examples: EPSG:4326, AUTO:42001")
    private String crs;
    @Parameter(alias="resampling", label="Resampling Method", description="The method used for resampling of floating-point raster data.", valueSet={"Nearest", "Bilinear", "Bicubic"}, defaultValue="Nearest")
    private String resamplingName;
    @Parameter(description="The X-position of the reference pixel.")
    private Double referencePixelX;
    @Parameter(description="The Y-position of the reference pixel.")
    private Double referencePixelY;
    @Parameter(description="The easting of the reference pixel.")
    private Double easting;
    @Parameter(description="The northing of the reference pixel.")
    private Double northing;
    @Parameter(description="The orientation of the output product (in degree).", defaultValue="0", interval="[-360,360]")
    private Double orientation;
    @Parameter(description="The pixel size in X direction given in CRS units.")
    private Double pixelSizeX;
    @Parameter(description="The pixel size in Y direction given in CRS units.")
    private Double pixelSizeY;
    @Parameter(description="The width of the target product.")
    private Integer width;
    @Parameter(description="The height of the target product.")
    private Integer height;
    @Parameter(description="The tile size in X direction.")
    private Integer tileSizeX;
    @Parameter(description="The tile size in Y direction.")
    private Integer tileSizeY;
    @Parameter(description="Whether the source product should be orthorectified. (Not applicable to all products)", defaultValue="false")
    private boolean orthorectify;
    @Parameter(description="The name of the elevation model for the orthorectification. If not given tie-point data is used.")
    private String elevationModelName;
    @Parameter(description="The value used to indicate no-data.")
    private Double noDataValue;
    @Parameter(description="Whether tie-point grids should be included in the output product.", defaultValue="true")
    private boolean includeTiePointGrids;
    @Parameter(description="Whether to add delta longitude and latitude bands.", defaultValue="false")
    private boolean addDeltaBands;
    private ElevationModel elevationModel;
    private MultiLevelModel sourceModel;
    private MultiLevelModel targetModel;
    private Reproject reprojection;

    @Override
    public void initialize() throws OperatorException {
        Dimension tileSize;
        this.validateCrsParameters();
        this.validateResamplingParameter();
        this.validateReferencingParameters();
        this.validateTargetGridParameters();
        CoordinateReferenceSystem targetCrs = this.createTargetCRS();
        ImageGeometry targetImageGeometry = this.createImageGeometry(targetCrs);
        Rectangle targetRect = targetImageGeometry.getImageRect();
        this.targetProduct = new Product("projected_" + this.sourceProduct.getName(), this.sourceProduct.getProductType(), targetRect.width, targetRect.height);
        this.targetProduct.setDescription(this.sourceProduct.getDescription());
        if (this.tileSizeX != null && this.tileSizeY != null) {
            tileSize = new Dimension(this.tileSizeX, this.tileSizeY);
        } else {
            tileSize = ImageManager.getPreferredTileSize((Product)this.targetProduct);
            Dimension sourceProductPreferredTileSize = this.sourceProduct.getPreferredTileSize();
            if (sourceProductPreferredTileSize != null && sourceProductPreferredTileSize.width == this.sourceProduct.getSceneRasterWidth()) {
                tileSize.width = this.targetProduct.getSceneRasterWidth();
                tileSize.height = Math.min(sourceProductPreferredTileSize.height, this.targetProduct.getSceneRasterHeight());
            }
        }
        this.targetProduct.setPreferredTileSize(tileSize);
        if (this.orthorectify) {
            this.elevationModel = this.createElevationModel();
        }
        ProductUtils.copyMetadata((Product)this.sourceProduct, (Product)this.targetProduct);
        ProductUtils.copyFlagCodings((Product)this.sourceProduct, (Product)this.targetProduct);
        this.copyIndexCoding();
        try {
            this.targetProduct.setGeoCoding((GeoCoding)new CrsGeoCoding(targetImageGeometry.getMapCrs(), targetRect, targetImageGeometry.getImage2MapTransform()));
        }
        catch (Exception e) {
            throw new OperatorException(e);
        }
        ProductData.UTC meanTime = this.getSourceMeanTime();
        this.targetProduct.setStartTime(meanTime);
        this.targetProduct.setEndTime(meanTime);
        this.sourceModel = ImageManager.getMultiLevelModel((RasterDataNode)this.sourceProduct.getBandAt(0));
        this.targetModel = ImageManager.createMultiLevelModel((ProductNode)this.targetProduct);
        this.reprojection = new Reproject(this.targetModel.getLevelCount());
        this.reprojectRasterDataNodes((RasterDataNode[])this.sourceProduct.getBands());
        if (this.includeTiePointGrids) {
            this.reprojectRasterDataNodes((RasterDataNode[])this.sourceProduct.getTiePointGrids());
        }
        ProductUtils.copyVectorData((Product)this.sourceProduct, (Product)this.targetProduct);
        ProductUtils.copyMasks((Product)this.sourceProduct, (Product)this.targetProduct);
        ProductUtils.copyOverlayMasks((Product)this.sourceProduct, (Product)this.targetProduct);
        this.targetProduct.setAutoGrouping(this.sourceProduct.getAutoGrouping());
        if (this.addDeltaBands) {
            this.addDeltaBands();
        }
    }

    @Override
    public void dispose() {
        if (this.elevationModel != null) {
            this.elevationModel.dispose();
        }
        super.dispose();
    }

    private ElevationModel createElevationModel() throws OperatorException {
        if (this.elevationModelName != null) {
            ElevationModelDescriptor demDescriptor = ElevationModelRegistry.getInstance().getDescriptor(this.elevationModelName);
            if (!demDescriptor.isDemInstalled()) {
                throw new OperatorException("DEM not installed: " + this.elevationModelName);
            }
            return demDescriptor.createDem(Resampling.BILINEAR_INTERPOLATION);
        }
        return null;
    }

    private GeoCoding getSourceGeoCoding(RasterDataNode sourceBand) {
        if (this.orthorectify && sourceBand.canBeOrthorectified()) {
            return this.createOrthorectifier(sourceBand);
        }
        return sourceBand.getGeoCoding();
    }

    private Orthorectifier createOrthorectifier(RasterDataNode sourceBand) {
        return new Orthorectifier2(sourceBand.getSceneRasterWidth(), sourceBand.getSceneRasterHeight(), sourceBand.getPointing(), this.elevationModel, 25);
    }

    private void reprojectRasterDataNodes(RasterDataNode[] rasterDataNodes) {
        for (RasterDataNode raster : rasterDataNodes) {
            this.reprojectSourceRaster(raster);
        }
    }

    private void reprojectSourceRaster(RasterDataNode sourceRaster) {
        MultiLevelImage sourceImage;
        int targetDataType;
        if (sourceRaster.isScalingApplied()) {
            targetDataType = sourceRaster.getGeophysicalDataType();
            sourceImage = sourceRaster.getGeophysicalImage();
        } else {
            targetDataType = sourceRaster.getDataType();
            sourceImage = sourceRaster.getSourceImage();
        }
        Number targetNoDataValue = this.getTargetNoDataValue(sourceRaster, targetDataType);
        Band targetBand = this.targetProduct.addBand(sourceRaster.getName(), targetDataType);
        targetBand.setLog10Scaled(sourceRaster.isLog10Scaled());
        targetBand.setNoDataValue(targetNoDataValue.doubleValue());
        targetBand.setNoDataValueUsed(true);
        targetBand.setDescription(sourceRaster.getDescription());
        targetBand.setUnit(sourceRaster.getUnit());
        GeoCoding sourceGeoCoding = this.getSourceGeoCoding(sourceRaster);
        String exp = sourceRaster.getValidMaskExpression();
        if (exp != null) {
            sourceImage = this.createNoDataReplacedImage(sourceRaster, targetNoDataValue);
        }
        Interpolation resampling = this.getResampling(targetBand);
        MultiLevelImage projectedImage = this.createProjectedImage(sourceGeoCoding, sourceImage, targetBand, resampling);
        if (this.mustReplaceNaN(sourceRaster, targetDataType, targetNoDataValue.doubleValue())) {
            projectedImage = this.createNaNReplacedImage(projectedImage, targetNoDataValue.doubleValue());
        }
        if (targetBand.isLog10Scaled()) {
            projectedImage = this.createLog10ScaledImage(projectedImage);
        }
        targetBand.setSourceImage(projectedImage);
        if (sourceRaster instanceof Band) {
            Band sourceBand = (Band)sourceRaster;
            ProductUtils.copySpectralBandProperties((Band)sourceBand, (Band)targetBand);
            FlagCoding sourceFlagCoding = sourceBand.getFlagCoding();
            IndexCoding sourceIndexCoding = sourceBand.getIndexCoding();
            if (sourceFlagCoding != null) {
                String flagCodingName = sourceFlagCoding.getName();
                FlagCoding destFlagCoding = (FlagCoding)this.targetProduct.getFlagCodingGroup().get(flagCodingName);
                targetBand.setSampleCoding((SampleCoding)destFlagCoding);
            } else if (sourceIndexCoding != null) {
                String indexCodingName = sourceIndexCoding.getName();
                IndexCoding destIndexCoding = (IndexCoding)this.targetProduct.getIndexCodingGroup().get(indexCodingName);
                targetBand.setSampleCoding((SampleCoding)destIndexCoding);
            }
        }
    }

    private MultiLevelImage createLog10ScaledImage(final MultiLevelImage projectedImage) {
        return new DefaultMultiLevelImage((MultiLevelSource)new AbstractMultiLevelSource(projectedImage.getModel()){

            public RenderedImage createImage(int level) {
                return new Log10OpImage(projectedImage.getImage(level));
            }
        });
    }

    private boolean mustReplaceNaN(RasterDataNode sourceRaster, int targetDataType, double targetNoDataValue) {
        boolean isFloat = ProductData.isFloatingPointType((int)targetDataType);
        boolean isNoDataGiven = sourceRaster.isNoDataValueUsed() || this.noDataValue != null;
        boolean isNoDataNaN = Double.isNaN(targetNoDataValue);
        return isFloat && isNoDataGiven && !isNoDataNaN;
    }

    private Number getTargetNoDataValue(RasterDataNode sourceRaster, int targetDataType) {
        double targetNoDataValue = Double.NaN;
        if (this.noDataValue != null) {
            targetNoDataValue = this.noDataValue;
        } else if (sourceRaster.isNoDataValueUsed()) {
            targetNoDataValue = sourceRaster.getNoDataValue();
        }
        Number targetNoDataNumber = targetDataType == 10 ? (Number)((byte)targetNoDataValue) : (Number)(targetDataType == 11 || targetDataType == 20 ? (Number)((short)targetNoDataValue) : (Number)(targetDataType == 12 || targetDataType == 21 ? (Number)((int)targetNoDataValue) : (Number)(targetDataType == 30 ? (Number)Float.valueOf((float)targetNoDataValue) : (Number)targetNoDataValue)));
        return targetNoDataNumber;
    }

    private MultiLevelImage createNaNReplacedImage(final MultiLevelImage projectedImage, final double value) {
        return new DefaultMultiLevelImage((MultiLevelSource)new AbstractMultiLevelSource(this.targetModel){

            public RenderedImage createImage(int targetLevel) {
                return new ReplaceNaNOpImage(projectedImage.getImage(targetLevel), value);
            }
        });
    }

    private MultiLevelImage createNoDataReplacedImage(RasterDataNode rasterDataNode, Number noData) {
        return ImageManager.createMaskedGeophysicalImage((RasterDataNode)rasterDataNode, (Number)noData);
    }

    private MultiLevelImage createProjectedImage(GeoCoding sourceGeoCoding, final MultiLevelImage sourceImage, final Band targetBand, final Interpolation resampling) {
        final CoordinateReferenceSystem sourceModelCrs = ImageManager.getModelCrs((GeoCoding)sourceGeoCoding);
        final CoordinateReferenceSystem targetModelCrs = ImageManager.getModelCrs((GeoCoding)this.targetProduct.getGeoCoding());
        final AffineTransform i2mSourceProduct = ImageManager.getImageToModelTransform((GeoCoding)sourceGeoCoding);
        final AffineTransform i2mTargetProduct = ImageManager.getImageToModelTransform((GeoCoding)this.targetProduct.getGeoCoding());
        return new DefaultMultiLevelImage((MultiLevelSource)new AbstractMultiLevelSource(this.targetModel){

            public RenderedImage createImage(int targetLevel) {
                double targetScale = ReprojectionOp.this.targetModel.getScale(targetLevel);
                int sourceLevel = ReprojectionOp.this.sourceModel.getLevel(targetScale);
                RenderedImage leveledSourceImage = sourceImage.getImage(sourceLevel);
                Rectangle sourceBounds = new Rectangle(leveledSourceImage.getWidth(), leveledSourceImage.getHeight());
                AffineTransform i2mSource = ReprojectionOp.this.sourceModel.getImageToModelTransform(sourceLevel);
                i2mSource.concatenate(ReprojectionOp.this.sourceModel.getModelToImageTransform(0));
                i2mSource.concatenate(i2mSourceProduct);
                ImageGeometry sourceGeometry = new ImageGeometry(sourceBounds, sourceModelCrs, i2mSource);
                ImageLayout imageLayout = ImageManager.createSingleBandedImageLayout((int)ImageManager.getDataBufferType((int)targetBand.getDataType()), (int)ReprojectionOp.this.targetProduct.getSceneRasterWidth(), (int)ReprojectionOp.this.targetProduct.getSceneRasterHeight(), (Dimension)ReprojectionOp.this.targetProduct.getPreferredTileSize(), (ResolutionLevel)ResolutionLevel.create((MultiLevelModel)this.getModel(), (int)targetLevel));
                Rectangle targetBounds = new Rectangle(imageLayout.getWidth(null), imageLayout.getHeight(null));
                AffineTransform i2mTarget = this.getModel().getImageToModelTransform(targetLevel);
                i2mTarget.concatenate(this.getModel().getModelToImageTransform(0));
                i2mTarget.concatenate(i2mTargetProduct);
                ImageGeometry targetGeometry = new ImageGeometry(targetBounds, targetModelCrs, i2mTarget);
                Hints hints = new Hints(JAI.KEY_IMAGE_LAYOUT, (Object)imageLayout);
                hints.put((Object)Hints.LENIENT_DATUM_SHIFT, (Object)Boolean.TRUE);
                Dimension tileSize = ImageManager.getPreferredTileSize((Product)ReprojectionOp.this.targetProduct);
                try {
                    return ReprojectionOp.this.reprojection.reproject(leveledSourceImage, sourceGeometry, targetGeometry, targetBand.getNoDataValue(), resampling, hints, targetLevel, tileSize);
                }
                catch (FactoryException | TransformException e) {
                    Debug.trace((Throwable)e);
                    throw new RuntimeException(e);
                }
            }
        });
    }

    private int getSourceLevel(MultiLevelModel sourceModel, int targetLevel) {
        int maxSourceLevel = sourceModel.getLevelCount() - 1;
        return maxSourceLevel < targetLevel ? maxSourceLevel : targetLevel;
    }

    private ProductData.UTC getSourceMeanTime() {
        ProductData.UTC startTime = this.sourceProduct.getStartTime();
        ProductData.UTC endTime = this.sourceProduct.getEndTime();
        Object meanTime = startTime != null && endTime != null ? new ProductData.UTC(0.5 * (startTime.getMJD() + endTime.getMJD())) : (startTime != null ? startTime : (endTime != null ? endTime : null));
        return meanTime;
    }

    private void copyIndexCoding() {
        ProductNodeGroup indexCodingGroup = this.sourceProduct.getIndexCodingGroup();
        for (int i = 0; i < indexCodingGroup.getNodeCount(); ++i) {
            IndexCoding sourceIndexCoding = (IndexCoding)indexCodingGroup.get(i);
            ProductUtils.copyIndexCoding((IndexCoding)sourceIndexCoding, (Product)this.targetProduct);
        }
    }

    private CoordinateReferenceSystem createTargetCRS() throws OperatorException {
        try {
            if (this.wktFile != null) {
                return CRS.parseWKT((String)FileUtils.readText((File)this.wktFile));
            }
            if (this.crs != null) {
                try {
                    return CRS.parseWKT((String)this.crs);
                }
                catch (FactoryException ignored) {
                    if (this.crs.matches("[0-9]*")) {
                        this.crs = "EPSG:" + this.crs;
                    }
                    if (this.crs.matches("AUTO:[0-9]*")) {
                        GeoPos centerGeoPos = ProductUtils.getCenterGeoPos((Product)this.sourceProduct);
                        this.crs = String.format("%s,%s,%s", this.crs, Float.valueOf(centerGeoPos.lon), Float.valueOf(centerGeoPos.lat));
                    }
                    return CRS.decode((String)this.crs, (boolean)true);
                }
            }
            if (this.collocationProduct != null && this.collocationProduct.getGeoCoding() != null) {
                return this.collocationProduct.getGeoCoding().getMapCRS();
            }
        }
        catch (IOException | FactoryException e) {
            throw new OperatorException(String.format("Target CRS could not be created: %s", e.getMessage()), e);
        }
        throw new OperatorException("Target CRS could not be created.");
    }

    protected void validateCrsParameters() {
        String msgPattern = "Invalid target CRS specification.\nSpecify {0} one of the ''wktFile'', ''crs'' or ''collocationProduct'' parameters.";
        if (this.wktFile == null && this.crs == null && this.collocationProduct == null) {
            throw new OperatorException(MessageFormat.format("Invalid target CRS specification.\nSpecify {0} one of the ''wktFile'', ''crs'' or ''collocationProduct'' parameters.", "at least"));
        }
        boolean crsDefined = false;
        String exceptionMsg = MessageFormat.format("Invalid target CRS specification.\nSpecify {0} one of the ''wktFile'', ''crs'' or ''collocationProduct'' parameters.", "only");
        if (this.wktFile != null) {
            crsDefined = true;
        }
        if (this.crs != null) {
            if (crsDefined) {
                throw new OperatorException(exceptionMsg);
            }
            crsDefined = true;
        }
        if (this.collocationProduct != null && crsDefined) {
            throw new OperatorException(exceptionMsg);
        }
    }

    private Interpolation getResampling(Band band) {
        int resampleType = this.getResampleType();
        if (!ProductData.isFloatingPointType((int)band.getDataType())) {
            resampleType = 0;
        }
        return Interpolation.getInstance((int)resampleType);
    }

    private int getResampleType() {
        int resamplingType = "Nearest".equalsIgnoreCase(this.resamplingName) ? 0 : ("Bilinear".equalsIgnoreCase(this.resamplingName) ? 1 : ("Bicubic".equalsIgnoreCase(this.resamplingName) ? 2 : -1));
        return resamplingType;
    }

    void validateResamplingParameter() {
        if (this.getResampleType() == -1) {
            throw new OperatorException("Invalid resampling method: " + this.resamplingName);
        }
    }

    void validateReferencingParameters() {
        if (!(this.referencePixelX == null && this.referencePixelY == null && this.easting == null && this.northing == null || this.referencePixelX != null && this.referencePixelY != null && this.easting != null && this.northing != null)) {
            throw new OperatorException("Invalid referencing parameters: \n'referencePixelX', 'referencePixelY', 'easting' and 'northing' have to be specified either all or not at all.");
        }
    }

    void validateTargetGridParameters() {
        if (this.pixelSizeX != null && this.pixelSizeY == null || this.pixelSizeX == null && this.pixelSizeY != null) {
            throw new OperatorException("'pixelSizeX' and 'pixelSizeY' must be specified both or not at all.");
        }
    }

    private ImageGeometry createImageGeometry(CoordinateReferenceSystem targetCrs) {
        ImageGeometry imageGeometry;
        if (this.collocationProduct != null) {
            imageGeometry = ImageGeometry.createCollocationTargetGeometry((Product)this.sourceProduct, (Product)this.collocationProduct);
        } else {
            imageGeometry = ImageGeometry.createTargetGeometry((Product)this.sourceProduct, (CoordinateReferenceSystem)targetCrs, (Double)this.pixelSizeX, (Double)this.pixelSizeY, (Integer)this.width, (Integer)this.height, (Double)this.orientation, (Double)this.easting, (Double)this.northing, (Double)this.referencePixelX, (Double)this.referencePixelY);
            AxisDirection targetAxisDirection = targetCrs.getCoordinateSystem().getAxis(1).getDirection();
            if (!AxisDirection.DISPLAY_DOWN.equals((Object)targetAxisDirection)) {
                imageGeometry.changeYAxisDirection();
            }
        }
        return imageGeometry;
    }

    private void addDeltaBands() {
        Band deltaLonBand = this.targetProduct.addBand("delta_lon_angular", "longitude - LON");
        deltaLonBand.setUnit("deg");
        deltaLonBand.setDescription("Delta between old longitude and new longitude in degree");
        deltaLonBand.setNoDataValueUsed(true);
        deltaLonBand.setNoDataValue(this.noDataValue == null ? Double.NaN : this.noDataValue);
        deltaLonBand.setImageInfo(this.createDeltaBandImageInfo(-0.015, 0.015));
        Band deltaLatBand = this.targetProduct.addBand("delta_lat_angular", "latitude - LAT");
        deltaLatBand.setUnit("deg");
        deltaLatBand.setDescription("Delta between old latitude and new latitude in degree");
        deltaLatBand.setNoDataValueUsed(true);
        deltaLatBand.setNoDataValue(this.noDataValue == null ? Double.NaN : this.noDataValue);
        deltaLatBand.setImageInfo(this.createDeltaBandImageInfo(-0.01, 0.01));
        Band deltaLonMetBand = this.targetProduct.addBand("delta_lon_metric", "cos(rad(LAT)) * 6378137 * rad(longitude - LON)");
        deltaLonMetBand.setUnit("m");
        deltaLonMetBand.setDescription("Delta between old longitude and new longitude in meters");
        deltaLonMetBand.setNoDataValueUsed(true);
        deltaLonMetBand.setNoDataValue(this.noDataValue == null ? Double.NaN : this.noDataValue);
        deltaLonMetBand.setImageInfo(this.createDeltaBandImageInfo(-1500.0, 1500.0));
        Band deltaLatMetBand = this.targetProduct.addBand("delta_lat_metric", "6378137 * rad(latitude - LAT)");
        deltaLatMetBand.setUnit("m");
        deltaLatMetBand.setDescription("Delta between old latitude and new latitude in meters");
        deltaLatMetBand.setNoDataValueUsed(true);
        deltaLatMetBand.setNoDataValue(this.noDataValue == null ? Double.NaN : this.noDataValue);
        deltaLatMetBand.setImageInfo(this.createDeltaBandImageInfo(-1000.0, 1000.0));
    }

    private ImageInfo createDeltaBandImageInfo(double p1, double p2) {
        return new ImageInfo(new ColorPaletteDef(new ColorPaletteDef.Point[]{new ColorPaletteDef.Point(p1, new Color(255, 0, 0)), new ColorPaletteDef.Point((p1 + p2) / 2.0, new Color(255, 255, 255)), new ColorPaletteDef.Point(p2, new Color(0, 0, 127))}));
    }

    public static class Spi
    extends OperatorSpi {
        public Spi() {
            super(ReprojectionOp.class);
        }
    }
}

