/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.watermask.operator;

import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.text.MessageFormat;
import java.util.Properties;
import javax.imageio.ImageIO;
import javax.media.jai.NullOpImage;
import javax.media.jai.OpImage;
import org.esa.snap.core.datamodel.GeoCoding;
import org.esa.snap.core.datamodel.GeoPos;
import org.esa.snap.core.datamodel.PixelPos;
import org.esa.snap.watermask.operator.PNGSourceImage;
import org.esa.snap.watermask.operator.SRTMOpImage;
import org.esa.snap.watermask.operator.WatermaskConstants;
import org.esa.snap.watermask.operator.WatermaskUtils;
import org.esa.snap.watermask.util.ImageDescriptor;
import org.esa.snap.watermask.util.ImageDescriptorBuilder;
import org.geotools.resources.image.ImageUtilities;

public class WatermaskClassifier {
    public static final int WATER_VALUE = 1;
    public static final int INVALID_VALUE = 127;
    public static final int LAND_VALUE = 0;
    static final int GC_TILE_WIDTH = 576;
    static final int GC_TILE_HEIGHT = 491;
    static final int GC_IMAGE_WIDTH = 129600;
    static final int GC_IMAGE_HEIGHT = 10800;
    private final ImageSource imageSource;
    private float[] samplingStepsX;
    private float[] samplingStepsY;
    private final int numSuperSamples;

    public WatermaskClassifier(int resolution) throws IOException {
        this(resolution, 1, 1);
    }

    public WatermaskClassifier(int resolution, int superSamplingX, int superSamplingY) throws IOException {
        if (!WatermaskClassifier.isValidResolution(resolution)) {
            throw new IllegalArgumentException(MessageFormat.format("Resolution needs to be {0}, {1}, or {2}.", 50, 150, 1000));
        }
        this.numSuperSamples = superSamplingX * superSamplingY;
        this.samplingStepsX = WatermaskClassifier.getSuperSamplingSteps(superSamplingX);
        this.samplingStepsY = WatermaskClassifier.getSuperSamplingSteps(superSamplingY);
        if (resolution == 50 || resolution == 150) {
            this.installAuxdata();
            File auxdataDir = WatermaskConstants.LOCAL_AUXDATA_PATH.toFile();
            SRTMOpImage centerImage = this.createCenterImage(resolution, auxdataDir);
            ImageDescriptor northDescriptor = WatermaskClassifier.getNorthDescriptor(auxdataDir);
            PNGSourceImage northImage = this.createBorderImage(northDescriptor);
            this.imageSource = new HighResImageSource((OpImage)centerImage, (OpImage)northImage);
        } else {
            this.imageSource = new LowResImageSource();
        }
    }

    public boolean isWater(float lat, float lon) throws IOException {
        int waterMaskSample = this.getWaterMaskSample(lat, lon);
        return waterMaskSample == 1;
    }

    public int getWaterMaskSample(float lat, float lon) {
        double normLon = (double)lon + 180.0;
        if (normLon >= 360.0) {
            normLon %= 360.0;
        }
        float normLat = Math.abs(lat - 90.0f);
        if (normLon < 0.0 || normLon > 360.0 || (double)normLat < 0.0 || (double)normLat > 180.0) {
            return 127;
        }
        return WatermaskClassifier.getSample(normLat, normLon, this.imageSource.getLatHeight(normLat), this.imageSource.getLonWidth(), this.imageSource.getImage(normLat));
    }

    public byte getWaterMaskFraction(GeoCoding geoCoding, int pixelPosX, int pixelPosY) {
        GeoPos geoPos = new GeoPos();
        PixelPos currentPos = new PixelPos();
        float valueSum = 0.0f;
        int invalidCount = 0;
        for (float samplingStepY : this.samplingStepsY) {
            currentPos.y = (float)pixelPosY + samplingStepY;
            for (float samplingStepX : this.samplingStepsX) {
                currentPos.x = (float)pixelPosX + samplingStepX;
                geoCoding.getGeoPos(currentPos, geoPos);
                int waterMaskSample = this.getWaterMaskSample(geoPos);
                if (waterMaskSample != 127) {
                    valueSum += (float)waterMaskSample;
                    continue;
                }
                ++invalidCount;
            }
        }
        return WatermaskClassifier.computeAverage(valueSum, invalidCount, this.numSuperSamples);
    }

    public static boolean isValidResolution(int resolution) {
        return resolution == 50 || resolution == 150 || resolution == 1000;
    }

    private SRTMOpImage createCenterImage(int resolution, File auxdataDir) throws IOException {
        int tileSize = WatermaskUtils.computeSideLength(resolution);
        int width = tileSize * 360;
        int height = tileSize * 180;
        Properties properties = new Properties();
        properties.setProperty("width", String.valueOf(width));
        properties.setProperty("height", String.valueOf(height));
        properties.setProperty("tileWidth", String.valueOf(tileSize));
        properties.setProperty("tileHeight", String.valueOf(tileSize));
        URL imageProperties = this.getClass().getResource("image.properties");
        properties.load(imageProperties.openStream());
        File zipFile = new File(auxdataDir, resolution + "m.zip");
        return SRTMOpImage.create(properties, zipFile);
    }

    private PNGSourceImage createBorderImage(ImageDescriptor descriptor) throws IOException {
        int width = descriptor.getImageWidth();
        int tileWidth = descriptor.getTileWidth();
        int height = descriptor.getImageHeight();
        int tileHeight = descriptor.getTileHeight();
        Properties properties = new Properties();
        properties.setProperty("width", String.valueOf(width));
        properties.setProperty("height", String.valueOf(height));
        properties.setProperty("tileWidth", String.valueOf(tileWidth));
        properties.setProperty("tileHeight", String.valueOf(tileHeight));
        URL imageProperties = this.getClass().getResource("image.properties");
        properties.load(imageProperties.openStream());
        File auxdataDir = descriptor.getAuxdataDir();
        String zipFileName = descriptor.getZipFileName();
        File zipFile = new File(auxdataDir, zipFileName);
        return PNGSourceImage.create(properties, zipFile);
    }

    private void installAuxdata() throws IOException {
        String remoteHTTPHost = "http://step.esa.int";
        String remoteHTTPPath = "/auxdata/watermask/images/";
        String baseURL = "http://step.esa.int/auxdata/watermask/images/";
        WatermaskUtils.installRemoteHttpFiles("http://step.esa.int/auxdata/watermask/images/");
    }

    private static int getSample(double lat, double lon, double latHeight, double lonWidth, OpImage image) {
        if (image == null || latHeight == -1.0) {
            return 127;
        }
        double pixelSizeX = lonWidth / (double)image.getWidth();
        double pixelSizeY = latHeight / (double)image.getHeight();
        int x = (int)Math.floor(lon / pixelSizeX);
        int y = (int)Math.floor(lat / pixelSizeY);
        Raster tile = image.getTile(image.XToTileX(x), image.YToTileY(y));
        if (tile == null) {
            return 127;
        }
        return tile.getSample(x, y, 0);
    }

    private static float[] getSuperSamplingSteps(int superSampling) {
        if (superSampling <= 1) {
            return new float[]{0.5f};
        }
        float[] samplingStep = new float[superSampling];
        for (int i = 0; i < samplingStep.length; ++i) {
            samplingStep[i] = ((float)i * 2.0f + 1.0f) / (2.0f * (float)superSampling);
        }
        return samplingStep;
    }

    private static byte computeAverage(float valueSum, int invalidCount, int numSuperSamples) {
        boolean allValuesInvalid;
        boolean bl = allValuesInvalid = invalidCount == numSuperSamples;
        if (allValuesInvalid) {
            return 127;
        }
        return (byte)(100.0f * valueSum / (float)numSuperSamples);
    }

    private int getWaterMaskSample(GeoPos geoPos) {
        int waterMaskSample = geoPos.isValid() ? this.getWaterMaskSample((float)geoPos.lat, (float)geoPos.lon) : 127;
        return waterMaskSample;
    }

    private static ImageDescriptor getNorthDescriptor(File auxdataDir) {
        return new ImageDescriptorBuilder().width(129600).height(10800).tileWidth(576).tileHeight(491).auxdataDir(auxdataDir).zipFileName("GC_water_mask.zip").build();
    }

    private static class LowResImageSource
    implements ImageSource {
        OpImage image;

        private LowResImageSource() {
        }

        @Override
        public float getLonWidth() {
            return 360.0f;
        }

        @Override
        public float getLatHeight(float latitude) {
            return 180.0f;
        }

        @Override
        public OpImage getImage(float latitude) {
            if (this.image != null) {
                return this.image;
            }
            try {
                BufferedImage waterImage = ImageIO.read(this.getClass().getResourceAsStream("water.png"));
                this.image = new NullOpImage((RenderedImage)waterImage, ImageUtilities.getImageLayout((RenderedImage)waterImage), null, 1);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            return this.image;
        }
    }

    private static class HighResImageSource
    implements ImageSource {
        private static final float INVALID_LAT_HEIGHT = -1.0f;
        private final OpImage centralImage;
        private final OpImage northImage;

        private HighResImageSource(OpImage centralImage, OpImage northImage) {
            this.centralImage = centralImage;
            this.northImage = northImage;
        }

        @Override
        public float getLonWidth() {
            return 360.0f;
        }

        @Override
        public float getLatHeight(float latitude) {
            if ((double)latitude < 150.0 && (double)latitude > 30.0) {
                return 180.0f;
            }
            if ((double)latitude <= 30.0) {
                return 30.0f;
            }
            return -1.0f;
        }

        @Override
        public OpImage getImage(float latitude) {
            if ((double)latitude < 150.0 && (double)latitude > 30.0) {
                return this.centralImage;
            }
            if ((double)latitude <= 30.0) {
                return this.northImage;
            }
            return null;
        }
    }

    private static interface ImageSource {
        public float getLonWidth();

        public float getLatHeight(float var1);

        public OpImage getImage(float var1);
    }
}

