/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.statistics;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.Logger;
import org.esa.snap.core.datamodel.GeoCoding;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.statistics.BandConfiguration;
import org.esa.snap.statistics.ProductValidator;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

public class ProductValidatorTest {
    private final int oneSecond = 1000;
    private ProductData.UTC _startDate;
    private ProductData.UTC _endDate;
    private List<BandConfiguration> _bandConfigurations;
    private ProductValidator _productValidator;
    private Logger _loggerMock;
    private Product _product;

    @Before
    public void setUp() throws Exception {
        this._loggerMock = (Logger)Mockito.mock(Logger.class);
        this._bandConfigurations = new ArrayList<BandConfiguration>();
        this._startDate = ProductData.UTC.parse((String)"2012-05-21 00:00:00", (String)"yyyy-MM-dd HH:mm:ss");
        this._endDate = ProductData.UTC.parse((String)"2012-11-08 00:00:00", (String)"yyyy-MM-dd HH:mm:ss");
        this._productValidator = new ProductValidator(this._bandConfigurations, this._startDate, this._endDate, this._loggerMock);
        this._product = (Product)Mockito.mock(Product.class);
        Mockito.when((Object)this._product.getSceneGeoCoding()).thenReturn(Mockito.mock(GeoCoding.class));
        Mockito.when((Object)this._product.getStartTime()).thenReturn((Object)this._startDate);
        Mockito.when((Object)this._product.getEndTime()).thenReturn((Object)this._endDate);
    }

    @Test
    public void testIsValid_IfIsEntirelyInTimeRange() {
        boolean valid = this._productValidator.isValid(this._product);
        Assert.assertEquals((Object)true, (Object)valid);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this._loggerMock});
    }

    @Test
    public void testIsInvalid_IfProductDoesNotContainAGeoCoding() {
        Mockito.when((Object)this._product.getSceneGeoCoding()).thenReturn(null);
        Mockito.when((Object)this._product.getName()).thenReturn((Object)"No Geocoding");
        boolean valid = this._productValidator.isValid(this._product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this._loggerMock)).info("Product skipped. The product 'No Geocoding' does not contain a geo coding.");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this._loggerMock});
    }

    @Test
    public void testIsInvalid_IfIsNotEntirelyInTimeRange_beforeTimeRange() {
        Mockito.when((Object)this._product.getStartTime()).thenReturn((Object)this.before(this._startDate));
        Mockito.when((Object)this._product.getEndTime()).thenReturn((Object)this._endDate);
        Mockito.when((Object)this._product.getName()).thenReturn((Object)"OutOfDateRange_before");
        boolean valid = this._productValidator.isValid(this._product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this._loggerMock)).info("Product skipped. The product 'OutOfDateRange_before' is not inside the date range from 21-MAY-2012 00:00:00.000000 to 08-NOV-2012 00:00:00.000000");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this._loggerMock});
    }

    @Test
    public void testIsInvalid_IfIsNotEntirelyInTimeRange_afterTimeRange() {
        Mockito.when((Object)this._product.getStartTime()).thenReturn((Object)this._startDate);
        Mockito.when((Object)this._product.getEndTime()).thenReturn((Object)this.after(this._endDate));
        Mockito.when((Object)this._product.getName()).thenReturn((Object)"OutOfDateRange_after");
        boolean valid = this._productValidator.isValid(this._product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this._loggerMock)).info("Product skipped. The product 'OutOfDateRange_after' is not inside the date range from 21-MAY-2012 00:00:00.000000 to 08-NOV-2012 00:00:00.000000");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this._loggerMock});
    }

    @Test
    public void testProductValidatorThatHasOnlyStartTime() throws Exception {
        this._productValidator = new ProductValidator(this._bandConfigurations, this._startDate, null, this._loggerMock);
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this._startDate, this.after(this._endDate))));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this._startDate, this.before(this._endDate))));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this._startDate, this._endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this._startDate, null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(null, null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(null, this._endDate)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(this.before(this._startDate), null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(this.before(this._startDate), this._endDate)));
    }

    @Test
    public void testProductValidatorThatHasOnlyEndTime() throws Exception {
        this._productValidator = new ProductValidator(this._bandConfigurations, null, this._endDate, this._loggerMock);
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(null, this._endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.after(this._startDate), this._endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.before(this._startDate), this._endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this._startDate, this._endDate)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(null, null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(this._startDate, null)));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(null, this.after(this._endDate))));
        Assert.assertFalse((boolean)this.isValid(this.configureProductTimes(this._startDate, this.after(this._endDate))));
    }

    @Test
    public void testProductValidatorThatHasNoTimes() throws Exception {
        this._productValidator = new ProductValidator(this._bandConfigurations, null, null, this._loggerMock);
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(null, this._endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.after(this._startDate), this._endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this.before(this._startDate), this._endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this._startDate, this._endDate)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(null, null)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this._startDate, null)));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(null, this.after(this._endDate))));
        Assert.assertTrue((boolean)this.isValid(this.configureProductTimes(this._startDate, this.after(this._endDate))));
    }

    private boolean isValid(Product product) {
        return this._productValidator.isValid(product);
    }

    private Product configureProductTimes(ProductData.UTC startDate, ProductData.UTC endDate) {
        Mockito.when((Object)this._product.getStartTime()).thenReturn((Object)startDate);
        Mockito.when((Object)this._product.getEndTime()).thenReturn((Object)endDate);
        return this._product;
    }

    @Test
    public void testIsValid_IfProductHasAllBandsNeededInBandConfigurations() {
        BandConfiguration bandConfiguration = new BandConfiguration();
        bandConfiguration.sourceBandName = "band-1";
        this._bandConfigurations.add(bandConfiguration);
        Mockito.when((Object)this._product.containsBand("band-1")).thenReturn((Object)true);
        boolean valid = this._productValidator.isValid(this._product);
        Assert.assertEquals((Object)true, (Object)valid);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this._loggerMock});
    }

    @Test
    public void testIsInvalid_IfProductDoesNotContainAllBandsNeededInBandConfigurations() {
        BandConfiguration bandConfiguration = new BandConfiguration();
        bandConfiguration.sourceBandName = "band-1";
        this._bandConfigurations.add(bandConfiguration);
        Mockito.when((Object)this._product.containsBand("band-1")).thenReturn((Object)false);
        Mockito.when((Object)this._product.getName()).thenReturn((Object)"InvalidProduct");
        boolean valid = this._productValidator.isValid(this._product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this._loggerMock)).info("Product skipped. The product 'InvalidProduct' does not contain the band 'band-1'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this._loggerMock});
    }

    @Test
    public void testIsInvalid_IfProductCanNotResolveTheExpressionNeededInBandConfigurations() {
        BandConfiguration bandConfiguration = new BandConfiguration();
        bandConfiguration.expression = "band_1 + 4";
        this._bandConfigurations.add(bandConfiguration);
        Mockito.when((Object)this._product.isCompatibleBandArithmeticExpression("band_1 + 4")).thenReturn((Object)false);
        Mockito.when((Object)this._product.getName()).thenReturn((Object)"InvalidProduct");
        boolean valid = this._productValidator.isValid(this._product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this._loggerMock)).info("Product skipped. The product 'InvalidProduct' can not resolve the band arithmetic expression 'band_1 + 4'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this._loggerMock});
    }

    @Test
    public void testIsInvalid_IfProductAlreadyContainsBandWithExpressionName() {
        BandConfiguration bandConfiguration = new BandConfiguration();
        bandConfiguration.expression = "band_1 + 4";
        this._bandConfigurations.add(bandConfiguration);
        Mockito.when((Object)this._product.isCompatibleBandArithmeticExpression("band_1 + 4")).thenReturn((Object)true);
        Mockito.when((Object)this._product.getName()).thenReturn((Object)"InvalidProduct");
        Mockito.when((Object)this._product.containsBand("band_1_+_4")).thenReturn((Object)true);
        boolean valid = this._productValidator.isValid(this._product);
        Assert.assertEquals((Object)false, (Object)valid);
        ((Logger)Mockito.verify((Object)this._loggerMock)).info("Product skipped. The product 'InvalidProduct' already contains a band 'band_1_+_4'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this._loggerMock});
    }

    private ProductData.UTC before(ProductData.UTC date) {
        long time = date.getAsDate().getTime() - 1000L;
        return ProductData.UTC.create((Date)new Date(time), (long)0L);
    }

    private ProductData.UTC after(ProductData.UTC date) {
        long time = date.getAsDate().getTime() + 1000L;
        return ProductData.UTC.create((Date)new Date(time), (long)0L);
    }
}

