/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.opendap.utils;

import com.bc.io.FileDownloader;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Map;
import org.esa.snap.opendap.ui.DownloadProgressBarPM;
import org.esa.snap.opendap.utils.OpendapUtils;
import org.esa.snap.util.StringUtils;
import ucar.ma2.Array;
import ucar.ma2.InvalidRangeException;
import ucar.nc2.Attribute;
import ucar.nc2.Dimension;
import ucar.nc2.FileWriter2;
import ucar.nc2.NetcdfFile;
import ucar.nc2.NetcdfFileWriter;
import ucar.nc2.Variable;
import ucar.nc2.dods.DODSNetcdfFile;
import ucar.nc2.util.EscapeStrings;

public class DAPDownloader {
    private static final int MAX_FILENAME_DISPLAY_LENGTH = 15;
    final Map<String, Boolean> dapUris;
    final List<String> fileURIs;
    private final DownloadContext downloadContext;
    private final DownloadProgressBarPM pm;

    public DAPDownloader(Map<String, Boolean> dapUris, List<String> fileURIs, DownloadContext downloadContext, DownloadProgressBarPM pm) {
        this.dapUris = dapUris;
        this.fileURIs = fileURIs;
        this.downloadContext = downloadContext;
        this.pm = pm;
    }

    public void saveProducts(File targetDir) throws IOException {
        if (targetDir == null || !targetDir.isDirectory()) {
            throw new IOException("No target directory specified.");
        }
        this.downloadFilesWithDapAccess(targetDir);
        this.downloadFilesWithFileAccess(targetDir);
    }

    private void downloadFilesWithDapAccess(File targetDir) throws IOException {
        for (Map.Entry<String, Boolean> entry : this.dapUris.entrySet()) {
            if (this.pm.isCanceled()) break;
            this.downloadDapFile(targetDir, entry.getKey(), entry.getValue());
        }
    }

    private void downloadDapFile(File targetDir, String dapURI, boolean isLargeFile) throws IOException {
        String[] uriComponents = dapURI.split("\\?");
        String constraintExpression = "";
        String fileName = dapURI.substring(uriComponents[0].lastIndexOf("/") + 1);
        if (uriComponents.length > 1) {
            constraintExpression = uriComponents[1];
        }
        this.updateProgressBar(fileName, 0);
        DODSNetcdfFile netcdfFile = new DODSNetcdfFile(dapURI);
        this.writeNetcdfFile(targetDir, fileName, constraintExpression, netcdfFile, isLargeFile);
    }

    void writeNetcdfFile(File targetDir, String fileName, String constraintExpression, final DODSNetcdfFile sourceNetcdfFile, boolean isLargeFile) throws IOException {
        String varName;
        final File file = new File(targetDir, fileName);
        if (file.exists() && !this.downloadContext.mayOverwrite(fileName)) {
            this.downloadContext.notifyFileDownloaded(file);
            this.updateProgressBar(fileName, (int)(file.length() / 1024L));
            return;
        }
        if (StringUtils.isNullOrEmpty((String)constraintExpression)) {
            try {
                Thread thread = new Thread(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            FileWriter2 fileWriter = new FileWriter2((NetcdfFile)sourceNetcdfFile, file.getAbsolutePath(), NetcdfFileWriter.Version.netcdf3, null);
                            fileWriter.write();
                        }
                        catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    }
                });
                thread.start();
                int downloadedBefore = 0;
                while (thread.isAlive()) {
                    int downloaded;
                    int delta;
                    try {
                        Thread.sleep(1000L);
                    }
                    catch (InterruptedException ignore) {
                        // empty catch block
                    }
                    if ((delta = (downloaded = (int)(file.length() / 1024L)) - downloadedBefore) <= 0) continue;
                    this.updateProgressBar(fileName, delta);
                    downloadedBefore = downloaded;
                }
            }
            catch (RuntimeException e) {
                if (e.getCause() instanceof IOException) {
                    throw (IOException)e.getCause();
                }
                throw e;
            }
            if (!this.pm.isCanceled()) {
                this.downloadContext.notifyFileDownloaded(file);
            }
            return;
        }
        List variables = sourceNetcdfFile.getVariables();
        ArrayList<String> variableNames = new ArrayList<String>();
        for (Variable variable : variables) {
            variableNames.add(variable.getFullName());
        }
        List<String> filteredVariables = DAPDownloader.filterVariables(variableNames, constraintExpression);
        List<Dimension> filteredDimensions = DAPDownloader.filterDimensions(filteredVariables, (NetcdfFile)sourceNetcdfFile);
        NetcdfFileWriter targetNetCDF = NetcdfFileWriter.createNew((NetcdfFileWriter.Version)NetcdfFileWriter.Version.netcdf3, (String)file.getAbsolutePath());
        for (Dimension filteredDimension : filteredDimensions) {
            targetNetCDF.addDimension(null, filteredDimension.getFullName(), filteredDimension.getLength(), filteredDimension.isShared(), filteredDimension.isUnlimited(), filteredDimension.isVariableLength());
        }
        for (String filteredVariable : filteredVariables) {
            varName = EscapeStrings.backslashEscape((String)filteredVariable, (String)"();,.\\");
            Variable variable = sourceNetcdfFile.findVariable(varName);
            Variable targetVariable = targetNetCDF.addVariable(null, variable.getFullName(), variable.getDataType(), variable.getDimensions());
            for (Attribute attribute : variable.getAttributes()) {
                targetVariable.addAttribute(attribute);
            }
        }
        for (Attribute attribute : sourceNetcdfFile.getGlobalAttributes()) {
            targetNetCDF.addGroupAttribute(null, attribute);
        }
        targetNetCDF.create();
        for (String filteredVariable : filteredVariables) {
            varName = EscapeStrings.backslashEscape((String)filteredVariable, (String)"();,.\\");
            Variable sourceVariable = sourceNetcdfFile.findVariable(varName);
            String ceForVariable = DAPDownloader.getConstraintExpression(filteredVariable, constraintExpression);
            Array values = sourceNetcdfFile.readWithCE(sourceVariable, ceForVariable);
            int[] origin = DAPDownloader.getOrigin(filteredVariable, constraintExpression, sourceVariable.getDimensions().size());
            try {
                targetNetCDF.write(sourceVariable, origin, values);
            }
            catch (InvalidRangeException e) {
                throw new IOException(MessageFormat.format("Unable to download variable ''{0}'' into file ''{1}''.", filteredVariable, fileName), e);
            }
        }
        targetNetCDF.close();
        this.downloadContext.notifyFileDownloaded(file);
    }

    private void updateProgressBar(String fileName, int work) {
        this.pm.worked(work);
        StringBuilder preMessageBuilder = new StringBuilder(fileName);
        int currentWork = this.pm.getCurrentWork();
        preMessageBuilder.append(" (").append(this.downloadContext.getAllDownloadedFilesCount() + 1).append("/").append(this.downloadContext.getAllFilesCount()).append(")");
        if (currentWork != 0) {
            long currentTime = new GregorianCalendar().getTimeInMillis();
            long durationInMillis = currentTime - this.pm.getStartTime();
            double downloadSpeed = DAPDownloader.getDownloadSpeed(durationInMillis, currentWork);
            char sizeIdentifier = downloadSpeed < 1000.0 ? (char)'k' : 'M';
            downloadSpeed = downloadSpeed < 1000.0 ? downloadSpeed : downloadSpeed / 1024.0;
            String speedString = OpendapUtils.format(downloadSpeed);
            preMessageBuilder.append(" @ ").append(speedString).append(" ").append(sizeIdentifier).append("B/s");
        }
        int totalWork = this.pm.getTotalWork();
        double percentage = (double)currentWork / (double)totalWork * 100.0;
        String workDone = OpendapUtils.format((double)currentWork / 1024.0);
        String totalWorkString = OpendapUtils.format((double)totalWork / 1024.0);
        this.pm.setPostMessage(workDone + " MB/" + totalWorkString + " MB (" + OpendapUtils.format(percentage) + "%)");
        String preMessageString = preMessageBuilder.toString();
        this.pm.setTooltip("Downloading " + preMessageString);
        String shortenedFilename = fileName.substring(0, Math.min(fileName.length(), 15));
        this.pm.setPreMessage("Downloading " + preMessageString.replace(fileName, shortenedFilename + "..."));
    }

    static double getDownloadSpeed(long durationInMillis, int kilobyteCount) {
        return (double)kilobyteCount / ((double)durationInMillis / 1000.0);
    }

    static String getConstraintExpression(String sourceVariable, String constraintExpression) {
        String[] constraintExpressions;
        for (String expression : constraintExpressions = constraintExpression.split(",")) {
            if (!expression.startsWith(sourceVariable + "[")) continue;
            return expression;
        }
        throw new IllegalArgumentException(MessageFormat.format("Source variable ''{0}'' must be included in expression ''{1}''.", sourceVariable, constraintExpression));
    }

    static int[] getOrigin(String variableName, String constraintExpression, int dimensionCount) {
        String[] variableConstraints;
        int[] origin = new int[dimensionCount];
        Arrays.fill(origin, 0);
        if (StringUtils.isNullOrEmpty((String)constraintExpression)) {
            return origin;
        }
        for (String variableConstraint : variableConstraints = constraintExpression.split(",")) {
            if (!variableConstraint.contains(variableName)) continue;
            if (!variableConstraint.contains("[")) {
                return origin;
            }
            String[] rangeConstraints = (variableConstraint = variableConstraint.replace("]", "")).split("\\[");
            if (rangeConstraints.length - 1 > dimensionCount) {
                throw new IllegalArgumentException(MessageFormat.format("Illegal expression: ''{0}'' for variable ''{1}''.", constraintExpression, variableName));
            }
            for (int i = 1; i < rangeConstraints.length; ++i) {
                String rangeConstraint = rangeConstraints[i];
                String[] rangeComponents = rangeConstraint.split(":");
                origin[i - 1] = Integer.parseInt(rangeComponents[0]);
            }
        }
        return origin;
    }

    static List<String> filterVariables(List<String> variableNames, String constraintExpression) {
        ArrayList<String> filteredVariables = new ArrayList<String>();
        List<String> constrainedVariableNames = DAPDownloader.getVariableNames(constraintExpression);
        if (constrainedVariableNames == null || constrainedVariableNames.isEmpty()) {
            return variableNames;
        }
        for (String variableName : variableNames) {
            if (!constrainedVariableNames.contains(variableName)) continue;
            filteredVariables.add(variableName);
        }
        if (filteredVariables.isEmpty()) {
            return variableNames;
        }
        return filteredVariables;
    }

    static List<Dimension> filterDimensions(List<String> variables, NetcdfFile netcdfFile) {
        ArrayList<Dimension> filteredDimensions = new ArrayList<Dimension>();
        for (String variableName : variables) {
            Variable variable = netcdfFile.findVariable(variableName);
            for (Dimension dimension : variable.getDimensions()) {
                if (filteredDimensions.contains(dimension)) continue;
                filteredDimensions.add(dimension);
            }
        }
        return filteredDimensions;
    }

    static List<String> getVariableNames(String constraintExpression) {
        String[] constraints;
        if (StringUtils.isNullOrEmpty((String)constraintExpression)) {
            return null;
        }
        ArrayList<String> variableNames = new ArrayList<String>();
        for (String constraint : constraints = constraintExpression.split(",")) {
            if (constraint.contains("[")) {
                variableNames.add(constraint.substring(0, constraint.indexOf("[")));
                continue;
            }
            variableNames.add(constraint);
        }
        return variableNames;
    }

    private void downloadFilesWithFileAccess(File targetDir) throws IOException {
        for (String fileURI : this.fileURIs) {
            if (this.pm.isCanceled()) break;
            try {
                this.downloadFile(targetDir, fileURI);
            }
            catch (Exception e) {
                throw new IOException("Unable to download file '" + fileURI + "'.", e);
            }
        }
    }

    void downloadFile(File targetDir, String fileURI) throws URISyntaxException, IOException {
        URL fileUrl = new URI(fileURI).toURL();
        this.updateProgressBar(fileUrl.getFile(), 0);
        File file = FileDownloader.downloadFile((URL)fileUrl, (File)targetDir, null);
        this.downloadContext.notifyFileDownloaded(file);
    }

    public static interface DownloadContext {
        public int getAllFilesCount();

        public int getAllDownloadedFilesCount();

        public void notifyFileDownloaded(File var1);

        public boolean mayOverwrite(String var1);
    }
}

