/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.core.gpf.common.reproject;

import com.bc.ceres.glevel.MultiLevelImage;
import com.bc.ceres.glevel.MultiLevelModel;
import com.bc.ceres.glevel.MultiLevelSource;
import com.bc.ceres.glevel.support.AbstractMultiLevelSource;
import com.bc.ceres.glevel.support.DefaultMultiLevelImage;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.HashMap;
import javax.media.jai.ImageLayout;
import javax.media.jai.Interpolation;
import javax.media.jai.JAI;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.ColorPaletteDef;
import org.esa.snap.core.datamodel.CrsGeoCoding;
import org.esa.snap.core.datamodel.FlagCoding;
import org.esa.snap.core.datamodel.GeoCoding;
import org.esa.snap.core.datamodel.GeoPos;
import org.esa.snap.core.datamodel.ImageGeometry;
import org.esa.snap.core.datamodel.ImageInfo;
import org.esa.snap.core.datamodel.IndexCoding;
import org.esa.snap.core.datamodel.MetadataElement;
import org.esa.snap.core.datamodel.PixelPos;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.core.datamodel.ProductNodeGroup;
import org.esa.snap.core.datamodel.RasterDataNode;
import org.esa.snap.core.datamodel.SampleCoding;
import org.esa.snap.core.dataop.dem.ElevationModel;
import org.esa.snap.core.dataop.dem.ElevationModelDescriptor;
import org.esa.snap.core.dataop.dem.ElevationModelRegistry;
import org.esa.snap.core.dataop.dem.Orthorectifier;
import org.esa.snap.core.dataop.dem.Orthorectifier2;
import org.esa.snap.core.dataop.resamp.Resampling;
import org.esa.snap.core.gpf.Operator;
import org.esa.snap.core.gpf.OperatorException;
import org.esa.snap.core.gpf.OperatorSpi;
import org.esa.snap.core.gpf.annotations.OperatorMetadata;
import org.esa.snap.core.gpf.annotations.Parameter;
import org.esa.snap.core.gpf.annotations.SourceProduct;
import org.esa.snap.core.gpf.annotations.TargetProduct;
import org.esa.snap.core.gpf.common.reproject.Log10OpImage;
import org.esa.snap.core.gpf.common.reproject.ReplaceNaNOpImage;
import org.esa.snap.core.gpf.common.reproject.Reproject;
import org.esa.snap.core.image.ImageManager;
import org.esa.snap.core.image.ResolutionLevel;
import org.esa.snap.core.util.Debug;
import org.esa.snap.core.util.ProductUtils;
import org.esa.snap.core.util.io.FileUtils;
import org.geotools.factory.Hints;
import org.geotools.referencing.CRS;
import org.opengis.referencing.FactoryException;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.cs.AxisDirection;
import org.opengis.referencing.operation.TransformException;

@OperatorMetadata(alias="Reproject", category="Raster/Geometric", version="1.0", authors="Marco Z\u00fchlke, Marco Peters, Ralf Quast, Norman Fomferra", copyright="(c) 2009 by Brockmann Consult", description="Reprojection of a source product to a target Coordinate Reference System.", internal=false)
public class ReprojectionOp
extends Operator {
    @SourceProduct(alias="source", description="The product which will be reprojected.")
    private Product sourceProduct;
    @SourceProduct(alias="collocateWith", optional=true, label="Collocation product", description="The source product will be collocated with this product.")
    private Product collocationProduct;
    @TargetProduct
    private Product targetProduct;
    @Parameter(description="A file which contains the target Coordinate Reference System in WKT format.")
    private File wktFile;
    @Parameter(description="A text specifying the target Coordinate Reference System, either in WKT or as an authority code. For appropriate EPSG authority codes see (www.epsg-registry.org). AUTO authority can be used with code 42001 (UTM), and 42002 (Transverse Mercator) where the scene center is used as reference. Examples: EPSG:4326, AUTO:42001")
    private String crs;
    @Parameter(alias="resampling", label="Resampling Method", description="The method used for resampling of floating-point raster data.", valueSet={"Nearest", "Bilinear", "Bicubic"}, defaultValue="Nearest")
    private String resamplingName;
    @Parameter(description="The X-position of the reference pixel.")
    private Double referencePixelX;
    @Parameter(description="The Y-position of the reference pixel.")
    private Double referencePixelY;
    @Parameter(description="The easting of the reference pixel.")
    private Double easting;
    @Parameter(description="The northing of the reference pixel.")
    private Double northing;
    @Parameter(description="The orientation of the output product (in degree).", defaultValue="0", interval="[-360,360]")
    private Double orientation;
    @Parameter(description="The pixel size in X direction given in CRS units.")
    private Double pixelSizeX;
    @Parameter(description="The pixel size in Y direction given in CRS units.")
    private Double pixelSizeY;
    @Parameter(description="The width of the target product.")
    private Integer width;
    @Parameter(description="The height of the target product.")
    private Integer height;
    @Parameter(description="The tile size in X direction.")
    private Integer tileSizeX;
    @Parameter(description="The tile size in Y direction.")
    private Integer tileSizeY;
    @Parameter(description="Whether the source product should be orthorectified. (Not applicable to all products)", defaultValue="false")
    private boolean orthorectify;
    @Parameter(description="The name of the elevation model for the orthorectification. If not given tie-point data is used.")
    private String elevationModelName;
    @Parameter(description="The value used to indicate no-data.")
    private Double noDataValue;
    @Parameter(description="Whether tie-point grids should be included in the output product.", defaultValue="true")
    private boolean includeTiePointGrids;
    @Parameter(description="Whether to add delta longitude and latitude bands.", defaultValue="false")
    private boolean addDeltaBands;
    private ReprojectionSettingsProvider reprojectionSettingsProvider;
    private ElevationModel elevationModel;

    @Override
    public void initialize() throws OperatorException {
        Dimension tileSize;
        if (this.sourceProduct.isMultiSizeProduct()) {
            throw this.createMultiSizeException(this.sourceProduct);
        }
        this.validateCrsParameters();
        this.validateResamplingParameter();
        this.validateReferencingParameters();
        this.validateTargetGridParameters();
        this.validateSARProduct();
        GeoPos centerGeoPos = this.getCenterGeoPos(this.sourceProduct.getSceneGeoCoding(), this.sourceProduct.getSceneRasterWidth(), this.sourceProduct.getSceneRasterHeight());
        CoordinateReferenceSystem targetCrs = this.createTargetCRS(centerGeoPos);
        ImageGeometry targetImageGeometry = this.createImageGeometry(targetCrs);
        Rectangle targetRect = targetImageGeometry.getImageRect();
        this.targetProduct = new Product("projected_" + this.sourceProduct.getName(), this.sourceProduct.getProductType(), targetRect.width, targetRect.height);
        this.targetProduct.setDescription(this.sourceProduct.getDescription());
        if (this.tileSizeX != null && this.tileSizeY != null) {
            tileSize = new Dimension(this.tileSizeX, this.tileSizeY);
        } else {
            tileSize = ImageManager.getPreferredTileSize((Product)this.targetProduct);
            Dimension sourceProductPreferredTileSize = this.sourceProduct.getPreferredTileSize();
            if (sourceProductPreferredTileSize != null && sourceProductPreferredTileSize.width == this.sourceProduct.getSceneRasterWidth()) {
                tileSize.width = this.targetProduct.getSceneRasterWidth();
                tileSize.height = Math.min(sourceProductPreferredTileSize.height, this.targetProduct.getSceneRasterHeight());
            }
        }
        this.targetProduct.setPreferredTileSize(tileSize);
        if (this.orthorectify) {
            this.elevationModel = this.createElevationModel();
        }
        ProductUtils.copyMetadata((Product)this.sourceProduct, (Product)this.targetProduct);
        ProductUtils.copyFlagCodings((Product)this.sourceProduct, (Product)this.targetProduct);
        this.copyIndexCoding();
        try {
            this.targetProduct.setSceneGeoCoding((GeoCoding)new CrsGeoCoding(targetImageGeometry.getMapCrs(), targetRect, targetImageGeometry.getImage2MapTransform()));
        }
        catch (Exception e) {
            throw new OperatorException(e);
        }
        ProductData.UTC meanTime = this.getSourceMeanTime();
        this.targetProduct.setStartTime(meanTime);
        this.targetProduct.setEndTime(meanTime);
        this.reprojectionSettingsProvider = new ReprojectionSettingsProvider(targetImageGeometry);
        this.reprojectRasterDataNodes((RasterDataNode[])this.sourceProduct.getBands());
        if (this.includeTiePointGrids) {
            this.reprojectRasterDataNodes((RasterDataNode[])this.sourceProduct.getTiePointGrids());
        }
        ProductUtils.copyVectorData((Product)this.sourceProduct, (Product)this.targetProduct);
        ProductUtils.copyMasks((Product)this.sourceProduct, (Product)this.targetProduct);
        ProductUtils.copyOverlayMasks((Product)this.sourceProduct, (Product)this.targetProduct);
        this.targetProduct.setAutoGrouping(this.sourceProduct.getAutoGrouping());
        if (this.addDeltaBands) {
            this.addDeltaBands();
        }
    }

    @Override
    public void dispose() {
        if (this.elevationModel != null) {
            this.elevationModel.dispose();
        }
        super.dispose();
    }

    private ElevationModel createElevationModel() throws OperatorException {
        if (this.elevationModelName != null) {
            ElevationModelDescriptor demDescriptor = ElevationModelRegistry.getInstance().getDescriptor(this.elevationModelName);
            return demDescriptor.createDem(Resampling.BILINEAR_INTERPOLATION);
        }
        return null;
    }

    private GeoCoding getSourceGeoCoding(RasterDataNode sourceBand) {
        if (this.orthorectify && sourceBand.canBeOrthorectified()) {
            return this.createOrthorectifier(sourceBand);
        }
        return sourceBand.getGeoCoding();
    }

    private Orthorectifier createOrthorectifier(RasterDataNode sourceBand) {
        return new Orthorectifier2(sourceBand.getRasterWidth(), sourceBand.getRasterHeight(), sourceBand.getPointing(), this.elevationModel, 25);
    }

    private void reprojectRasterDataNodes(RasterDataNode[] rasterDataNodes) {
        for (RasterDataNode raster : rasterDataNodes) {
            this.reprojectSourceRaster(raster);
        }
    }

    private void reprojectSourceRaster(RasterDataNode sourceRaster) {
        Reproject reprojection;
        MultiLevelImage sourceImage;
        int targetDataType;
        ReprojectionSettings reprojectionSettings = this.reprojectionSettingsProvider.getReprojectionSettings(sourceRaster);
        if (sourceRaster.isScalingApplied()) {
            targetDataType = sourceRaster.getGeophysicalDataType();
            sourceImage = sourceRaster.getGeophysicalImage();
        } else {
            targetDataType = sourceRaster.getDataType();
            sourceImage = sourceRaster.getSourceImage();
        }
        Number targetNoDataValue = this.getTargetNoDataValue(sourceRaster, targetDataType);
        Rectangle imageRect = reprojectionSettings.getImageGeometry().getImageRect();
        Band targetBand = new Band(sourceRaster.getName(), targetDataType, (int)imageRect.getWidth(), (int)imageRect.getHeight());
        this.targetProduct.addBand(targetBand);
        targetBand.setLog10Scaled(sourceRaster.isLog10Scaled());
        targetBand.setNoDataValue(targetNoDataValue.doubleValue());
        targetBand.setNoDataValueUsed(targetBand.getRasterWidth() == this.targetProduct.getSceneRasterWidth() && targetBand.getRasterHeight() == this.targetProduct.getSceneRasterHeight());
        targetBand.setDescription(sourceRaster.getDescription());
        targetBand.setUnit(sourceRaster.getUnit());
        GeoCoding bandGeoCoding = reprojectionSettings.getGeoCoding();
        if (bandGeoCoding != null) {
            targetBand.setGeoCoding(bandGeoCoding);
        }
        GeoCoding sourceGeoCoding = this.getSourceGeoCoding(sourceRaster);
        String exp = sourceRaster.getValidMaskExpression();
        if (exp != null) {
            sourceImage = this.createNoDataReplacedImage(sourceRaster, targetNoDataValue);
        }
        Interpolation resampling = this.getResampling(targetBand);
        MultiLevelModel targetModel = reprojectionSettings.getTargetModel();
        if (targetModel == null) {
            targetModel = targetBand.getMultiLevelModel();
            reprojectionSettings.setTargetModel(targetModel);
        }
        if ((reprojection = reprojectionSettings.getReprojection()) == null) {
            reprojection = new Reproject(targetModel.getLevelCount());
            reprojectionSettings.setReprojection(reprojection);
        }
        MultiLevelImage projectedImage = this.createProjectedImage(sourceGeoCoding, sourceImage, reprojectionSettings.getSourceModel(), targetBand, resampling, targetModel, reprojection);
        if (this.mustReplaceNaN(sourceRaster, targetDataType, targetNoDataValue.doubleValue())) {
            projectedImage = this.createNaNReplacedImage(projectedImage, targetModel, targetNoDataValue.doubleValue());
        }
        if (targetBand.isLog10Scaled()) {
            projectedImage = this.createLog10ScaledImage(projectedImage);
        }
        targetBand.setSourceImage(projectedImage);
        if (sourceRaster instanceof Band) {
            Band sourceBand = (Band)sourceRaster;
            ProductUtils.copySpectralBandProperties((Band)sourceBand, (Band)targetBand);
            FlagCoding sourceFlagCoding = sourceBand.getFlagCoding();
            IndexCoding sourceIndexCoding = sourceBand.getIndexCoding();
            if (sourceFlagCoding != null) {
                String flagCodingName = sourceFlagCoding.getName();
                FlagCoding destFlagCoding = (FlagCoding)this.targetProduct.getFlagCodingGroup().get(flagCodingName);
                targetBand.setSampleCoding((SampleCoding)destFlagCoding);
            } else if (sourceIndexCoding != null) {
                String indexCodingName = sourceIndexCoding.getName();
                IndexCoding destIndexCoding = (IndexCoding)this.targetProduct.getIndexCodingGroup().get(indexCodingName);
                targetBand.setSampleCoding((SampleCoding)destIndexCoding);
            }
        }
    }

    private MultiLevelImage createLog10ScaledImage(final MultiLevelImage projectedImage) {
        return new DefaultMultiLevelImage((MultiLevelSource)new AbstractMultiLevelSource(projectedImage.getModel()){

            public RenderedImage createImage(int level) {
                return new Log10OpImage(projectedImage.getImage(level));
            }
        });
    }

    private boolean mustReplaceNaN(RasterDataNode sourceRaster, int targetDataType, double targetNoDataValue) {
        boolean isFloat = ProductData.isFloatingPointType((int)targetDataType);
        boolean isNoDataGiven = sourceRaster.isNoDataValueUsed() || this.noDataValue != null;
        boolean isNoDataNaN = Double.isNaN(targetNoDataValue);
        return isFloat && isNoDataGiven && !isNoDataNaN;
    }

    private Number getTargetNoDataValue(RasterDataNode sourceRaster, int targetDataType) {
        double targetNoDataValue = Double.NaN;
        if (this.noDataValue != null) {
            targetNoDataValue = this.noDataValue;
        } else if (sourceRaster.isNoDataValueUsed()) {
            targetNoDataValue = sourceRaster.getNoDataValue();
        }
        Number targetNoDataNumber = targetDataType == 10 ? (Number)((byte)targetNoDataValue) : (Number)(targetDataType == 11 || targetDataType == 20 ? (Number)((short)targetNoDataValue) : (Number)(targetDataType == 12 || targetDataType == 21 ? (Number)((int)targetNoDataValue) : (Number)(targetDataType == 30 ? (Number)Float.valueOf((float)targetNoDataValue) : (Number)targetNoDataValue)));
        return targetNoDataNumber;
    }

    private MultiLevelImage createNaNReplacedImage(final MultiLevelImage projectedImage, MultiLevelModel targetModel, final double value) {
        return new DefaultMultiLevelImage((MultiLevelSource)new AbstractMultiLevelSource(targetModel){

            public RenderedImage createImage(int targetLevel) {
                return new ReplaceNaNOpImage(projectedImage.getImage(targetLevel), value);
            }
        });
    }

    private MultiLevelImage createNoDataReplacedImage(RasterDataNode rasterDataNode, Number noData) {
        return ImageManager.createMaskedGeophysicalImage((RasterDataNode)rasterDataNode, (Number)noData);
    }

    private MultiLevelImage createProjectedImage(GeoCoding sourceGeoCoding, final MultiLevelImage sourceImage, final MultiLevelModel sourceModel, final Band targetBand, final Interpolation resampling, final MultiLevelModel targetModel, final Reproject reprojection) {
        final CoordinateReferenceSystem sourceModelCrs = Product.findModelCRS((GeoCoding)sourceGeoCoding);
        final CoordinateReferenceSystem targetModelCrs = Product.findModelCRS((GeoCoding)targetBand.getGeoCoding());
        final AffineTransform sourceImageToMapTransform = Product.findImageToModelTransform((GeoCoding)sourceGeoCoding);
        final AffineTransform targetImageToMapTransform = Product.findImageToModelTransform((GeoCoding)targetBand.getGeoCoding());
        return new DefaultMultiLevelImage((MultiLevelSource)new AbstractMultiLevelSource(targetModel){

            public RenderedImage createImage(int targetLevel) {
                double targetScale = targetModel.getScale(targetLevel);
                int sourceLevel = sourceImage.getModel().getLevel(targetScale);
                RenderedImage leveledSourceImage = sourceImage.getImage(sourceLevel);
                Rectangle sourceBounds = new Rectangle(leveledSourceImage.getMinX(), leveledSourceImage.getMinY(), leveledSourceImage.getWidth(), leveledSourceImage.getHeight());
                AffineTransform i2mSource = sourceModel.getImageToModelTransform(sourceLevel);
                i2mSource.concatenate(sourceModel.getModelToImageTransform(0));
                i2mSource.concatenate(sourceImageToMapTransform);
                ImageGeometry sourceGeometry = new ImageGeometry(sourceBounds, sourceModelCrs, i2mSource);
                ImageLayout imageLayout = ImageManager.createSingleBandedImageLayout((int)ImageManager.getDataBufferType((int)targetBand.getDataType()), (int)targetBand.getRasterWidth(), (int)targetBand.getRasterHeight(), (Dimension)ReprojectionOp.this.targetProduct.getPreferredTileSize(), (ResolutionLevel)ResolutionLevel.create((MultiLevelModel)this.getModel(), (int)targetLevel));
                Rectangle targetBounds = new Rectangle(imageLayout.getMinX(null), imageLayout.getMinY(null), imageLayout.getWidth(null), imageLayout.getHeight(null));
                AffineTransform i2mTarget = this.getModel().getImageToModelTransform(targetLevel);
                i2mTarget.concatenate(this.getModel().getModelToImageTransform(0));
                i2mTarget.concatenate(targetImageToMapTransform);
                ImageGeometry targetGeometry = new ImageGeometry(targetBounds, targetModelCrs, i2mTarget);
                Hints hints = new Hints(JAI.KEY_IMAGE_LAYOUT, (Object)imageLayout);
                hints.put((Object)Hints.LENIENT_DATUM_SHIFT, (Object)Boolean.TRUE);
                Dimension tileSize = ImageManager.getPreferredTileSize((Product)ReprojectionOp.this.targetProduct);
                try {
                    return reprojection.reproject(leveledSourceImage, sourceGeometry, targetGeometry, targetBand.getNoDataValue(), resampling, hints, targetLevel, tileSize);
                }
                catch (FactoryException | TransformException e) {
                    Debug.trace((Throwable)e);
                    throw new RuntimeException(e);
                }
            }
        });
    }

    private int getSourceLevel(MultiLevelModel sourceModel, int targetLevel) {
        int maxSourceLevel = sourceModel.getLevelCount() - 1;
        return maxSourceLevel < targetLevel ? maxSourceLevel : targetLevel;
    }

    private ProductData.UTC getSourceMeanTime() {
        ProductData.UTC startTime = this.sourceProduct.getStartTime();
        ProductData.UTC endTime = this.sourceProduct.getEndTime();
        Object meanTime = startTime != null && endTime != null ? new ProductData.UTC(0.5 * (startTime.getMJD() + endTime.getMJD())) : (startTime != null ? startTime : (endTime != null ? endTime : null));
        return meanTime;
    }

    private void copyIndexCoding() {
        ProductNodeGroup indexCodingGroup = this.sourceProduct.getIndexCodingGroup();
        for (int i = 0; i < indexCodingGroup.getNodeCount(); ++i) {
            IndexCoding sourceIndexCoding = (IndexCoding)indexCodingGroup.get(i);
            ProductUtils.copyIndexCoding((IndexCoding)sourceIndexCoding, (Product)this.targetProduct);
        }
    }

    private GeoPos getCenterGeoPos(GeoCoding geoCoding, int width, int height) {
        PixelPos centerPixelPos = new PixelPos(0.5 * (double)width + 0.5, 0.5 * (double)height + 0.5);
        return geoCoding.getGeoPos(centerPixelPos, null);
    }

    private CoordinateReferenceSystem createTargetCRS(GeoPos centerGeoPos) throws OperatorException {
        try {
            if (this.wktFile != null) {
                return CRS.parseWKT((String)FileUtils.readText((File)this.wktFile));
            }
            if (this.crs != null) {
                try {
                    return CRS.parseWKT((String)this.crs);
                }
                catch (FactoryException ignored) {
                    if (this.crs.matches("[0-9]*")) {
                        this.crs = "EPSG:" + this.crs;
                    }
                    if (this.crs.matches("AUTO:[0-9]*")) {
                        this.crs = String.format("%s,%s,%s", this.crs, centerGeoPos.lon, centerGeoPos.lat);
                    }
                    return CRS.decode((String)this.crs, (boolean)true);
                }
            }
            if (this.collocationProduct != null && this.collocationProduct.getSceneGeoCoding() != null) {
                return this.collocationProduct.getSceneGeoCoding().getMapCRS();
            }
        }
        catch (IOException | FactoryException e) {
            throw new OperatorException(String.format("Target CRS could not be created: %s", e.getMessage()), e);
        }
        throw new OperatorException("Target CRS could not be created.");
    }

    protected void validateCrsParameters() {
        String msgPattern = "Invalid target CRS specification.\nSpecify {0} one of the ''wktFile'', ''crs'' or ''collocationProduct'' parameters.";
        if (this.wktFile == null && this.crs == null && this.collocationProduct == null) {
            throw new OperatorException(MessageFormat.format("Invalid target CRS specification.\nSpecify {0} one of the ''wktFile'', ''crs'' or ''collocationProduct'' parameters.", "at least"));
        }
        boolean crsDefined = false;
        String exceptionMsg = MessageFormat.format("Invalid target CRS specification.\nSpecify {0} one of the ''wktFile'', ''crs'' or ''collocationProduct'' parameters.", "only");
        if (this.wktFile != null) {
            crsDefined = true;
        }
        if (this.crs != null) {
            if (crsDefined) {
                throw new OperatorException(exceptionMsg);
            }
            crsDefined = true;
        }
        if (this.collocationProduct != null && crsDefined) {
            throw new OperatorException(exceptionMsg);
        }
    }

    private Interpolation getResampling(Band band) {
        int resampleType = this.getResampleType();
        if (!ProductData.isFloatingPointType((int)band.getDataType())) {
            resampleType = 0;
        }
        return Interpolation.getInstance((int)resampleType);
    }

    private int getResampleType() {
        int resamplingType = "Nearest".equalsIgnoreCase(this.resamplingName) ? 0 : ("Bilinear".equalsIgnoreCase(this.resamplingName) ? 1 : ("Bicubic".equalsIgnoreCase(this.resamplingName) ? 2 : -1));
        return resamplingType;
    }

    void validateResamplingParameter() {
        if (this.getResampleType() == -1) {
            throw new OperatorException("Invalid resampling method: " + this.resamplingName);
        }
    }

    void validateReferencingParameters() {
        if (!(this.referencePixelX == null && this.referencePixelY == null && this.easting == null && this.northing == null || this.referencePixelX != null && this.referencePixelY != null && this.easting != null && this.northing != null)) {
            throw new OperatorException("Invalid referencing parameters: \n'referencePixelX', 'referencePixelY', 'easting' and 'northing' have to be specified either all or not at all.");
        }
    }

    void validateTargetGridParameters() {
        if (this.pixelSizeX != null && this.pixelSizeY == null || this.pixelSizeX == null && this.pixelSizeY != null) {
            throw new OperatorException("'pixelSizeX' and 'pixelSizeY' must be specified both or not at all.");
        }
    }

    void validateSARProduct() {
        MetadataElement absRoot;
        MetadataElement root = this.sourceProduct.getMetadataRoot();
        if (root != null && (absRoot = root.getElement("Abstracted_Metadata")) != null) {
            boolean isRadar;
            boolean bl = isRadar = absRoot.getAttributeDouble("radar_frequency", 99999.0) != 99999.0;
            if (isRadar && !(this.sourceProduct.getSceneGeoCoding() instanceof CrsGeoCoding)) {
                throw new OperatorException("SAR products should be terrain corrected or ellipsoid corrected");
            }
        }
    }

    private ImageGeometry createImageGeometry(CoordinateReferenceSystem targetCrs) {
        ImageGeometry imageGeometry;
        if (this.collocationProduct != null) {
            imageGeometry = ImageGeometry.createCollocationTargetGeometry((Product)this.sourceProduct, (Product)this.collocationProduct);
        } else {
            imageGeometry = ImageGeometry.createTargetGeometry((Product)this.sourceProduct, (CoordinateReferenceSystem)targetCrs, (Double)this.pixelSizeX, (Double)this.pixelSizeY, (Integer)this.width, (Integer)this.height, (Double)this.orientation, (Double)this.easting, (Double)this.northing, (Double)this.referencePixelX, (Double)this.referencePixelY);
            AxisDirection targetAxisDirection = targetCrs.getCoordinateSystem().getAxis(1).getDirection();
            if (!AxisDirection.DISPLAY_DOWN.equals((Object)targetAxisDirection)) {
                imageGeometry.changeYAxisDirection();
            }
        }
        return imageGeometry;
    }

    private void addDeltaBands() {
        Band deltaLonBand = this.targetProduct.addBand("delta_lon_angular", "longitude - LON");
        deltaLonBand.setUnit("deg");
        deltaLonBand.setDescription("Delta between old longitude and new longitude in degree");
        deltaLonBand.setNoDataValueUsed(true);
        deltaLonBand.setNoDataValue(this.noDataValue == null ? Double.NaN : this.noDataValue);
        deltaLonBand.setImageInfo(this.createDeltaBandImageInfo(-0.015, 0.015));
        Band deltaLatBand = this.targetProduct.addBand("delta_lat_angular", "latitude - LAT");
        deltaLatBand.setUnit("deg");
        deltaLatBand.setDescription("Delta between old latitude and new latitude in degree");
        deltaLatBand.setNoDataValueUsed(true);
        deltaLatBand.setNoDataValue(this.noDataValue == null ? Double.NaN : this.noDataValue);
        deltaLatBand.setImageInfo(this.createDeltaBandImageInfo(-0.01, 0.01));
        Band deltaLonMetBand = this.targetProduct.addBand("delta_lon_metric", "cos(rad(LAT)) * 6378137 * rad(longitude - LON)");
        deltaLonMetBand.setUnit("m");
        deltaLonMetBand.setDescription("Delta between old longitude and new longitude in meters");
        deltaLonMetBand.setNoDataValueUsed(true);
        deltaLonMetBand.setNoDataValue(this.noDataValue == null ? Double.NaN : this.noDataValue);
        deltaLonMetBand.setImageInfo(this.createDeltaBandImageInfo(-1500.0, 1500.0));
        Band deltaLatMetBand = this.targetProduct.addBand("delta_lat_metric", "6378137 * rad(latitude - LAT)");
        deltaLatMetBand.setUnit("m");
        deltaLatMetBand.setDescription("Delta between old latitude and new latitude in meters");
        deltaLatMetBand.setNoDataValueUsed(true);
        deltaLatMetBand.setNoDataValue(this.noDataValue == null ? Double.NaN : this.noDataValue);
        deltaLatMetBand.setImageInfo(this.createDeltaBandImageInfo(-1000.0, 1000.0));
    }

    private ImageInfo createDeltaBandImageInfo(double p1, double p2) {
        return new ImageInfo(new ColorPaletteDef(new ColorPaletteDef.Point[]{new ColorPaletteDef.Point(p1, new Color(255, 0, 0)), new ColorPaletteDef.Point((p1 + p2) / 2.0, new Color(255, 255, 255)), new ColorPaletteDef.Point(p2, new Color(0, 0, 127))}));
    }

    private class ReprojectionSettings {
        private GeoCoding geoCoding;
        private MultiLevelModel sourceModel;
        private ImageGeometry imageGeometry;
        private MultiLevelModel targetModel;
        private Reproject reprojection;

        public void setTargetModel(MultiLevelModel targetModel) {
            this.targetModel = targetModel;
        }

        public void setReprojection(Reproject reprojection) {
            this.reprojection = reprojection;
        }

        ReprojectionSettings(GeoCoding geoCoding, MultiLevelModel sourceModel, ImageGeometry imageGeometry) {
            this.geoCoding = geoCoding;
            this.sourceModel = sourceModel;
            this.imageGeometry = imageGeometry;
        }

        public GeoCoding getGeoCoding() {
            return this.geoCoding;
        }

        public MultiLevelModel getSourceModel() {
            return this.sourceModel;
        }

        public ImageGeometry getImageGeometry() {
            return this.imageGeometry;
        }

        public MultiLevelModel getTargetModel() {
            return this.targetModel;
        }

        public Reproject getReprojection() {
            return this.reprojection;
        }
    }

    private class MultiResolutionReprojectionSettingsProvider
    extends ReprojectionSettingsProvider {
        HashMap<String, ReprojectionSettings> reprojectionSettingsMap;

        MultiResolutionReprojectionSettingsProvider() {
            this.reprojectionSettingsMap = new HashMap();
            ProductNodeGroup sourceBands = ReprojectionOp.this.sourceProduct.getBandGroup();
            for (int i = 0; i < sourceBands.getNodeCount(); ++i) {
                this.addReprojectionSettingsIfNecessary((RasterDataNode)sourceBands.get(i));
            }
            if (ReprojectionOp.this.includeTiePointGrids) {
                ProductNodeGroup tiePointGridGroup = ReprojectionOp.this.sourceProduct.getTiePointGridGroup();
                for (int i = 0; i < tiePointGridGroup.getNodeCount(); ++i) {
                    this.addReprojectionSettingsIfNecessary((RasterDataNode)tiePointGridGroup.get(i));
                }
            }
        }

        @Override
        protected ReprojectionSettings getReprojectionSettings(RasterDataNode rasterDataNode) {
            return this.reprojectionSettingsMap.get(this.getKey(rasterDataNode));
        }

        private void addReprojectionSettingsIfNecessary(RasterDataNode rasterDataNode) {
            String key = this.getKey(rasterDataNode);
            if (!this.reprojectionSettingsMap.containsKey(key)) {
                GeoPos centerGeoPos = ReprojectionOp.this.getCenterGeoPos(rasterDataNode.getGeoCoding(), rasterDataNode.getRasterWidth(), rasterDataNode.getRasterHeight());
                CoordinateReferenceSystem targetCrs = ReprojectionOp.this.createTargetCRS(centerGeoPos);
                ImageGeometry targetImageGeometry = ImageGeometry.createTargetGeometry((RasterDataNode)rasterDataNode, (CoordinateReferenceSystem)targetCrs, (Double)ReprojectionOp.this.pixelSizeX, (Double)ReprojectionOp.this.pixelSizeY, (Integer)ReprojectionOp.this.width, (Integer)ReprojectionOp.this.height, (Double)ReprojectionOp.this.orientation, (Double)ReprojectionOp.this.easting, (Double)ReprojectionOp.this.northing, (Double)ReprojectionOp.this.referencePixelX, (Double)ReprojectionOp.this.referencePixelY);
                AxisDirection targetAxisDirection = targetCrs.getCoordinateSystem().getAxis(1).getDirection();
                if (!AxisDirection.DISPLAY_DOWN.equals((Object)targetAxisDirection)) {
                    targetImageGeometry.changeYAxisDirection();
                }
                Rectangle targetRect = targetImageGeometry.getImageRect();
                try {
                    CrsGeoCoding geoCoding = new CrsGeoCoding(targetImageGeometry.getMapCrs(), targetRect, targetImageGeometry.getImage2MapTransform());
                    MultiLevelModel sourceModel = rasterDataNode.getMultiLevelModel();
                    this.reprojectionSettingsMap.put(key, new ReprojectionSettings((GeoCoding)geoCoding, sourceModel, targetImageGeometry));
                }
                catch (FactoryException | TransformException e) {
                    throw new OperatorException(e);
                }
            }
        }

        private String getKey(RasterDataNode rasterDataNode) {
            return rasterDataNode.getGeoCoding().toString() + " " + rasterDataNode.getRasterWidth() + " " + rasterDataNode.getRasterHeight();
        }
    }

    private class DefaultReprojectionSettingsProvider
    extends ReprojectionSettingsProvider {
        ReprojectionSettings defaultReprojectionSettings;

        DefaultReprojectionSettingsProvider(ImageGeometry imageGeometry) {
            Band firstBand = (Band)ReprojectionOp.this.sourceProduct.getBandGroup().get(0);
            MultiLevelModel sourceModel = firstBand.getMultiLevelModel();
            MultiLevelModel targetModel = ReprojectionOp.this.targetProduct.createMultiLevelModel();
            Reproject reprojection = new Reproject(targetModel.getLevelCount());
            this.defaultReprojectionSettings = new ReprojectionSettings(null, sourceModel, imageGeometry);
            this.defaultReprojectionSettings.setTargetModel(targetModel);
            this.defaultReprojectionSettings.setReprojection(reprojection);
        }

        @Override
        protected ReprojectionSettings getReprojectionSettings(RasterDataNode rasterDataNode) {
            return this.defaultReprojectionSettings;
        }
    }

    class ReprojectionSettingsProvider {
        ReprojectionSettingsProvider provider;

        ReprojectionSettingsProvider() {
        }

        ReprojectionSettingsProvider(ImageGeometry targetImageGeometry) {
            this.provider = ProductUtils.areRastersEqualInSize((RasterDataNode[])ReprojectionOp.this.sourceProduct.getBands()) ? new DefaultReprojectionSettingsProvider(targetImageGeometry) : new MultiResolutionReprojectionSettingsProvider();
        }

        protected ReprojectionSettings getReprojectionSettings(RasterDataNode rasterDataNode) {
            return this.provider.getReprojectionSettings(rasterDataNode);
        }
    }

    public static class Spi
    extends OperatorSpi {
        public Spi() {
            super(ReprojectionOp.class);
        }
    }
}

