/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.core.gpf.common;

import com.bc.ceres.core.ProgressMonitor;
import java.util.ArrayList;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.core.datamodel.RasterDataNode;
import org.esa.snap.core.datamodel.Stx;
import org.esa.snap.core.datamodel.VirtualBand;
import org.esa.snap.core.gpf.Operator;
import org.esa.snap.core.gpf.OperatorException;
import org.esa.snap.core.gpf.OperatorSpi;
import org.esa.snap.core.gpf.Tile;
import org.esa.snap.core.gpf.annotations.OperatorMetadata;
import org.esa.snap.core.gpf.annotations.Parameter;
import org.esa.snap.core.gpf.annotations.SourceProduct;
import org.esa.snap.core.gpf.annotations.TargetProduct;
import org.esa.snap.core.util.ProductUtils;
import org.esa.snap.core.util.math.Histogram;
import org.esa.snap.core.util.math.Range;

@OperatorMetadata(alias="Convert-Datatype", category="Raster/Data Conversion", authors="Jun Lu, Luis Veci", copyright="Copyright (C) 2015 by Array Systems Computing Inc.", description="Convert product data type")
public class ConvertDataTypeOp
extends Operator {
    @SourceProduct(alias="source")
    private Product sourceProduct;
    @TargetProduct
    private Product targetProduct;
    @Parameter(description="The list of source bands.", alias="sourceBands", rasterDataNodeType=Band.class, label="Source Bands")
    private String[] sourceBandNames;
    @Parameter(valueSet={"int8", "int16", "int32", "uint8", "uint16", "uint32", "float32", "float64"}, defaultValue="uint8", label="Target Data Type")
    private String targetDataType = "uint8";
    private int dataType = 20;
    @Parameter(valueSet={"Truncate", "Linear (slope and intercept)", "Linear (between 95% clipped histogram)", "Linear (peak clipped histogram)", "Logarithmic"}, defaultValue="Linear (between 95% clipped histogram)", label="Scaling")
    private String targetScalingStr = "Linear (between 95% clipped histogram)";
    public static final String SCALING_TRUNCATE = "Truncate";
    public static final String SCALING_LINEAR = "Linear (slope and intercept)";
    public static final String SCALING_LINEAR_CLIPPED = "Linear (between 95% clipped histogram)";
    public static final String SCALING_LINEAR_PEAK_CLIPPED = "Linear (peak clipped histogram)";
    public static final String SCALING_LOGARITHMIC = "Logarithmic";
    private ScalingType targetScaling = ScalingType.LINEAR_CLIPPED;

    @Override
    public void initialize() throws OperatorException {
        if (this.sourceProduct.isMultiSizeProduct()) {
            throw this.createMultiSizeException(this.sourceProduct);
        }
        try {
            this.targetProduct = new Product(this.sourceProduct.getName(), this.sourceProduct.getProductType(), this.sourceProduct.getSceneRasterWidth(), this.sourceProduct.getSceneRasterHeight());
            ProductUtils.copyProductNodes((Product)this.sourceProduct, (Product)this.targetProduct);
            this.dataType = ProductData.getType((String)this.targetDataType);
            this.targetScaling = ConvertDataTypeOp.getScaling(this.targetScalingStr);
            this.addSelectedBands();
        }
        catch (Throwable e) {
            throw new OperatorException(e);
        }
    }

    private static ScalingType getScaling(String scalingStr) {
        switch (scalingStr) {
            case "Linear (slope and intercept)": {
                return ScalingType.LINEAR;
            }
            case "Linear (between 95% clipped histogram)": {
                return ScalingType.LINEAR_CLIPPED;
            }
            case "Linear (peak clipped histogram)": {
                return ScalingType.LINEAR_PEAK_CLIPPED;
            }
            case "Logarithmic": {
                return ScalingType.LOGARITHMIC;
            }
            case "Truncate": {
                return ScalingType.TRUNC;
            }
        }
        return ScalingType.NONE;
    }

    public static Band[] getSourceBands(Product sourceProduct, String[] sourceBandNames, boolean includeVirtualBands) throws OperatorException {
        if (sourceBandNames == null || sourceBandNames.length == 0) {
            Band[] bands = sourceProduct.getBands();
            ArrayList<String> bandNameList = new ArrayList<String>(sourceProduct.getNumBands());
            Band[] bandArray = bands;
            int n = bandArray.length;
            for (int i = 0; i < n; ++i) {
                Band band = bandArray[i];
                if (band instanceof VirtualBand && !includeVirtualBands) continue;
                bandNameList.add(band.getName());
            }
            sourceBandNames = bandNameList.toArray(new String[bandNameList.size()]);
        }
        ArrayList<Band> sourceBandList = new ArrayList<Band>(sourceBandNames.length);
        for (String sourceBandName : sourceBandNames) {
            Band sourceBand = sourceProduct.getBand(sourceBandName);
            if (sourceBand == null) continue;
            sourceBandList.add(sourceBand);
        }
        return sourceBandList.toArray(new Band[sourceBandList.size()]);
    }

    private void addSelectedBands() {
        Band[] sourceBands;
        for (Band srcBand : sourceBands = ConvertDataTypeOp.getSourceBands(this.sourceProduct, this.sourceBandNames, false)) {
            Band targetBand = new Band(srcBand.getName(), this.dataType, this.sourceProduct.getSceneRasterWidth(), this.sourceProduct.getSceneRasterHeight());
            targetBand.setUnit(srcBand.getUnit());
            targetBand.setDescription(srcBand.getDescription());
            this.targetProduct.addBand(targetBand);
        }
    }

    @Override
    public void computeTile(Band targetBand, Tile targetTile, ProgressMonitor pm) throws OperatorException {
        try {
            Histogram histogram;
            Band sourceBand = this.sourceProduct.getBand(targetBand.getName());
            Tile srcTile = this.getSourceTile((RasterDataNode)sourceBand, targetTile.getRectangle());
            Stx stx = sourceBand.getStx();
            double origMin = stx.getMinimum();
            double origMax = stx.getMaximum();
            ScalingType scaling = ConvertDataTypeOp.verifyScaling(this.targetScaling, this.dataType);
            double newMin = ConvertDataTypeOp.getMin(this.dataType);
            double newMax = ConvertDataTypeOp.getMax(this.dataType);
            double newRange = newMax - newMin;
            if (origMax <= newMax && origMin >= newMin && sourceBand.getDataType() < 30) {
                scaling = ScalingType.NONE;
            }
            ProductData srcData = srcTile.getRawSamples();
            ProductData dstData = targetTile.getRawSamples();
            double srcNoDataValue = sourceBand.getNoDataValue();
            double destNoDataValue = targetBand.getNoDataValue();
            if (scaling == ScalingType.LINEAR_PEAK_CLIPPED) {
                histogram = new Histogram(stx.getHistogramBins(), origMin, origMax);
                int[] bitCounts = histogram.getBinCounts();
                double rightPct = 0.025;
                for (int i = bitCounts.length - 1; i > 0; --i) {
                    if (bitCounts[i] <= 10) continue;
                    rightPct = (double)i / (double)bitCounts.length;
                    break;
                }
                Range autoStretchRange = histogram.findRange(0.025, rightPct);
                origMin = autoStretchRange.getMin();
                origMax = autoStretchRange.getMax();
            } else if (scaling == ScalingType.LINEAR_CLIPPED) {
                histogram = new Histogram(stx.getHistogramBins(), origMin, origMax);
                Range autoStretchRange = histogram.findRangeFor95Percent();
                origMin = autoStretchRange.getMin();
                origMax = autoStretchRange.getMax();
            }
            double origRange = origMax - origMin;
            int numElem = dstData.getNumElems();
            for (int i = 0; i < numElem; ++i) {
                double srcValue = srcData.getElemDoubleAt(i);
                if (srcValue == srcNoDataValue) {
                    dstData.setElemDoubleAt(i, destNoDataValue);
                    continue;
                }
                if (scaling == ScalingType.NONE) {
                    dstData.setElemDoubleAt(i, srcValue);
                    continue;
                }
                if (scaling == ScalingType.TRUNC) {
                    dstData.setElemDoubleAt(i, ConvertDataTypeOp.truncate(srcValue, newMin, newMax));
                    continue;
                }
                if (scaling == ScalingType.LOGARITHMIC) {
                    dstData.setElemDoubleAt(i, ConvertDataTypeOp.logScale(srcValue, origMin, newMin, origRange, newRange));
                    continue;
                }
                if (srcValue > origMax) {
                    srcValue = origMax;
                }
                if (srcValue < origMin) {
                    srcValue = origMin;
                }
                dstData.setElemDoubleAt(i, ConvertDataTypeOp.scale(srcValue, origMin, newMin, origRange, newRange));
            }
            targetTile.setRawSamples(dstData);
        }
        catch (Exception e) {
            throw new OperatorException(e.getMessage());
        }
    }

    private static double getMin(int dataType) {
        switch (dataType) {
            case 10: {
                return -128.0;
            }
            case 11: {
                return -32768.0;
            }
            case 12: {
                return -2.147483648E9;
            }
            case 20: {
                return 0.0;
            }
            case 21: {
                return 0.0;
            }
            case 22: {
                return 0.0;
            }
            case 30: {
                return 1.4E-45f;
            }
        }
        return Double.MIN_VALUE;
    }

    private static double getMax(int dataType) {
        switch (dataType) {
            case 10: {
                return 127.0;
            }
            case 11: {
                return 32767.0;
            }
            case 12: {
                return 2.147483647E9;
            }
            case 20: {
                return 255.0;
            }
            case 21: {
                return 65535.0;
            }
            case 22: {
                return 9.223372036854776E18;
            }
            case 30: {
                return 3.4028234663852886E38;
            }
        }
        return Double.MAX_VALUE;
    }

    private static ScalingType verifyScaling(ScalingType targetScaling, int targetDataType) {
        if (targetDataType == 30 || targetDataType == 31 || targetDataType == 12) {
            return ScalingType.NONE;
        }
        return targetScaling;
    }

    private static double truncate(double origValue, double newMin, double newMax) {
        if (origValue > newMax) {
            return newMax;
        }
        if (origValue < newMin) {
            return newMin;
        }
        return origValue;
    }

    private static double scale(double origValue, double origMin, double newMin, double origRange, double newRange) {
        return (origValue - origMin) / origRange * newRange + newMin;
    }

    private static double logScale(double origValue, double origMin, double newMin, double origRange, double newRange) {
        return 10.0 * Math.log10((origValue - origMin) / origRange * newRange + newMin);
    }

    public void setTargetDataType(String newType) {
        this.targetDataType = newType;
    }

    public void setScaling(String newScaling) {
        this.targetScalingStr = newScaling;
    }

    public static class Spi
    extends OperatorSpi {
        public Spi() {
            super(ConvertDataTypeOp.class);
        }
    }

    public static enum ScalingType {
        NONE,
        TRUNC,
        LINEAR,
        LINEAR_CLIPPED,
        LINEAR_PEAK_CLIPPED,
        LOGARITHMIC;

    }
}

