/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.binning.operator;

import com.bc.ceres.core.Assert;
import com.bc.ceres.core.VirtualDir;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.esa.snap.binning.BinningContext;
import org.esa.snap.binning.SpatialBin;
import org.esa.snap.binning.operator.DeleteDirThread;
import org.esa.snap.binning.operator.SpatialBinCollection;
import org.esa.snap.binning.operator.SpatialBinCollector;
import org.esa.snap.core.util.io.FileUtils;

class FileBackedSpatialBinCollector
implements SpatialBinCollector {
    private static final int DEFAULT_NUM_BINS_PER_FILE = 100000;
    private static final int MAX_NUMBER_OF_CACHE_FILES = 10000;
    private static final String FILE_NAME_PATTERN = "bins-%05d.tmp";
    private final int numBinsPerFile;
    private final List<SpatialBin> binList;
    private final AtomicBoolean consumingCompleted;
    private final File tempDir;
    private int currentFileIndex;
    private long numBinsComsumed;

    FileBackedSpatialBinCollector(long maximumNumberOfBins) throws IOException {
        Assert.argument((maximumNumberOfBins > 0L ? 1 : 0) != 0, (String)"maximumNumberOfBins > 0");
        this.numBinsPerFile = FileBackedSpatialBinCollector.getNumBinsPerFile(maximumNumberOfBins);
        this.tempDir = VirtualDir.createUniqueTempDir();
        Runtime.getRuntime().addShutdownHook(new DeleteDirThread(this.tempDir));
        this.binList = new ArrayList<SpatialBin>();
        this.consumingCompleted = new AtomicBoolean(false);
        this.currentFileIndex = 0;
        this.numBinsComsumed = 0L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void consumeSpatialBins(BinningContext ignored, List<SpatialBin> spatialBins) throws Exception {
        if (this.consumingCompleted.get()) {
            throw new IllegalStateException("Consuming of bins has already been completed.");
        }
        List<SpatialBin> list = this.binList;
        synchronized (list) {
            for (SpatialBin spatialBin : spatialBins) {
                ++this.numBinsComsumed;
                long spatialBinIndex = spatialBin.getIndex();
                int nextFileIndex = this.calculateNextFileIndex(spatialBinIndex);
                if (nextFileIndex != this.currentFileIndex) {
                    this.writeListToFile(this.currentFileIndex);
                    this.currentFileIndex = nextFileIndex;
                }
                this.binList.add(spatialBin);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void consumingCompleted() throws IOException {
        this.consumingCompleted.set(true);
        List<SpatialBin> list = this.binList;
        synchronized (list) {
            this.writeListToFile(this.currentFileIndex);
        }
    }

    @Override
    public SpatialBinCollection getSpatialBinCollection() throws IOException {
        List<File> cacheFiles = FileBackedSpatialBinCollector.getCacheFiles(this.tempDir);
        return new FileBackedBinCollection(cacheFiles, this.numBinsComsumed);
    }

    @Override
    public void close() {
        FileUtils.deleteTree((File)this.tempDir);
    }

    static void writeToStream(List<SpatialBin> spatialBins, DataOutputStream dos) throws IOException {
        for (SpatialBin spatialBin : spatialBins) {
            dos.writeLong(spatialBin.getIndex());
            spatialBin.write(dos);
        }
    }

    static void readFromStream(DataInputStream dis, SortedMap<Long, List<SpatialBin>> map) throws IOException {
        try {
            while (true) {
                long binIndex;
                ArrayList<SpatialBin> spatialBins;
                if ((spatialBins = (ArrayList<SpatialBin>)map.get(binIndex = dis.readLong())) == null) {
                    spatialBins = new ArrayList<SpatialBin>();
                    map.put(binIndex, spatialBins);
                }
                spatialBins.add(SpatialBin.read(binIndex, dis));
            }
        }
        catch (EOFException eof) {
            return;
        }
    }

    private static int getNumBinsPerFile(long maxBinCount) {
        int numCacheFiles = (int)Math.ceil((float)maxBinCount / 100000.0f);
        numCacheFiles = Math.min(numCacheFiles, 10000);
        int binsPerFile = (int)Math.ceil((float)maxBinCount / (float)numCacheFiles);
        return Math.max(100000, binsPerFile);
    }

    private void writeListToFile(int fileIndex) throws IOException {
        if (!this.binList.isEmpty()) {
            File file = this.getFile(fileIndex);
            this.writeToFile(this.binList, file);
            this.binList.clear();
        }
    }

    private void writeToFile(List<SpatialBin> spatialBins, File file) throws IOException {
        FileOutputStream fos = new FileOutputStream(file, true);
        try (DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(fos, 0x500000));){
            FileBackedSpatialBinCollector.writeToStream(spatialBins, dos);
        }
    }

    private static void readIntoMap(File file, SortedMap<Long, List<SpatialBin>> map) throws IOException {
        FileInputStream fis = new FileInputStream(file);
        try (DataInputStream dis = new DataInputStream(new BufferedInputStream(fis, 0x500000));){
            FileBackedSpatialBinCollector.readFromStream(dis, map);
        }
    }

    private File getFile(int fileIndex) throws IOException {
        return new File(this.tempDir, String.format(FILE_NAME_PATTERN, fileIndex));
    }

    private int calculateNextFileIndex(long binIndex) {
        return (int)(binIndex / (long)this.numBinsPerFile);
    }

    private static List<File> getCacheFiles(File cacheFileDir) {
        Object[] files = cacheFileDir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                String fileName = file.getName();
                return file.isFile() && fileName.startsWith("bins-") && fileName.endsWith(".tmp");
            }
        });
        if (files == null) {
            return Collections.emptyList();
        }
        Arrays.sort(files);
        ArrayList<File> fileList = new ArrayList<File>(files.length);
        Collections.addAll(fileList, files);
        return fileList;
    }

    private static class FileBackedBinIterator
    implements Iterator<List<SpatialBin>> {
        private final Iterator<File> binFiles;
        private Iterator<List<SpatialBin>> binIterator;

        private FileBackedBinIterator(Iterator<File> binFiles) {
            this.binFiles = binFiles;
        }

        @Override
        public boolean hasNext() {
            return this.iteratorHasBins() || this.binFiles.hasNext();
        }

        @Override
        public List<SpatialBin> next() {
            File currentFile;
            if (!this.iteratorHasBins() && (currentFile = this.binFiles.next()).exists()) {
                TreeMap map = new TreeMap();
                try {
                    FileBackedSpatialBinCollector.readIntoMap(currentFile, map);
                }
                catch (IOException e) {
                    throw new IllegalStateException(e.getMessage(), e);
                }
                if (!currentFile.delete()) {
                    currentFile.deleteOnExit();
                }
                this.binIterator = map.values().iterator();
            }
            return this.binIterator.next();
        }

        private boolean iteratorHasBins() {
            return this.binIterator != null && this.binIterator.hasNext();
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    private static class FileBackedBinCollection
    implements SpatialBinCollection {
        private final List<File> cacheFiles;
        private final long size;

        private FileBackedBinCollection(List<File> cacheFiles, long size) {
            this.cacheFiles = cacheFiles;
            this.size = size;
        }

        @Override
        public Iterable<List<SpatialBin>> getBinCollection() {
            return new Iterable<List<SpatialBin>>(){

                @Override
                public Iterator<List<SpatialBin>> iterator() {
                    return new FileBackedBinIterator(cacheFiles.iterator());
                }
            };
        }

        @Override
        public long size() {
            return this.size;
        }

        @Override
        public boolean isEmpty() {
            return false;
        }
    }
}

