/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.core.gpf.descriptor.dependency;

import com.bc.ceres.core.ProgressMonitor;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.BiConsumer;
import java.util.logging.Logger;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import org.esa.snap.core.gpf.descriptor.OSFamily;
import org.esa.snap.core.gpf.descriptor.ToolAdapterOperatorDescriptor;
import org.esa.snap.core.gpf.descriptor.dependency.Bundle;
import org.esa.snap.core.gpf.operators.tooladapter.DefaultOutputConsumer;
import org.esa.snap.core.gpf.operators.tooladapter.ProcessExecutor;
import org.esa.snap.core.gpf.operators.tooladapter.ToolAdapterIO;
import org.esa.snap.core.util.SystemUtils;

public class BundleInstaller
implements AutoCloseable {
    private static final int TIMEOUT = 20000;
    private static final int BUFFER_SIZE = 262144;
    private static Logger logger = Logger.getLogger(BundleInstaller.class.getName());
    private static final Path baseModulePath;
    private static final OSFamily currentOS;
    private final ExecutorService executor;
    private ToolAdapterOperatorDescriptor descriptor;
    private ProgressMonitor progressMonitor;
    private Callable<Void> callback;
    private int taskCount;

    public BundleInstaller(ToolAdapterOperatorDescriptor descriptor) {
        this.descriptor = descriptor;
        this.executor = Executors.newSingleThreadExecutor();
    }

    public static boolean isBundleFileAvailable(Bundle bundle) {
        return bundle != null && (BundleInstaller.getLocalSourcePath(bundle) != null || bundle.getDownloadURL() != null);
    }

    public void setProgressMonitor(ProgressMonitor monitor) {
        this.progressMonitor = monitor;
    }

    public void setCallback(Callable<Void> completionCallback) {
        this.callback = completionCallback;
    }

    public void install(boolean async) {
        Bundle descriptorBundle = this.descriptor.getBundle(currentOS);
        if (descriptorBundle != null) {
            File source;
            Path sourcePath = baseModulePath.resolve(descriptorBundle.getEntryPoint());
            if (!Files.exists(sourcePath, new LinkOption[0]) && (source = descriptorBundle.getSource()) != null) {
                sourcePath = descriptorBundle.getSource().toPath();
            }
            MethodReference<Path, Bundle> firstStep = null;
            if (!descriptorBundle.isLocal()) {
                firstStep = this::download;
                ++this.taskCount;
            }
            switch (descriptorBundle.getBundleType()) {
                case ZIP: {
                    Action action;
                    try {
                        MethodReference<Path, Bundle> secondStep = this::uncompress;
                        firstStep = firstStep != null ? MethodReference.from(firstStep.andThen(secondStep)) : secondStep;
                        ++this.taskCount;
                        action = new Action(sourcePath, descriptorBundle, firstStep);
                        if (this.progressMonitor != null) {
                            this.progressMonitor.beginTask("Installing bundle", 100);
                        }
                        if (async) {
                            this.executor.submit(action);
                            break;
                        }
                        action.call();
                    }
                    catch (Exception e) {
                        logger.warning(e.getMessage());
                    }
                    break;
                }
                case INSTALLER: {
                    Action action;
                    try {
                        MethodReference<Path, Bundle> secondStep = this::install;
                        firstStep = firstStep != null ? MethodReference.from(firstStep.andThen(secondStep)) : secondStep;
                        ++this.taskCount;
                        action = new Action(sourcePath, descriptorBundle, firstStep);
                        if (this.progressMonitor != null) {
                            this.progressMonitor.beginTask("Installing bundle", 100);
                        }
                        if (async) {
                            this.executor.submit(action);
                            break;
                        }
                        action.call();
                    }
                    catch (Exception e) {
                        logger.warning(e.getMessage());
                    }
                    break;
                }
            }
        }
    }

    @Override
    public void close() throws Exception {
        this.executor.shutdownNow();
    }

    private boolean isInstalled() {
        Bundle bundle = this.descriptor.getBundle(currentOS);
        return bundle != null && bundle.isInstalled();
    }

    private static Path getLocalSourcePath(Bundle bundle) {
        Path bundlePath = null;
        if (bundle != null) {
            File source = bundle.getSource();
            if (source != null && source.exists()) {
                bundlePath = source.toPath();
            } else {
                String entryPoint = bundle.getEntryPoint();
                if (entryPoint != null && Files.exists(baseModulePath.resolve(entryPoint), new LinkOption[0])) {
                    bundlePath = baseModulePath.resolve(entryPoint);
                }
            }
        }
        return bundlePath;
    }

    private void copy(Path source, Bundle bundle) throws IOException {
        Path targetFile;
        String targetLocation = bundle.getTargetLocation();
        if (targetLocation == null) {
            throw new IOException("No target defined");
        }
        Path targetPath = this.descriptor.resolveVariables(targetLocation).toPath();
        if (!Files.exists(targetPath, new LinkOption[0])) {
            Files.createDirectories(targetPath, new FileAttribute[0]);
        }
        if (!Files.exists(targetFile = targetPath.resolve(source.getFileName()), new LinkOption[0])) {
            Files.copy(source, targetFile, new CopyOption[0]);
        }
    }

    private Path download(Path target, Bundle bundle) throws IOException {
        String remoteURL = bundle.getDownloadURL();
        if (remoteURL == null || remoteURL.isEmpty()) {
            throw new IOException("No remote URL");
        }
        Path downloaded = this.download(remoteURL, target);
        if (this.progressMonitor != null) {
            this.progressMonitor.worked(50);
        }
        return downloaded;
    }

    private void uncompress(Path source, Bundle bundle) throws IOException {
        String targetLocation = bundle.getTargetLocation();
        if (targetLocation == null) {
            throw new IOException("No target defined");
        }
        ToolAdapterIO.unzip(source, this.descriptor.resolveVariables(targetLocation).toPath(), this.progressMonitor, this.taskCount);
        if (!bundle.isLocal()) {
            Files.deleteIfExists(source);
        }
    }

    private void install(Path source, Bundle bundle) throws IOException {
        int exit;
        String targetLocation = bundle.getTargetLocation();
        if (targetLocation == null) {
            throw new IOException("No target defined");
        }
        try {
            if (this.progressMonitor != null) {
                this.progressMonitor.setSubTaskName("Installing...");
            }
            this.copy(source, bundle);
            if (this.progressMonitor != null) {
                this.progressMonitor.worked(50 + 50 / this.taskCount);
            }
            Path exePath = this.descriptor.resolveVariables(targetLocation).toPath().resolve(bundle.getEntryPoint());
            ToolAdapterIO.fixPermissions(exePath);
            ArrayList<String> arguments = new ArrayList<String>();
            arguments.add(exePath.toString());
            String[] args = bundle.getCommandLineArguments();
            if (args != null) {
                Collections.addAll(arguments, args);
            }
            ProcessExecutor executor = new ProcessExecutor();
            executor.setConsumer(new DefaultOutputConsumer());
            executor.setWorkingDirectory(exePath.getParent().toFile());
            switch (Bundle.getCurrentOS()) {
                case linux: 
                case macosx: {
                    exit = this.executeAsBinary(executor, arguments);
                    if (exit == 0) break;
                    exit = this.executeAsUnixScript(executor, arguments);
                    break;
                }
                default: {
                    exit = this.executeAsBinary(executor, arguments);
                }
            }
            if (this.progressMonitor != null) {
                this.progressMonitor.worked(100);
            }
            Files.deleteIfExists(exePath);
        }
        catch (Exception ex) {
            logger.severe(ex.getMessage());
            throw new IOException(ex);
        }
        if (exit != 0) {
            throw new RuntimeException(String.format("Not successfully installed [exit code = %s]", exit));
        }
    }

    private int executeAsUnixScript(ProcessExecutor executor, List<String> arguments) {
        int exit = -1;
        try {
            ArrayList<String> newArgs = new ArrayList<String>();
            newArgs.add("/bin/bash");
            newArgs.add("-c");
            newArgs.add(String.join((CharSequence)" ", arguments));
            exit = executor.execute(newArgs);
        }
        catch (IOException ioex) {
            logger.warning(ioex.getMessage());
        }
        return exit;
    }

    private int executeAsBinary(ProcessExecutor executor, List<String> arguments) {
        int exit = -1;
        try {
            exit = executor.execute(arguments);
        }
        catch (IOException ioex) {
            logger.warning(ioex.getMessage());
        }
        return exit;
    }

    private static void fixUnsignedCertificates() throws KeyManagementException, NoSuchAlgorithmException {
        TrustManager[] trustAllCerts = new TrustManager[]{new X509TrustManager(){

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return null;
            }

            @Override
            public void checkClientTrusted(X509Certificate[] certs, String authType) {
            }

            @Override
            public void checkServerTrusted(X509Certificate[] certs, String authType) {
            }
        }};
        SSLContext sc = SSLContext.getInstance("SSL");
        sc.init(null, trustAllCerts, new SecureRandom());
        HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
        HostnameVerifier allHostsValid = new HostnameVerifier(){

            @Override
            public boolean verify(String hostname, SSLSession session) {
                return true;
            }
        };
        HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
    }

    private Path download(String remoteUrl, Path targetFile) throws IOException {
        if (remoteUrl == null || remoteUrl.isEmpty() || targetFile == null) {
            throw new IllegalArgumentException("Invalid download parameters");
        }
        URL url = new URL(remoteUrl);
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setConnectTimeout(20000);
        connection.setReadTimeout(20000);
        long length = connection.getContentLengthLong();
        double taskWeight = 100 / this.taskCount;
        double worked = 0.0;
        if (!Files.exists(targetFile, new LinkOption[0]) || length != Files.size(targetFile)) {
            Files.deleteIfExists(targetFile);
            Files.createDirectories(targetFile.getParent(), new FileAttribute[0]);
            Path tmpFile = Files.createTempFile(targetFile.getParent(), "tmp", null, new FileAttribute[0]);
            if (this.progressMonitor != null) {
                this.progressMonitor.setSubTaskName("Downloading...");
            }
            try (InputStream inputStream = connection.getInputStream();
                 OutputStream outputStream = Files.newOutputStream(tmpFile, new OpenOption[0]);){
                int read;
                byte[] buffer = new byte[262144];
                int totalRead = 0;
                while ((read = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, read);
                    totalRead += read;
                    if (this.progressMonitor != null) {
                        worked = (double)totalRead / (double)length * taskWeight;
                        this.progressMonitor.worked((int)worked);
                    }
                    Thread.yield();
                }
                outputStream.flush();
            }
            catch (IOException ex) {
                Files.deleteIfExists(tmpFile);
                throw new IOException(ex.getCause());
            }
            Files.move(tmpFile, targetFile, new CopyOption[0]);
        }
        return targetFile;
    }

    private void installFinished() {
        String alias = this.descriptor.getAlias();
        if (this.isInstalled()) {
            logger.info(String.format("Installation of bundle for %s completed", alias));
        } else {
            logger.severe(String.format("Bundle for %s has not been installed", alias));
        }
        if (this.callback != null) {
            try {
                this.callback.call();
            }
            catch (Exception e) {
                logger.warning(e.getMessage());
            }
        }
    }

    static {
        currentOS = Bundle.getCurrentOS();
        baseModulePath = SystemUtils.getApplicationDataDir().toPath().resolve("modules").resolve("lib");
        try {
            BundleInstaller.fixUnsignedCertificates();
        }
        catch (KeyManagementException e) {
            logger.warning(e.getMessage());
        }
        catch (NoSuchAlgorithmException e) {
            logger.warning(e.getMessage());
        }
    }

    class Action
    implements Callable<Void> {
        private Path source;
        private Bundle bundle;
        private BiConsumer<Path, Bundle> method;

        Action(Path source, Bundle bundle, BiConsumer<Path, Bundle> methodRef) throws Exception {
            this.source = source;
            this.bundle = bundle;
            this.method = methodRef;
        }

        @Override
        public Void call() throws Exception {
            try {
                this.method.accept(this.source, this.bundle);
                Void void_ = null;
                return void_;
            }
            finally {
                BundleInstaller.this.installFinished();
            }
        }
    }

    @FunctionalInterface
    static interface MethodReference<T, U>
    extends BiConsumer<T, U> {
        public static <T, U> MethodReference<T, U> from(BiConsumer<T, U> consumer) {
            return consumer::accept;
        }

        @Override
        default public void accept(T t, U u) {
            try {
                this.throwingAccept(t, u);
            }
            catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        }

        public void throwingAccept(T var1, U var2) throws Exception;
    }
}

