/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.utils;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.esa.snap.core.gpf.OperatorException;
import org.esa.snap.core.gpf.descriptor.ToolAdapterOperatorDescriptor;
import org.esa.snap.core.gpf.operators.tooladapter.ToolAdapterIO;
import org.esa.snap.core.gpf.operators.tooladapter.ToolAdapterRegistry;
import org.esa.snap.modules.ModulePackager;
import org.openide.modules.Places;

public enum AdapterWatcher {
    INSTANCE;

    private final Logger logger = Logger.getLogger(AdapterWatcher.class.getName());
    private WatchService watcher;
    private final WatchEvent.Kind[] eventTypes = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE};
    private Thread thread;
    private volatile boolean isRunning;
    private volatile boolean isSuspended;
    private Map<Path, String> jarAliases;
    private Map<Path, WatchKey> monitoredPaths;

    private AdapterWatcher() {
        try {
            this.watcher = FileSystems.getDefault().newWatchService();
            this.monitoredPaths = new HashMap<Path, WatchKey>();
            this.jarAliases = new HashMap<Path, String>();
            Path adaptersFolder = ToolAdapterIO.getAdaptersPath();
            File userDirectory = Places.getUserDirectory();
            Path nbUserModulesPath = Paths.get(userDirectory != null ? userDirectory.getAbsolutePath() : "", "modules");
            if (!Files.exists(nbUserModulesPath, new LinkOption[0])) {
                Files.createDirectory(nbUserModulesPath, new FileAttribute[0]);
            }
            this.readMap();
            this.monitorPath(adaptersFolder);
            this.monitorPath(nbUserModulesPath);
            File[] jars = nbUserModulesPath.toFile().listFiles(pathname -> pathname.getName().toLowerCase().endsWith("jar"));
            if (jars != null) {
                Arrays.stream(jars).forEach(f -> {
                    try {
                        this.processJarFile(f.toPath());
                    }
                    catch (Exception ex) {
                        this.logger.warning(ex.getMessage());
                    }
                });
            }
            this.handleUninstalledModules();
            this.thread = new Thread(() -> {
                while (this.isRunning) {
                    WatchKey key;
                    try {
                        key = this.watcher.take();
                    }
                    catch (InterruptedException ex) {
                        return;
                    }
                    key.pollEvents().forEach(event -> {
                        WatchEvent.Kind kind = event.kind();
                        WatchEvent ev = event;
                        Path fileName = (Path)ev.context();
                        boolean isJar = fileName.toString().endsWith(".jar");
                        if (!this.isSuspended) {
                            if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                                if (!isJar) {
                                    this.folderAdded(adaptersFolder.resolve(fileName));
                                } else {
                                    this.jarAdded(nbUserModulesPath.resolve(fileName));
                                }
                            } else if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                                if (!isJar) {
                                    this.folderDeleted(adaptersFolder.resolve(fileName));
                                } else {
                                    this.jarDeleted(nbUserModulesPath.resolve(fileName));
                                }
                            }
                        }
                    });
                    boolean valid = key.reset();
                    if (valid) continue;
                    break;
                }
            });
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public void startMonitor() {
        this.isRunning = true;
        this.isSuspended = false;
        this.thread.start();
    }

    public void stopMonitor() {
        this.isRunning = false;
    }

    public void suspend() {
        this.isSuspended = true;
    }

    public void resume() {
        this.isSuspended = false;
    }

    public void monitorPath(Path path) throws IOException {
        if (path != null && Files.isDirectory(path, new LinkOption[0])) {
            WatchKey key = path.register(this.watcher, this.eventTypes);
            this.monitoredPaths.put(path, key);
            this.logger.fine(String.format("Registered %s for watching", path.toString()));
        }
    }

    public void unmonitorPath(Path path) {
        WatchKey key;
        if (path != null && Files.isDirectory(path, new LinkOption[0]) && (key = this.monitoredPaths.remove(path)) != null) {
            key.cancel();
            this.logger.fine(String.format("Unregistered %s for watching", path.toString()));
        }
    }

    private void handleUninstalledModules() {
        Path[] paths = new Path[this.jarAliases.size()];
        this.jarAliases.keySet().toArray(paths);
        for (Path path : paths) {
            if (Files.exists(path, new LinkOption[0])) continue;
            this.jarDeleted(path);
        }
    }

    private void folderAdded(Path folder) {
        try {
            Thread.sleep(500L);
            ToolAdapterIO.registerAdapter((Path)folder);
        }
        catch (InterruptedException | OperatorException ex) {
            this.logger.warning("Could not load adapter for folder added in repository: " + folder.toString() + " (error:" + ex.getMessage());
        }
    }

    private void folderDeleted(Path folder) {
        String alias;
        ToolAdapterOperatorDescriptor operatorDescriptor;
        if (folder != null && (operatorDescriptor = ToolAdapterRegistry.INSTANCE.findByAlias(alias = folder.toFile().getName())) != null) {
            ToolAdapterIO.removeOperator((ToolAdapterOperatorDescriptor)operatorDescriptor);
        }
    }

    private void jarAdded(Path jarFile) {
        Path unpackLocation = this.processJarFile(jarFile);
        if (unpackLocation != null) {
            this.suspend();
            this.folderAdded(unpackLocation);
            this.saveMap();
            this.resume();
        } else {
            this.logger.warning(String.format("Jar %s has not been unpacked.", jarFile.toString()));
        }
    }

    private void jarDeleted(Path jarFile) {
        ToolAdapterOperatorDescriptor operatorDescriptor;
        String alias = this.jarAliases.get(jarFile);
        if (alias == null) {
            String fileName = jarFile.getFileName().toString().replace(".jar", "");
            int idx = fileName.lastIndexOf(".");
            alias = idx > 0 ? fileName.substring(idx + 1) : fileName;
        }
        if ((operatorDescriptor = ToolAdapterRegistry.INSTANCE.findByAlias(alias)) != null) {
            this.suspend();
            ToolAdapterIO.removeOperator((ToolAdapterOperatorDescriptor)operatorDescriptor);
            this.jarAliases.remove(jarFile);
            this.saveMap();
            this.resume();
        } else {
            this.logger.warning(String.format("Cannot find adapter for %s", jarFile.toString()));
        }
    }

    private Path processJarFile(Path jarFile) {
        Path destination = null;
        String aliasOrName = null;
        try {
            aliasOrName = ModulePackager.getAdapterAlias(jarFile.toFile());
        }
        catch (IOException iOException) {
            // empty catch block
        }
        if (aliasOrName != null) {
            this.jarAliases.put(jarFile, aliasOrName);
            destination = ToolAdapterIO.getAdaptersPath().resolve(aliasOrName);
            try {
                if (!Files.exists(destination, new LinkOption[0])) {
                    ModulePackager.unpackAdapterJar(jarFile.toFile(), destination.toFile());
                } else {
                    Path versionFile = destination.resolve("version.txt");
                    if (Files.exists(versionFile, new LinkOption[0])) {
                        String versionText = new String(Files.readAllBytes(versionFile));
                        String jarVersion = ModulePackager.getAdapterVersion(jarFile.toFile());
                        if (jarVersion != null && !versionText.equals(jarVersion)) {
                            ModulePackager.unpackAdapterJar(jarFile.toFile(), destination.toFile());
                            this.logger.fine(String.format("The adapter with the name %s and version %s was replaced by version %s", aliasOrName, versionText, jarVersion));
                        } else {
                            this.logger.fine(String.format("An adapter with the name %s and version %s already exists", aliasOrName, versionText));
                        }
                    } else {
                        ModulePackager.unpackAdapterJar(jarFile.toFile(), destination.toFile());
                    }
                }
            }
            catch (Exception e) {
                this.logger.severe(e.getMessage());
            }
        }
        return destination;
    }

    private void saveMap() {
        Path path = ToolAdapterIO.getAdaptersPath().resolve("installed.dat");
        StringBuilder builder = new StringBuilder();
        for (Map.Entry<Path, String> entry : this.jarAliases.entrySet()) {
            builder.append(entry.getKey().toString()).append(",").append(entry.getValue()).append("\n");
        }
        try {
            Files.write(path, builder.toString().getBytes(), new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.severe(e.getMessage());
        }
    }

    private void readMap() {
        Path path = ToolAdapterIO.getAdaptersPath().resolve("installed.dat");
        this.jarAliases.clear();
        try {
            if (Files.exists(path, new LinkOption[0])) {
                List<String> lines = Files.readAllLines(path);
                for (String line : lines) {
                    String[] tokens = line.split(",");
                    this.jarAliases.put(Paths.get(tokens[0], new String[0]), tokens[1]);
                }
            }
        }
        catch (IOException e) {
            this.logger.severe(e.getMessage());
        }
    }
}

