/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.unmixing;

import com.bc.ceres.core.ProgressMonitor;
import java.awt.Rectangle;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.RasterDataNode;
import org.esa.snap.core.gpf.Operator;
import org.esa.snap.core.gpf.OperatorException;
import org.esa.snap.core.gpf.OperatorSpi;
import org.esa.snap.core.gpf.Tile;
import org.esa.snap.core.gpf.annotations.OperatorMetadata;
import org.esa.snap.core.gpf.annotations.Parameter;
import org.esa.snap.core.gpf.annotations.SourceProduct;
import org.esa.snap.core.gpf.annotations.TargetProduct;
import org.esa.snap.core.util.ProductUtils;
import org.esa.snap.core.util.io.CsvReader;
import org.esa.snap.core.util.math.ConstrainedLSU;
import org.esa.snap.core.util.math.FullyConstrainedLSU;
import org.esa.snap.core.util.math.SpectralUnmixing;
import org.esa.snap.core.util.math.UnconstrainedLSU;
import org.esa.snap.unmixing.Endmember;

@OperatorMetadata(alias="Unmix", category="Raster/Image Analysis", version="1.0", authors="Norman Fomferra, Helmut Schiller", copyright="(c) 2007 by Brockmann Consult", description="Performs a linear spectral unmixing.")
public class SpectralUnmixingOp
extends Operator {
    private final String UC_LSU = "Unconstrained LSU";
    private final String C_LSU = "Constrained LSU";
    private final String FC_LSU = "Fully Constrained LSU";
    @SourceProduct(alias="source", description="The source product.")
    Product sourceProduct;
    @TargetProduct(description="The target product.")
    Product targetProduct;
    @Parameter(description="The list of spectral bands providing the source spectrum.", alias="sourceBands", itemAlias="band", rasterDataNodeType=Band.class)
    String[] sourceBandNames;
    @Parameter(description="The list of endmember spectra. Wavelengths must be given in nanometers.", itemAlias="endmember")
    Endmember[] endmembers;
    @Parameter(description="A text file containing (additional) endmembers in a table. Wavelengths must be given in nanometers.")
    File endmemberFile;
    @Parameter(description="The unmixing model.", valueSet={"Unconstrained LSU", "Constrained LSU", "Fully Constrained LSU"}, defaultValue="Constrained LSU")
    String unmixingModelName;
    @Parameter(description="The suffix for the generated abundance band names (name = endmember + suffix).", pattern="[a-zA-Z_0-9]*", notNull=true, defaultValue="_abundance")
    String abundanceBandNameSuffix;
    @Parameter(description="The suffix for the generated error band names (name = source + suffix).", pattern="[a-zA-Z_0-9]*", notNull=true, defaultValue="_error")
    String errorBandNameSuffix;
    @Parameter(description="If 'true', error bands for all source bands will be generated.", defaultValue="false")
    boolean computeErrorBands;
    @Parameter(description="Minimum spectral bandwidth used for endmember wavelength matching.", defaultValue="10.0", interval="(0,*)", unit="nm")
    double minBandwidth;
    private Band[] sourceBands;
    private Band[] abundanceBands;
    private Band[] errorBands;
    private Band summaryErrorBand;
    private SpectralUnmixing spectralUnmixing;
    private boolean computeTileMethodUsable = true;

    public String[] getSourceBandNames() {
        return this.sourceBandNames;
    }

    public void setSourceBandNames(String[] sourceBandNames) {
        this.sourceBandNames = sourceBandNames;
    }

    public Endmember[] getEndmembers() {
        return this.endmembers;
    }

    public void setEndmembers(Endmember[] endmembers) {
        this.endmembers = endmembers;
    }

    public File getEndmemberFile() {
        return this.endmemberFile;
    }

    public void setEndmemberFile(File endmemberFile) {
        this.endmemberFile = endmemberFile;
    }

    public String getUnmixingModelName() {
        return this.unmixingModelName;
    }

    public void setUnmixingModelName(String unmixingModelName) {
        this.unmixingModelName = unmixingModelName;
    }

    public String getAbundanceBandNameSuffix() {
        return this.abundanceBandNameSuffix;
    }

    public void setAbundanceBandNameSuffix(String abundanceBandNameSuffix) {
        this.abundanceBandNameSuffix = abundanceBandNameSuffix;
    }

    public String getErrorBandNameSuffix() {
        return this.errorBandNameSuffix;
    }

    public void setErrorBandNameSuffix(String errorBandNameSuffix) {
        this.errorBandNameSuffix = errorBandNameSuffix;
    }

    public boolean getComputeErrorBands() {
        return this.computeErrorBands;
    }

    public void setComputeErrorBands(boolean computeErrorBands) {
        this.computeErrorBands = computeErrorBands;
    }

    public double getMinBandwidth() {
        return this.minBandwidth;
    }

    public void setMinBandwidth(double minBandwidth) {
        this.minBandwidth = minBandwidth;
    }

    public boolean canComputeTile() {
        return !this.computeErrorBands;
    }

    public boolean canComputeTileStack() {
        return true;
    }

    public void initialize() throws OperatorException {
        int i;
        if (this.endmemberFile != null) {
            this.loadEndmemberFile();
        }
        if (this.sourceBandNames == null || this.sourceBandNames.length == 0) {
            Band[] bands = this.sourceProduct.getBands();
            ArrayList<String> bandNameList = new ArrayList<String>();
            for (Band band : bands) {
                if (!(band.getSpectralWavelength() > 0.0f)) continue;
                bandNameList.add(band.getName());
            }
            this.sourceBandNames = bandNameList.toArray(new String[bandNameList.size()]);
        }
        this.validateParameters();
        this.sourceBands = new Band[this.sourceBandNames.length];
        for (int i2 = 0; i2 < this.sourceBandNames.length; ++i2) {
            String sourceBandName = this.sourceBandNames[i2];
            Band sourceBand = this.sourceProduct.getBand(sourceBandName);
            if (sourceBand == null) {
                throw new OperatorException("Source band not found: " + sourceBandName);
            }
            if (sourceBand.getSpectralWavelength() <= 0.0f) {
                throw new OperatorException("Source band without spectral wavelength: " + sourceBandName);
            }
            this.sourceBands[i2] = sourceBand;
        }
        this.ensureSingleRasterSize((RasterDataNode[])this.sourceBands);
        int numSourceBands = this.sourceBands.length;
        int numEndmembers = this.endmembers.length;
        if (numSourceBands < numEndmembers) {
            throw new OperatorException("Number of source bands must be >= number of endmembers.");
        }
        double[][] lsuMatrixElements = new double[numSourceBands][numEndmembers];
        for (int j = 0; j < numEndmembers; ++j) {
            Endmember endmember = this.endmembers[j];
            double[] wavelengths = endmember.getWavelengths();
            double[] radiations = endmember.getRadiations();
            for (int i3 = 0; i3 < numSourceBands; ++i3) {
                float bandwidth;
                Band sourceBand = this.sourceBands[i3];
                float wavelength = sourceBand.getSpectralWavelength();
                int k = SpectralUnmixingOp.findEndmemberSpectralIndex(wavelengths, wavelength, Math.max((double)(bandwidth = sourceBand.getSpectralBandwidth()), this.minBandwidth));
                if (k == -1) {
                    throw new OperatorException(String.format("Band %s: No matching endmember wavelength found (%f nm)", sourceBand.getName(), Float.valueOf(wavelength)));
                }
                lsuMatrixElements[i3][j] = radiations[k];
            }
        }
        if ("Unconstrained LSU".equals(this.unmixingModelName)) {
            this.spectralUnmixing = new UnconstrainedLSU(lsuMatrixElements);
        } else if ("Constrained LSU".equals(this.unmixingModelName)) {
            this.spectralUnmixing = new ConstrainedLSU(lsuMatrixElements);
        } else if ("Fully Constrained LSU".equals(this.unmixingModelName)) {
            this.spectralUnmixing = new FullyConstrainedLSU(lsuMatrixElements);
        } else if (this.unmixingModelName == null) {
            this.spectralUnmixing = new UnconstrainedLSU(lsuMatrixElements);
        }
        int width = this.sourceBands[0].getRasterWidth();
        int height = this.sourceBands[0].getRasterHeight();
        this.targetProduct = new Product(this.sourceProduct.getName() + "_unmixed", "SpectralUnmixing", width, height);
        this.abundanceBands = new Band[numEndmembers];
        for (i = 0; i < numEndmembers; ++i) {
            this.abundanceBands[i] = this.targetProduct.addBand(this.endmembers[i].getName() + this.abundanceBandNameSuffix, 30);
        }
        if (this.computeErrorBands) {
            this.errorBands = new Band[numSourceBands];
            for (i = 0; i < this.errorBands.length; ++i) {
                String erroBandName = this.sourceBands[i].getName() + this.errorBandNameSuffix;
                this.errorBands[i] = this.targetProduct.addBand(erroBandName, 30);
                ProductUtils.copySpectralBandProperties((Band)this.sourceBands[i], (Band)this.errorBands[i]);
            }
            this.summaryErrorBand = this.targetProduct.addBand("summary_error", 30);
            this.summaryErrorBand.setDescription("Root mean square error");
        }
        ProductUtils.copyMetadata((Product)this.sourceProduct, (Product)this.targetProduct);
        if (this.sourceProduct.getSceneRasterSize().equals(this.targetProduct.getSceneRasterSize())) {
            ProductUtils.copyTiePointGrids((Product)this.sourceProduct, (Product)this.targetProduct);
            ProductUtils.copyGeoCoding((Product)this.sourceProduct, (Product)this.targetProduct);
        }
    }

    public void computeTile(Band band, Tile targetTile, ProgressMonitor pm) throws OperatorException {
        Rectangle rectangle = targetTile.getRectangle();
        int i = this.getTargetBandIndex(targetTile);
        if (i == -1) {
            return;
        }
        Tile[] sourceRaster = this.getSourceTiles(rectangle);
        for (int y = rectangle.y; y < rectangle.y + rectangle.height; ++y) {
            double[][] ia = this.getLineSpectra(sourceRaster, rectangle, y);
            double[][] oa = this.unmix(ia);
            SpectralUnmixingOp.setAbundances(rectangle, targetTile, y, oa[i]);
            this.checkForCancellation();
        }
    }

    public void computeTileStack(Map<Band, Tile> targetTiles, Rectangle targetTileRectangle, ProgressMonitor pm) throws OperatorException {
        Tile[] abundanceTiles = new Tile[this.abundanceBands.length];
        Tile[] intensityTiles = this.getSourceTiles(targetTileRectangle);
        for (int i = 0; i < this.abundanceBands.length; ++i) {
            abundanceTiles[i] = targetTiles.get(this.abundanceBands[i]);
        }
        Tile[] errorTiles = null;
        Tile summaryErrorTile = null;
        if (this.computeErrorBands) {
            errorTiles = new Tile[intensityTiles.length];
            for (int i = 0; i < errorTiles.length; ++i) {
                errorTiles[i] = targetTiles.get(this.errorBands[i]);
            }
            summaryErrorTile = targetTiles.get(this.summaryErrorBand);
        }
        for (int y = targetTileRectangle.y; y < targetTileRectangle.y + targetTileRectangle.height; ++y) {
            double[][] ia = this.getLineSpectra(intensityTiles, targetTileRectangle, y);
            double[][] oa = this.unmix(ia);
            for (int i = 0; i < this.abundanceBands.length; ++i) {
                SpectralUnmixingOp.setAbundances(targetTileRectangle, abundanceTiles[i], y, oa[i]);
                this.checkForCancellation();
            }
            if (!this.computeErrorBands) continue;
            double[][] ia2 = this.mix(oa);
            SpectralUnmixingOp.computeErrorTiles(targetTileRectangle, errorTiles, summaryErrorTile, y, ia, ia2);
        }
    }

    private static void computeErrorTiles(Rectangle rectangle, Tile[] errorTilesc, Tile summaryErrorTile, int y, double[][] ia, double[][] ia2) {
        double[] errSqrSumRow = new double[rectangle.width];
        for (int i = 0; i < errorTilesc.length; ++i) {
            Tile errorTile = errorTilesc[i];
            double[] iaRow = ia[i];
            double[] ia2Row = ia2[i];
            int j = 0;
            for (int x = rectangle.x; x < rectangle.x + rectangle.width; ++x) {
                double err = iaRow[j] - ia2Row[j];
                errorTile.setSample(x, y, err);
                int n = j++;
                errSqrSumRow[n] = errSqrSumRow[n] + err * err;
            }
        }
        if (summaryErrorTile != null) {
            int j = 0;
            for (int x = rectangle.x; x < rectangle.x + rectangle.width; ++x) {
                summaryErrorTile.setSample(x, y, Math.sqrt(errSqrSumRow[j] / (double)errorTilesc.length));
                ++j;
            }
        }
    }

    private Tile[] getSourceTiles(Rectangle rectangle) throws OperatorException {
        Tile[] sourceRaster = new Tile[this.sourceBands.length];
        for (int i = 0; i < this.sourceBands.length; ++i) {
            sourceRaster[i] = this.getSourceTile((RasterDataNode)this.sourceBands[i], rectangle);
        }
        return sourceRaster;
    }

    private static void setAbundances(Rectangle rectangle, Tile targetTile, int y, double[] oaRow) {
        int i = 0;
        for (int x = rectangle.x; x < rectangle.x + rectangle.width; ++x) {
            targetTile.setSample(x, y, oaRow[i]);
            ++i;
        }
    }

    private double[][] getLineSpectra(Tile[] sourceRasters, Rectangle rectangle, int y) throws OperatorException {
        double[][] ia = new double[this.sourceBands.length][rectangle.width];
        for (int i = 0; i < this.sourceBands.length; ++i) {
            for (int x = rectangle.x; x < rectangle.x + rectangle.width; ++x) {
                ia[i][x - rectangle.x] = sourceRasters[i].getSampleDouble(x, y);
            }
        }
        return ia;
    }

    private int getTargetBandIndex(Tile targetTile) {
        int index = -1;
        for (int i = 0; i < this.abundanceBands.length; ++i) {
            Band targetBand = this.abundanceBands[i];
            if (targetTile.getRasterDataNode() != targetBand) continue;
            index = i;
            break;
        }
        return index;
    }

    private double[][] unmix(double[][] ia) {
        return this.spectralUnmixing.unmix(ia);
    }

    private double[][] mix(double[][] ia) {
        return this.spectralUnmixing.mix(ia);
    }

    public static int findEndmemberSpectralIndex(double[] endmemberWavelengths, double sourceBandWavelength, double maxBandwidth) {
        double minDelta = Double.MAX_VALUE;
        int bestIndex = -1;
        for (int i = 0; i < endmemberWavelengths.length; ++i) {
            double delta = Math.abs(endmemberWavelengths[i] - sourceBandWavelength);
            if (!(delta <= maxBandwidth) || !(delta <= minDelta)) continue;
            minDelta = delta;
            bestIndex = i;
        }
        return bestIndex;
    }

    private void loadEndmemberFile() throws OperatorException {
        try (FileReader fileReader = new FileReader(this.endmemberFile);){
            List<Endmember> newEndmembers = SpectralUnmixingOp.readGraphs(fileReader);
            ArrayList<Endmember> list = new ArrayList<Endmember>();
            if (this.endmembers != null) {
                list.addAll(Arrays.asList(this.endmembers));
            }
            list.addAll(newEndmembers);
            this.endmembers = list.toArray(new Endmember[list.size()]);
        }
        catch (IOException e) {
            throw new OperatorException((Throwable)e);
        }
    }

    private static List<Endmember> readGraphs(Reader reader) throws IOException {
        CsvReader csvReader = new CsvReader(reader, new char[]{'\t'});
        ArrayList<Endmember> endmemberList = new ArrayList<Endmember>(5);
        ArrayList<double[]> dataRecords = new ArrayList<double[]>(20);
        String[] headerRecord = csvReader.readRecord();
        while (true) {
            if (headerRecord.length < 2) {
                throw new IOException("Invalid format.");
            }
            String[] record = csvReader.readRecord();
            if (record == null) break;
            double[] dataRecord = SpectralUnmixingOp.toDoubles(record);
            if (dataRecord != null) {
                if (dataRecord.length != headerRecord.length) {
                    throw new IOException("Invalid format.");
                }
                dataRecords.add(dataRecord);
                continue;
            }
            SpectralUnmixingOp.readGraphGroup(headerRecord, dataRecords, endmemberList);
            headerRecord = record;
        }
        SpectralUnmixingOp.readGraphGroup(headerRecord, dataRecords, endmemberList);
        return endmemberList;
    }

    public static double[] toDoubles(String[] textRecord) throws IOException {
        double[] doubleRecord = new double[textRecord.length];
        for (int i = 0; i < textRecord.length; ++i) {
            try {
                doubleRecord[i] = Double.valueOf(textRecord[i]);
                continue;
            }
            catch (NumberFormatException e) {
                return null;
            }
        }
        return doubleRecord;
    }

    private static void readGraphGroup(String[] headerRecord, List<double[]> dataRecords, List<Endmember> endmemberList) {
        if (dataRecords.size() > 0) {
            double[] xValues = new double[dataRecords.size()];
            for (int j = 0; j < dataRecords.size(); ++j) {
                xValues[j] = dataRecords.get(j)[0];
            }
            double[] dataRecord0 = dataRecords.get(0);
            for (int i = 1; i < dataRecord0.length; ++i) {
                double[] yValues = new double[dataRecords.size()];
                for (int j = 0; j < dataRecords.size(); ++j) {
                    yValues[j] = dataRecords.get(j)[i];
                }
                endmemberList.add(new Endmember(headerRecord[i], xValues, yValues));
            }
        }
        dataRecords.clear();
    }

    private void validateParameters() throws OperatorException {
        if (this.sourceBandNames == null || this.sourceBandNames.length == 0) {
            throw new OperatorException("Parameter 'sourceBandNames' not set.");
        }
        if (this.endmemberFile == null && (this.endmembers == null || this.endmembers.length == 0)) {
            throw new OperatorException("Parameter 'endmemberFile' and 'endmembers' not set.");
        }
    }

    public static class Spi
    extends OperatorSpi {
        public Spi() {
            super(SpectralUnmixingOp.class);
        }
    }
}

