/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.core.datamodel;

import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import org.esa.snap.core.datamodel.GeoCoding;
import org.esa.snap.core.datamodel.GeoPos;
import org.esa.snap.core.datamodel.PixelPos;
import org.esa.snap.core.datamodel.RasterDataNode;
import org.esa.snap.core.datamodel.TiePointGeoCoding;
import org.esa.snap.core.util.Debug;
import org.esa.snap.core.util.Guardian;
import org.esa.snap.core.util.ProductUtils;

public class Graticule {
    private final GeneralPath[] _linePaths;
    private final TextGlyph[] _textGlyphs;

    private Graticule(GeneralPath[] paths, TextGlyph[] textGlyphs) {
        this._linePaths = paths;
        this._textGlyphs = textGlyphs;
    }

    public GeneralPath[] getLinePaths() {
        return this._linePaths;
    }

    public TextGlyph[] getTextGlyphs() {
        return this._textGlyphs;
    }

    public static Graticule create(RasterDataNode raster, boolean autoDeterminingSteps, int gridCellSize, float latMajorStep, float lonMajorStep) {
        Guardian.assertNotNull("product", raster);
        GeoCoding geoCoding = raster.getGeoCoding();
        if (geoCoding == null || raster.getRasterWidth() < 16 || raster.getRasterHeight() < 16) {
            return null;
        }
        if (autoDeterminingSteps) {
            PixelPos pixelPos1 = new PixelPos(0.5f * (float)raster.getRasterWidth(), 0.5f * (float)raster.getRasterHeight());
            PixelPos pixelPos2 = new PixelPos(pixelPos1.x + 1.0, pixelPos1.y + 1.0);
            GeoPos geoPos1 = geoCoding.getGeoPos(pixelPos1, null);
            GeoPos geoPos2 = geoCoding.getGeoPos(pixelPos2, null);
            double deltaLat = Math.abs(geoPos2.lat - geoPos1.lat);
            double deltaLon = Math.abs(geoPos2.lon - geoPos1.lon);
            if (deltaLon > 180.0) {
                deltaLon += 360.0;
            }
            Debug.trace("Graticule.create: deltaLat=" + deltaLat + ", deltaLon=" + deltaLon);
            lonMajorStep = latMajorStep = (float)Graticule.compose(Graticule.normalize((double)gridCellSize * 0.5 * (deltaLon + deltaLat), null));
        }
        Debug.trace("Graticule.create: latMajorStep=" + latMajorStep + ", lonMajorStep=" + lonMajorStep);
        float latMinorStep = latMajorStep / 4.0f;
        float lonMinorStep = lonMajorStep / 4.0f;
        int geoBoundaryStep = Graticule.getGeoBoundaryStep(geoCoding);
        Debug.trace("Graticule.create: geoBoundaryStep=" + geoBoundaryStep);
        GeoPos[] geoBoundary = ProductUtils.createGeoBoundary(raster, null, geoBoundaryStep);
        ProductUtils.normalizeGeoPolygon(geoBoundary);
        double xMin = 1.0E10;
        double yMin = 1.0E10;
        double xMax = -1.0E10;
        double yMax = -1.0E10;
        for (GeoPos geoPos : geoBoundary) {
            xMin = Math.min(xMin, geoPos.lon);
            yMin = Math.min(yMin, geoPos.lat);
            xMax = Math.max(xMax, geoPos.lon);
            yMax = Math.max(yMax, geoPos.lat);
        }
        List<List<Coord>> parallelList = Graticule.computeParallelList(raster.getGeoCoding(), geoBoundary, latMajorStep, lonMinorStep, yMin, yMax);
        List<List<Coord>> meridianList = Graticule.computeMeridianList(raster.getGeoCoding(), geoBoundary, lonMajorStep, latMinorStep, xMin, xMax);
        GeneralPath[] paths = Graticule.createPaths(parallelList, meridianList);
        TextGlyph[] textGlyphs = Graticule.createTextGlyphs(parallelList, meridianList);
        return new Graticule(paths, textGlyphs);
    }

    private static int getGeoBoundaryStep(GeoCoding geoCoding) {
        int step = 16;
        if (geoCoding instanceof TiePointGeoCoding) {
            TiePointGeoCoding tiePointGeoCoding = (TiePointGeoCoding)geoCoding;
            step = (int)Math.round(Math.min(tiePointGeoCoding.getLonGrid().getSubSamplingX(), tiePointGeoCoding.getLonGrid().getSubSamplingY()));
        }
        return step;
    }

    private static List<List<Coord>> computeParallelList(GeoCoding geoCoding, GeoPos[] geoBoundary, double latMajorStep, double lonMinorStep, double yMin, double yMax) {
        ArrayList<List<Coord>> parallelList = new ArrayList<List<Coord>>();
        ArrayList<GeoPos> intersectionList = new ArrayList<GeoPos>();
        for (double my = latMajorStep * Math.floor(yMin / latMajorStep); my <= yMax; my += latMajorStep) {
            intersectionList.clear();
            Graticule.computeParallelIntersections(geoBoundary, my, intersectionList);
            if (intersectionList.size() <= 0 || intersectionList.size() % 2 != 0) continue;
            GeoPos[] intersections = intersectionList.toArray(new GeoPos[intersectionList.size()]);
            Arrays.sort(intersections, new GeoPosLonComparator());
            ArrayList<Coord> parallel = new ArrayList<Coord>();
            for (int i = 0; i < intersections.length; i += 2) {
                GeoPos int1 = intersections[i];
                GeoPos int2 = intersections[i + 1];
                double lat = int1.lat;
                double lon = int1.lon;
                int k = 0;
                while (k <= 1) {
                    GeoPos geoPos = new GeoPos(lat, Graticule.limitLon(lon));
                    PixelPos pixelPos = geoCoding.getPixelPos(geoPos, null);
                    parallel.add(new Coord(geoPos, pixelPos));
                    if (!((lon += lonMinorStep) >= int2.lon)) continue;
                    lon = int2.lon;
                    ++k;
                }
            }
            parallelList.add(parallel);
        }
        return parallelList;
    }

    private static List<List<Coord>> computeMeridianList(GeoCoding geoCoding, GeoPos[] geoBoundary, double lonMajorStep, double latMinorStep, double xMin, double xMax) {
        ArrayList<List<Coord>> meridianList = new ArrayList<List<Coord>>();
        ArrayList<GeoPos> intersectionList = new ArrayList<GeoPos>();
        for (double mx = lonMajorStep * Math.floor(xMin / lonMajorStep); mx <= xMax; mx += lonMajorStep) {
            intersectionList.clear();
            Graticule.computeMeridianIntersections(geoBoundary, mx, intersectionList);
            if (intersectionList.size() <= 0 || intersectionList.size() % 2 != 0) continue;
            GeoPos[] intersections = intersectionList.toArray(new GeoPos[intersectionList.size()]);
            Arrays.sort(intersections, new GeoPosLatComparator());
            ArrayList<Coord> meridian = new ArrayList<Coord>();
            for (int i = intersections.length - 2; i >= 0; i -= 2) {
                GeoPos int1 = intersections[i + 1];
                GeoPos int2 = intersections[i];
                double lat = int1.lat;
                double lon = int1.lon;
                int k = 0;
                while (k <= 1) {
                    GeoPos geoPos = new GeoPos(lat, Graticule.limitLon(lon));
                    PixelPos pixelPos = geoCoding.getPixelPos(geoPos, null);
                    meridian.add(new Coord(geoPos, pixelPos));
                    if (!((lat -= latMinorStep) <= int2.lat)) continue;
                    lat = int2.lat;
                    ++k;
                }
            }
            meridianList.add(meridian);
        }
        return meridianList;
    }

    private static void computeParallelIntersections(GeoPos[] geoBoundary, double my, List<GeoPos> intersectionList) {
        double p0x = 0.0;
        double p0y = 0.0;
        for (int i = 0; i < geoBoundary.length; ++i) {
            double pa;
            GeoPos geoPos = geoBoundary[i];
            double p1x = geoPos.lon;
            double p1y = geoPos.lat;
            if (i > 0 && (my >= p0y && my <= p1y || my >= p1y && my <= p0y) && p1y - p0y != 0.0 && (pa = (my - p0y) / (p1y - p0y)) >= 0.0 && pa < 1.0) {
                double mx = p0x + pa * (p1x - p0x);
                intersectionList.add(new GeoPos(my, mx));
            }
            p0x = p1x;
            p0y = p1y;
        }
    }

    private static void computeMeridianIntersections(GeoPos[] geoBoundary, double mx, List<GeoPos> intersectionList) {
        double p0x = 0.0;
        double p0y = 0.0;
        for (int i = 0; i < geoBoundary.length; ++i) {
            double pa;
            GeoPos geoPos = geoBoundary[i];
            double p1x = geoPos.lon;
            double p1y = geoPos.lat;
            if (i > 0 && (mx >= p0x && mx <= p1x || mx >= p1x && mx <= p0x) && p1x - p0x != 0.0 && (pa = (mx - p0x) / (p1x - p0x)) >= 0.0 && pa < 1.0) {
                double my = p0y + pa * (p1y - p0y);
                intersectionList.add(new GeoPos(my, mx));
            }
            p0x = p1x;
            p0y = p1y;
        }
    }

    private static GeneralPath[] createPaths(List<List<Coord>> parallelList, List<List<Coord>> meridianList) {
        ArrayList<GeneralPath> generalPathList = new ArrayList<GeneralPath>();
        Graticule.addToPath(parallelList, generalPathList);
        Graticule.addToPath(meridianList, generalPathList);
        return generalPathList.toArray(new GeneralPath[generalPathList.size()]);
    }

    private static void addToPath(List<List<Coord>> lineList, List<GeneralPath> generalPathList) {
        for (List<Coord> coordList : lineList) {
            if (coordList.size() < 2) continue;
            GeneralPath generalPath = new GeneralPath();
            boolean restart = true;
            for (Coord coord : coordList) {
                PixelPos pixelPos = coord.pixelPos;
                if (pixelPos.isValid()) {
                    if (restart) {
                        generalPath.moveTo(pixelPos.x, pixelPos.y);
                    } else {
                        generalPath.lineTo(pixelPos.x, pixelPos.y);
                    }
                    restart = false;
                    continue;
                }
                restart = true;
            }
            generalPathList.add(generalPath);
        }
    }

    private static TextGlyph[] createTextGlyphs(List<List<Coord>> parallelList, List<List<Coord>> meridianList) {
        ArrayList<TextGlyph> textGlyphList = new ArrayList<TextGlyph>();
        Graticule.createParallelTextGlyphs(parallelList, textGlyphList);
        Graticule.createMeridianTextGlyphs(meridianList, textGlyphList);
        return textGlyphList.toArray(new TextGlyph[textGlyphList.size()]);
    }

    private static void createParallelTextGlyphs(List<List<Coord>> parallelList, List<TextGlyph> textGlyphList) {
        for (List<Coord> parallel : parallelList) {
            Coord coord2;
            Coord coord1;
            if (parallel.size() >= 3) {
                coord1 = parallel.get(1);
                if (!Graticule.isCoordPairValid(coord1, coord2 = parallel.get(2))) continue;
                textGlyphList.add(Graticule.createLatTextGlyph(coord1, coord2));
                continue;
            }
            if (parallel.size() < 2 || !Graticule.isCoordPairValid(coord1 = parallel.get(0), coord2 = parallel.get(1))) continue;
            textGlyphList.add(Graticule.createLatTextGlyph(coord1, coord2));
        }
    }

    private static void createMeridianTextGlyphs(List<List<Coord>> meridianList, List<TextGlyph> textGlyphList) {
        for (List<Coord> meridian : meridianList) {
            Coord coord2;
            Coord coord1;
            if (meridian.size() >= 3) {
                coord1 = meridian.get(1);
                if (!Graticule.isCoordPairValid(coord1, coord2 = meridian.get(2))) continue;
                textGlyphList.add(Graticule.createLonTextGlyph(coord1, coord2));
                continue;
            }
            if (meridian.size() < 2 || !Graticule.isCoordPairValid(coord1 = meridian.get(0), coord2 = meridian.get(1))) continue;
            textGlyphList.add(Graticule.createLonTextGlyph(coord1, coord2));
        }
    }

    private static boolean isCoordPairValid(Coord coord1, Coord coord2) {
        return coord1.pixelPos.isValid() && coord2.pixelPos.isValid();
    }

    private static TextGlyph createLatTextGlyph(Coord coord1, Coord coord2) {
        return Graticule.createTextGlyph(coord1.geoPos.getLatString(), coord1, coord2);
    }

    private static TextGlyph createLonTextGlyph(Coord coord1, Coord coord2) {
        return Graticule.createTextGlyph(coord1.geoPos.getLonString(), coord1, coord2);
    }

    private static TextGlyph createTextGlyph(String text, Coord coord1, Coord coord2) {
        double angle = Math.atan2(coord2.pixelPos.y - coord1.pixelPos.y, coord2.pixelPos.x - coord1.pixelPos.x);
        return new TextGlyph(text, coord1.pixelPos.x, coord1.pixelPos.y, angle);
    }

    private static double limitLon(double lon) {
        while (lon < -180.0) {
            lon += 360.0;
        }
        while (lon > 180.0) {
            lon -= 360.0;
        }
        return lon;
    }

    private static double[] normalize(double x, double[] result) {
        double mantissa;
        double exponent = x == 0.0 ? 0.0 : Math.ceil(Math.log(Math.abs(x)) / Math.log(10.0));
        double d = mantissa = x == 0.0 ? 0.0 : x / Math.pow(10.0, exponent);
        if (result == null) {
            result = new double[]{mantissa, exponent};
        }
        return result;
    }

    private static double compose(double[] components) {
        double mantissa = components[0];
        double exponent = components[1];
        double mantissaRounded = mantissa < 0.15 ? 0.1 : (mantissa < 0.225 ? 0.2 : (mantissa < 0.375 ? 0.25 : (mantissa < 0.75 ? 0.5 : 1.0)));
        return mantissaRounded * Math.pow(10.0, exponent);
    }

    private static GeneralPath createPixelBoundaryPath(GeoCoding geoCoding, GeoPos[] geoBoundary) {
        GeneralPath generalPath = new GeneralPath();
        boolean restart = true;
        for (GeoPos geoPos : geoBoundary) {
            geoPos.lon = Graticule.limitLon(geoPos.lon);
            PixelPos pixelPos = geoCoding.getPixelPos(geoPos, null);
            if (pixelPos.isValid()) {
                if (restart) {
                    generalPath.moveTo(pixelPos.x, pixelPos.y);
                } else {
                    generalPath.lineTo(pixelPos.x, pixelPos.y);
                }
                restart = false;
                continue;
            }
            restart = true;
        }
        return generalPath;
    }

    private static class GeoPosLonComparator
    implements Comparator<GeoPos> {
        private GeoPosLonComparator() {
        }

        @Override
        public int compare(GeoPos geoPos1, GeoPos geoPos2) {
            double delta = geoPos1.lon - geoPos2.lon;
            if (delta < 0.0) {
                return -1;
            }
            if (delta > 0.0) {
                return 1;
            }
            return 0;
        }
    }

    private static class GeoPosLatComparator
    implements Comparator<GeoPos> {
        private GeoPosLatComparator() {
        }

        @Override
        public int compare(GeoPos geoPos1, GeoPos geoPos2) {
            double delta = geoPos1.lat - geoPos2.lat;
            if (delta < 0.0) {
                return -1;
            }
            if (delta > 0.0) {
                return 1;
            }
            return 0;
        }
    }

    private static class Coord {
        GeoPos geoPos;
        PixelPos pixelPos;

        public Coord(GeoPos geoPos, PixelPos pixelPos) {
            this.geoPos = geoPos;
            this.pixelPos = pixelPos;
        }
    }

    public static class TextGlyph {
        private final String text;
        private final double x;
        private final double y;
        private final double angle;

        public TextGlyph(String text, double x, double y, double angle) {
            this.text = text;
            this.x = x;
            this.y = y;
            this.angle = angle;
        }

        public String getText() {
            return this.text;
        }

        public double getX() {
            return this.x;
        }

        public double getY() {
            return this.y;
        }

        public double getAngle() {
            return this.angle;
        }
    }
}

