/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.worldwind.layers.rpf;

import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.util.Logging;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class RPFFileIndex {
    private final Table rpfFileTable = new Table();
    private final Table waveletTable = new Table();
    private final Table directoryTable = new Table();
    private final IndexProperties properties;
    private static final String FILE_ID = "RPF_FILE_INDEX";
    private static final String VERSION = "VERSION_0_1";
    private static final int FILE_ID_LENGTH = 16;
    private static final int VERSION_LENGTH = 16;
    private static final String CHARACTER_ENCODING = "UTF-8";

    public RPFFileIndex() {
        this.rpfFileTable.setRecordFactory(new RecordFactory(){

            @Override
            public Record newRecord(long l) {
                return new RPFFileRecord(l);
            }
        });
        this.waveletTable.setRecordFactory(new RecordFactory(){

            @Override
            public Record newRecord(long l) {
                return new WaveletRecord(l);
            }
        });
        this.directoryTable.setRecordFactory(new RecordFactory(){

            @Override
            public Record newRecord(long l) {
                return new DirectoryRecord(l);
            }
        });
        this.properties = new IndexProperties();
    }

    public Table getRPFFileTable() {
        return this.rpfFileTable;
    }

    public Table getWaveletTable() {
        return this.waveletTable;
    }

    public Table getDirectoryTable() {
        return this.directoryTable;
    }

    public IndexProperties getIndexProperties() {
        return this.properties;
    }

    public File getRPFFile(long l) {
        if (l == -1L) {
            String string = "key is invalid: " + l;
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        File file = null;
        RPFFileRecord rPFFileRecord = (RPFFileRecord)this.rpfFileTable.getRecord(l);
        if (rPFFileRecord != null) {
            file = this.getFile(rPFFileRecord.getFilename(), rPFFileRecord.getDirectorySecondaryKey());
        }
        return file;
    }

    public File getWaveletFile(long l) {
        if (l == -1L) {
            String string = "key is invalid: " + l;
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        File file = null;
        WaveletRecord waveletRecord = (WaveletRecord)this.waveletTable.getRecord(l);
        if (waveletRecord != null) {
            file = this.getFile(waveletRecord.getFilename(), waveletRecord.getDirectorySecondaryKey());
        }
        return file;
    }

    private File getFile(String string, long l) {
        File file = null;
        if (l != -1L) {
            DirectoryRecord directoryRecord = (DirectoryRecord)this.directoryTable.getRecord(l);
            file = new File(directoryRecord != null ? directoryRecord.getPath() : null, string);
        }
        return file;
    }

    public Record createRPFFileRecord(File file) {
        if (file == null) {
            String string = Logging.getMessage("nullValue.FileIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        String string = file.getParent();
        Record record = this.createDirectoryRecord(string);
        String string2 = file.getName();
        Record record2 = this.rpfFileTable.createRecord();
        ((RPFFileRecord)record2).filename = string2;
        ((RPFFileRecord)record2).directorySecondaryKey = record != null ? record.getKey() : -1L;
        return record2;
    }

    public Record createWaveletRecord(File file, long l) {
        if (file == null) {
            String string = Logging.getMessage("nullValue.FileIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        String string = file.getParent();
        Record record = this.createDirectoryRecord(string);
        String string2 = file.getName();
        Record record2 = this.waveletTable.createRecord();
        ((WaveletRecord)record2).filename = string2;
        ((WaveletRecord)record2).directorySecondaryKey = record != null ? record.getKey() : -1L;
        Record record3 = this.rpfFileTable.getRecord(l);
        if (record3 != null) {
            ((RPFFileRecord)record3).waveletSecondaryKey = record2.getKey();
            ((WaveletRecord)record2).rpfFileSecondaryKey = record3.getKey();
        }
        return record2;
    }

    private synchronized Record createDirectoryRecord(String string) {
        Record record = null;
        if (string != null) {
            for (Record record2 : this.directoryTable.getRecords()) {
                if (!((DirectoryRecord)record2).path.equals(string)) continue;
                record = record2;
                break;
            }
            if (record == null) {
                record = this.directoryTable.createRecord();
                ((DirectoryRecord)record).path = string;
            }
        }
        return record;
    }

    public void updateBoundingSector() {
        Sector sector = null;
        for (Record record : this.rpfFileTable.getRecords()) {
            RPFFileRecord rPFFileRecord = (RPFFileRecord)record;
            Sector sector2 = rPFFileRecord.getSector();
            if (sector2 == null) continue;
            sector = sector != null ? sector.union(sector2) : sector2;
        }
        if (sector != null) {
            sector = Sector.fromDegrees(RPFFileIndex.clamp(sector.getMinLatitude().degrees, -90.0, 90.0), RPFFileIndex.clamp(sector.getMaxLatitude().degrees, -90.0, 90.0), RPFFileIndex.clamp(sector.getMinLongitude().degrees, -180.0, 180.0), RPFFileIndex.clamp(sector.getMaxLongitude().degrees, -180.0, 180.0));
        }
        this.properties.setBoundingSector(sector);
    }

    private static double clamp(double d, double d2, double d3) {
        return d < d2 ? d2 : (d > d3 ? d3 : d);
    }

    public void load(ByteBuffer byteBuffer) throws IOException {
        if (byteBuffer == null) {
            String string = Logging.getMessage("nullValue.ByteBufferIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        String string = RPFFileIndex.getString(byteBuffer, 16);
        if (!FILE_ID.equals(string)) {
            String string2 = "buffer does not contain an RPFFileIndex";
            Logging.logger().severe(string2);
            throw new IOException(string2);
        }
        String string3 = RPFFileIndex.getString(byteBuffer, 16);
        LocationSection locationSection = new LocationSection(byteBuffer);
        this.properties.load(byteBuffer, locationSection.getInformationSectionLocation());
        this.rpfFileTable.load(byteBuffer, locationSection.getRPFFileTableSectionLocation());
        this.waveletTable.load(byteBuffer, locationSection.getWaveletTableSectionLocation());
        this.directoryTable.load(byteBuffer, locationSection.getDirectoryTableSectionLocation());
    }

    public ByteBuffer save() throws IOException {
        ByteBuffer byteBuffer = this.properties.save();
        ByteBuffer byteBuffer2 = this.rpfFileTable.save();
        ByteBuffer byteBuffer3 = this.waveletTable.save();
        ByteBuffer byteBuffer4 = this.directoryTable.save();
        int n = 32;
        LocationSection locationSection = new LocationSection();
        locationSection.setInformationSection(byteBuffer.limit(), n += locationSection.locationSectionLength);
        locationSection.setRPFFileTableSection(byteBuffer2.limit(), n += byteBuffer.limit());
        locationSection.setWaveletTableSection(byteBuffer3.limit(), n += byteBuffer2.limit());
        locationSection.setDirectoryTableSection(byteBuffer4.limit(), n += byteBuffer3.limit());
        n += byteBuffer4.limit();
        ByteBuffer byteBuffer5 = locationSection.save();
        int n2 = 32 + byteBuffer5.limit() + byteBuffer.limit() + byteBuffer2.limit() + byteBuffer3.limit() + byteBuffer4.limit();
        ByteBuffer byteBuffer6 = ByteBuffer.allocate(n2);
        RPFFileIndex.putString(byteBuffer6, FILE_ID, 16);
        RPFFileIndex.putString(byteBuffer6, VERSION, 16);
        byteBuffer6.put(byteBuffer5);
        byteBuffer6.put(byteBuffer);
        byteBuffer6.put(byteBuffer2);
        byteBuffer6.put(byteBuffer3);
        byteBuffer6.put(byteBuffer4);
        byteBuffer6.flip();
        return byteBuffer6;
    }

    private static String getString(ByteBuffer byteBuffer, int n) throws IOException {
        String string = null;
        if (byteBuffer != null && byteBuffer.remaining() >= n) {
            byte[] byArray = new byte[n];
            byteBuffer.get(byArray, 0, n);
            string = new String(byArray, CHARACTER_ENCODING).trim();
        }
        return string;
    }

    private static void putString(ByteBuffer byteBuffer, String string, int n) throws IOException {
        if (byteBuffer != null) {
            byte[] byArray;
            byte[] byArray2 = new byte[n];
            if (string != null && (byArray = string.getBytes(CHARACTER_ENCODING)) != null) {
                System.arraycopy(byArray, 0, byArray2, 0, byArray.length);
            }
            byteBuffer.put(byArray2, 0, n);
        }
    }

    private static Angle getAngle(ByteBuffer byteBuffer) {
        double d;
        Angle angle = null;
        if (byteBuffer != null && !Double.isNaN(d = byteBuffer.getDouble())) {
            angle = Angle.fromDegrees(d);
        }
        return angle;
    }

    private static void putAngle(ByteBuffer byteBuffer, Angle angle) {
        if (byteBuffer != null) {
            double d = angle != null ? angle.degrees : Double.NaN;
            byteBuffer.putDouble(d);
        }
    }

    private static class LocationSection {
        public int locationSectionLength;
        public int componentLocationTableOffset;
        public int numOfComponentLocationRecords;
        private Map<Integer, ComponentLocationRecord> table = new HashMap<Integer, ComponentLocationRecord>();

        public LocationSection() {
            for (int i = 1; i <= 4; ++i) {
                this.table.put(i, new ComponentLocationRecord(i, -1, -1));
            }
            this.locationSectionLength = 12 + this.table.size() * 3 * 32 / 8;
            this.componentLocationTableOffset = 12;
            this.numOfComponentLocationRecords = this.table.size();
        }

        public LocationSection(ByteBuffer byteBuffer) throws IOException {
            int n = byteBuffer.position();
            this.locationSectionLength = byteBuffer.getInt();
            this.componentLocationTableOffset = byteBuffer.getInt();
            this.numOfComponentLocationRecords = byteBuffer.getInt();
            byteBuffer.position(n + this.componentLocationTableOffset);
            for (int i = 0; i < this.numOfComponentLocationRecords; ++i) {
                int n2 = byteBuffer.getInt();
                this.table.put(n2, new ComponentLocationRecord(n2, byteBuffer.getInt(), byteBuffer.getInt()));
            }
            byteBuffer.position(n);
        }

        public ByteBuffer save() throws IOException {
            ByteBuffer byteBuffer = ByteBuffer.allocate(this.locationSectionLength);
            byteBuffer.putInt(this.locationSectionLength);
            byteBuffer.putInt(this.componentLocationTableOffset);
            byteBuffer.putInt(this.numOfComponentLocationRecords);
            Collection<ComponentLocationRecord> collection = this.table.values();
            ComponentLocationRecord[] componentLocationRecordArray = new ComponentLocationRecord[collection.size()];
            collection.toArray(componentLocationRecordArray);
            byteBuffer.position(this.componentLocationTableOffset);
            for (int i = 0; i < this.numOfComponentLocationRecords; ++i) {
                ComponentLocationRecord componentLocationRecord = componentLocationRecordArray[i];
                byteBuffer.putInt(componentLocationRecord.getId());
                byteBuffer.putInt(componentLocationRecord.getLength());
                byteBuffer.putInt(componentLocationRecord.getLocation());
            }
            byteBuffer.flip();
            return byteBuffer;
        }

        public int getInformationSectionLocation() {
            return this.getLocation(1);
        }

        public int getInformationSectionLength() {
            return this.getLength(1);
        }

        public void setInformationSection(int n, int n2) {
            this.set(1, n, n2);
        }

        public int getRPFFileTableSectionLocation() {
            return this.getLocation(2);
        }

        public int getRPFFileTableSectionLength() {
            return this.getLength(2);
        }

        public void setRPFFileTableSection(int n, int n2) {
            this.set(2, n, n2);
        }

        public int getWaveletTableSectionLocation() {
            return this.getLocation(3);
        }

        public int getWaveletTableSectionLength() {
            return this.getLength(3);
        }

        public void setWaveletTableSection(int n, int n2) {
            this.set(3, n, n2);
        }

        public int getDirectoryTableSectionLocation() {
            return this.getLocation(4);
        }

        public int getDirectoryTableSectionLength() {
            return this.getLength(4);
        }

        public void setDirectoryTableSection(int n, int n2) {
            this.set(4, n, n2);
        }

        private int getLocation(int n) {
            ComponentLocationRecord componentLocationRecord = this.getRecord(n);
            return null != componentLocationRecord ? componentLocationRecord.getLocation() : 0;
        }

        private int getLength(int n) {
            ComponentLocationRecord componentLocationRecord = this.getRecord(n);
            return null != componentLocationRecord ? componentLocationRecord.getLength() : 0;
        }

        private void set(int n, int n2, int n3) {
            ComponentLocationRecord componentLocationRecord = this.getRecord(n);
            if (componentLocationRecord != null) {
                componentLocationRecord.length = n2;
                componentLocationRecord.location = n3;
            }
        }

        private ComponentLocationRecord getRecord(int n) {
            if (this.table.containsKey(n)) {
                return this.table.get(n);
            }
            return null;
        }

        public static class ComponentLocationRecord {
            private int id;
            private int length;
            private int location;

            public ComponentLocationRecord(int n, int n2, int n3) {
                this.id = n;
                this.length = n2;
                this.location = n3;
            }

            public int getId() {
                return this.id;
            }

            public int getLength() {
                return this.length;
            }

            public int getLocation() {
                return this.location;
            }
        }
    }

    public static class IndexProperties {
        public String rootPath;
        public String dataSeriesIdentifier;
        public String description;
        public Angle minLatitude;
        public Angle maxLatitude;
        public Angle minLongitude;
        public Angle maxLongitude;
        private static int ROOT_PATH_LENGTH = 512;
        private static int DATA_SERIES_ID_LENGTH = 512;
        private static int DESCRIPTION_LENGTH = 4096;
        private static int SIZE = 32 + ROOT_PATH_LENGTH * 8 + DATA_SERIES_ID_LENGTH * 8 + DESCRIPTION_LENGTH * 8 + 256;

        public String getRootPath() {
            return this.rootPath;
        }

        public void setRootPath(String string) {
            this.rootPath = string;
        }

        public String getDataSeriesIdentifier() {
            return this.dataSeriesIdentifier;
        }

        public void setDataSeriesIdentifier(String string) {
            this.dataSeriesIdentifier = string;
        }

        public String getDescription() {
            return this.description;
        }

        public void setDescription(String string) {
            this.description = string;
        }

        public Sector getBoundingSector() {
            Sector sector = null;
            if (this.minLatitude != null && this.maxLatitude != null && this.minLongitude != null && this.maxLongitude != null) {
                sector = new Sector(this.minLatitude, this.maxLatitude, this.minLongitude, this.maxLongitude);
            }
            return sector;
        }

        public void setBoundingSector(Sector sector) {
            this.minLatitude = sector != null ? sector.getMinLatitude() : null;
            this.maxLatitude = sector != null ? sector.getMaxLatitude() : null;
            this.minLongitude = sector != null ? sector.getMinLongitude() : null;
            this.maxLongitude = sector != null ? sector.getMaxLongitude() : null;
        }

        void load(ByteBuffer byteBuffer, int n) throws IOException {
            if (byteBuffer == null) {
                String string = Logging.getMessage("nullValue.ByteBufferIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            int n2 = byteBuffer.position();
            byteBuffer.position(n);
            int n3 = byteBuffer.getInt();
            this.rootPath = RPFFileIndex.getString(byteBuffer, IndexProperties.ROOT_PATH_LENGTH);
            this.dataSeriesIdentifier = RPFFileIndex.getString(byteBuffer, IndexProperties.DATA_SERIES_ID_LENGTH);
            this.description = RPFFileIndex.getString(byteBuffer, IndexProperties.DESCRIPTION_LENGTH);
            this.minLatitude = RPFFileIndex.getAngle(byteBuffer);
            this.maxLatitude = RPFFileIndex.getAngle(byteBuffer);
            this.minLongitude = RPFFileIndex.getAngle(byteBuffer);
            this.maxLongitude = RPFFileIndex.getAngle(byteBuffer);
            byteBuffer.position(n2);
        }

        ByteBuffer save() throws IOException {
            int n = SIZE / 8;
            ByteBuffer byteBuffer = ByteBuffer.allocate(n);
            byteBuffer.putInt(n);
            RPFFileIndex.putString(byteBuffer, this.rootPath, IndexProperties.ROOT_PATH_LENGTH);
            RPFFileIndex.putString(byteBuffer, this.dataSeriesIdentifier, IndexProperties.DATA_SERIES_ID_LENGTH);
            RPFFileIndex.putString(byteBuffer, this.description, IndexProperties.DESCRIPTION_LENGTH);
            RPFFileIndex.putAngle(byteBuffer, this.minLatitude);
            RPFFileIndex.putAngle(byteBuffer, this.maxLatitude);
            RPFFileIndex.putAngle(byteBuffer, this.minLongitude);
            RPFFileIndex.putAngle(byteBuffer, this.maxLongitude);
            byteBuffer.flip();
            return byteBuffer;
        }
    }

    public static class DirectoryRecord
    extends Record {
        private String path;
        private static int PATH_LENGTH = 512;
        private static int SIZE = PATH_LENGTH * 8;

        public DirectoryRecord(long l) {
            super(l);
        }

        public String getPath() {
            return this.path;
        }

        public void setPath(String string) {
            if (string == null) {
                String string2 = Logging.getMessage("nullValue.StringIsNull");
                Logging.logger().severe(string2);
                throw new IllegalArgumentException(string2);
            }
            this.path = string;
        }

        @Override
        void load(ByteBuffer byteBuffer) throws IOException {
            if (byteBuffer == null) {
                String string = Logging.getMessage("nullValue.ByteBufferIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            this.path = RPFFileIndex.getString(byteBuffer, DirectoryRecord.PATH_LENGTH);
        }

        @Override
        void save(ByteBuffer byteBuffer) throws IOException {
            if (byteBuffer == null) {
                String string = Logging.getMessage("nullValue.ByteBufferIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            RPFFileIndex.putString(byteBuffer, this.path, DirectoryRecord.PATH_LENGTH);
        }

        @Override
        int getSizeInBits() {
            return SIZE + super.getSizeInBits();
        }
    }

    public static class WaveletRecord
    extends Record {
        private String filename;
        private long directorySecondaryKey = -1L;
        private long rpfFileSecondaryKey = -1L;
        private static int FILENAME_LENGTH = 16;
        private static int SIZE = FILENAME_LENGTH * 8 + 64 + 64;

        public WaveletRecord(long l) {
            super(l);
        }

        public String getFilename() {
            return this.filename;
        }

        public void setFilename(String string) {
            this.filename = string;
        }

        public long getDirectorySecondaryKey() {
            return this.directorySecondaryKey;
        }

        public long getRPFFileSecondaryKey() {
            return this.rpfFileSecondaryKey;
        }

        @Override
        void load(ByteBuffer byteBuffer) throws IOException {
            if (byteBuffer == null) {
                String string = Logging.getMessage("nullValue.ByteBufferIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            this.filename = RPFFileIndex.getString(byteBuffer, WaveletRecord.FILENAME_LENGTH);
            this.directorySecondaryKey = byteBuffer.getLong();
            this.rpfFileSecondaryKey = byteBuffer.getLong();
        }

        @Override
        void save(ByteBuffer byteBuffer) throws IOException {
            if (byteBuffer == null) {
                String string = Logging.getMessage("nullValue.ByteBufferIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            RPFFileIndex.putString(byteBuffer, this.filename, WaveletRecord.FILENAME_LENGTH);
            byteBuffer.putLong(this.directorySecondaryKey);
            byteBuffer.putLong(this.rpfFileSecondaryKey);
        }

        @Override
        int getSizeInBits() {
            return SIZE + super.getSizeInBits();
        }
    }

    public static class RPFFileRecord
    extends Record {
        private String filename;
        private long directorySecondaryKey = -1L;
        private long waveletSecondaryKey = -1L;
        private Angle minLatitude;
        private Angle maxLatitude;
        private Angle minLongitude;
        private Angle maxLongitude;
        private static int FILENAME_LENGTH = 12;
        private static int SIZE = FILENAME_LENGTH * 8 + 64 + 64 + 256;

        public RPFFileRecord(long l) {
            super(l);
        }

        public String getFilename() {
            return this.filename;
        }

        public void setFilename(String string) {
            this.filename = string;
        }

        public long getDirectorySecondaryKey() {
            return this.directorySecondaryKey;
        }

        public long getWaveletSecondaryKey() {
            return this.waveletSecondaryKey;
        }

        public Sector getSector() {
            Sector sector = null;
            if (this.minLatitude != null && this.maxLatitude != null && this.minLongitude != null && this.maxLongitude != null) {
                sector = new Sector(this.minLatitude, this.maxLatitude, this.minLongitude, this.maxLongitude);
            }
            return sector;
        }

        public void setSector(Sector sector) {
            this.minLatitude = sector != null ? sector.getMinLatitude() : null;
            this.maxLatitude = sector != null ? sector.getMaxLatitude() : null;
            this.minLongitude = sector != null ? sector.getMinLongitude() : null;
            this.maxLongitude = sector != null ? sector.getMaxLongitude() : null;
        }

        @Override
        void load(ByteBuffer byteBuffer) throws IOException {
            if (byteBuffer == null) {
                String string = Logging.getMessage("nullValue.ByteBufferIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            this.filename = RPFFileIndex.getString(byteBuffer, RPFFileRecord.FILENAME_LENGTH);
            this.directorySecondaryKey = byteBuffer.getLong();
            this.waveletSecondaryKey = byteBuffer.getLong();
            this.minLatitude = RPFFileIndex.getAngle(byteBuffer);
            this.maxLatitude = RPFFileIndex.getAngle(byteBuffer);
            this.minLongitude = RPFFileIndex.getAngle(byteBuffer);
            this.maxLongitude = RPFFileIndex.getAngle(byteBuffer);
        }

        @Override
        void save(ByteBuffer byteBuffer) throws IOException {
            if (byteBuffer == null) {
                String string = Logging.getMessage("nullValue.ByteBufferIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            RPFFileIndex.putString(byteBuffer, this.filename, RPFFileRecord.FILENAME_LENGTH);
            byteBuffer.putLong(this.directorySecondaryKey);
            byteBuffer.putLong(this.waveletSecondaryKey);
            RPFFileIndex.putAngle(byteBuffer, this.minLatitude);
            RPFFileIndex.putAngle(byteBuffer, this.maxLatitude);
            RPFFileIndex.putAngle(byteBuffer, this.minLongitude);
            RPFFileIndex.putAngle(byteBuffer, this.maxLongitude);
        }

        @Override
        int getSizeInBits() {
            return SIZE + super.getSizeInBits();
        }
    }

    private static class DefaultRecordFactory
    implements RecordFactory {
        private DefaultRecordFactory() {
        }

        @Override
        public Record newRecord(long l) {
            return new Record(l);
        }
    }

    public static interface RecordFactory {
        public Record newRecord(long var1);
    }

    public static class Record {
        private final long key;

        public Record(long l) {
            this.key = l;
        }

        public final long getKey() {
            return this.key;
        }

        void load(ByteBuffer byteBuffer) throws IOException {
        }

        void save(ByteBuffer byteBuffer) throws IOException {
        }

        int getSizeInBits() {
            return 0;
        }
    }

    public static class Table {
        private final List<Record> records = new ArrayList<Record>();
        private final Map<Long, Record> keyIndex = new HashMap<Long, Record>();
        private RecordFactory recordFactory = new DefaultRecordFactory();
        private volatile long uniqueKey = -1L;
        static final long INVALID_KEY = -1L;

        public final List<Record> getRecords() {
            return this.records;
        }

        public final Record getRecord(long l) {
            Record record = null;
            if (l != -1L) {
                record = this.keyIndex.get(l);
            }
            return record;
        }

        public RecordFactory getRecordFactory() {
            return this.recordFactory;
        }

        public void setRecordFactory(RecordFactory recordFactory) {
            if (recordFactory == null) {
                String string = "RecordFactory is null";
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            this.recordFactory = recordFactory;
        }

        public Record createRecord() {
            long l = this.createUniqueKey();
            return this.newRecord(l);
        }

        Record newRecord(long l) {
            Record record = this.recordFactory.newRecord(l);
            this.putRecord(l, record);
            return record;
        }

        private void putRecord(long l, Record record) {
            this.records.add(record);
            this.keyIndex.put(l, record);
        }

        private synchronized long createUniqueKey() {
            return ++this.uniqueKey;
        }

        void load(ByteBuffer byteBuffer, int n) throws IOException {
            if (byteBuffer == null) {
                String string = Logging.getMessage("nullValue.ByteBufferIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            this.records.clear();
            this.keyIndex.clear();
            int n2 = byteBuffer.position();
            byteBuffer.position(n);
            int n3 = byteBuffer.getInt();
            int n4 = byteBuffer.getInt();
            int n5 = byteBuffer.getInt();
            byteBuffer.position(n + n4);
            for (int i = 0; i < n5; ++i) {
                long l = byteBuffer.getLong();
                Record record = this.newRecord(l);
                record.load(byteBuffer);
            }
            byteBuffer.position(n2);
        }

        ByteBuffer save() throws IOException {
            int n;
            List<Record> list = this.getRecords();
            int n2 = list.size();
            int n3 = n = 12;
            for (Record object : list) {
                n3 += (64 + object.getSizeInBits()) / 8;
            }
            ByteBuffer byteBuffer = ByteBuffer.allocate(n3);
            byteBuffer.putInt(n3);
            byteBuffer.putInt(n);
            byteBuffer.putInt(n2);
            for (Record record : list) {
                byteBuffer.putLong(record.getKey());
                record.save(byteBuffer);
            }
            byteBuffer.flip();
            return byteBuffer;
        }
    }
}

