/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.statistics.tools;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.core.gpf.main.GPT;
import org.esa.snap.core.util.FeatureUtils;
import org.esa.snap.core.util.SystemUtils;
import org.esa.snap.core.util.io.FileUtils;
import org.esa.snap.statistics.tools.DatabaseRecord;
import org.esa.snap.statistics.tools.FilenameDateExtractor;
import org.esa.snap.statistics.tools.ObservationYear;
import org.esa.snap.statistics.tools.ParameterName;
import org.esa.snap.statistics.tools.StatisticsDatabase;
import org.geotools.feature.FeatureCollection;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;

public class SummaryCSVTool {
    private static final String EXE_NAME = "SummaryCSVTool";
    private static final String EXE_VERSION = "1.0";
    private static final String FILENAME_PATTERN_SHAPEFILE = "yyyyMMdd_*.shp";
    private final Logger logger;
    private final FilenameDateExtractor filenameDateExtractor;
    private final StatisticsDatabase statisticsDatabase;
    private final String TAB = "\t";
    private ShapeFileReader shapeFileReader;

    public static void main(String[] args) throws IOException {
        File outputDir;
        Options options = SummaryCSVTool.createOptions();
        GnuParser parser = new GnuParser();
        CommandLine commandLine = null;
        try {
            commandLine = parser.parse(options, args);
        }
        catch (ParseException e) {
            SummaryCSVTool.printHelp(options);
            System.exit(-1);
        }
        File inputDir = new File(commandLine.getOptionValue("input"));
        SummaryCSVTool.ensureDirectory(inputDir);
        if (commandLine.hasOption("output")) {
            outputDir = new File(commandLine.getOptionValue("output"));
            SummaryCSVTool.ensureDirectory(outputDir);
        } else {
            outputDir = inputDir;
        }
        SummaryCSVTool.initSystem();
        Logger logger = SystemUtils.LOG;
        ShapeFileReader shapeFileReader = FeatureUtils::loadFeatureCollectionFromShapefile;
        String waterbodyColumnName = "NAME";
        if (commandLine.hasOption("waterbodyNameColumn")) {
            waterbodyColumnName = commandLine.getOptionValue("waterbodyNameColumn");
        }
        SummaryCSVTool summaryCSVTool = new SummaryCSVTool(logger, shapeFileReader, waterbodyColumnName);
        summaryCSVTool.summarize(inputDir);
        summaryCSVTool.putOutSummerizedData(outputDir);
    }

    private static void ensureDirectory(File directory) throws IOException {
        if (!directory.isDirectory()) {
            throw new IOException("'" + directory.getAbsolutePath() + "' is not a directory");
        }
    }

    private static Options createOptions() {
        Options options = new Options();
        options.addOption(SummaryCSVTool.createOption("i", "input", "FILE", "The directory where the shapefiles reside.", true));
        options.addOption(SummaryCSVTool.createOption("o", "output", "FILE", "The output directory. If not provided, output will be written to input directory.", false));
        options.addOption(SummaryCSVTool.createOption("n", "waterbodyNameColumn", "STRING", "The name of the column that contains the waterbody name.", false));
        return options;
    }

    private static Option createOption(String shortOpt, String longOpt, String argName, String description, boolean required) {
        Option from = new Option(shortOpt, longOpt, argName != null, description);
        from.setRequired(required);
        from.setArgName(argName);
        return from;
    }

    private static void initSystem() {
        if (System.getProperty("snap.context") == null) {
            System.setProperty("snap.context", "snap");
        }
        Locale.setDefault(Locale.ENGLISH);
        SystemUtils.init3rdPartyLibs(GPT.class);
    }

    public SummaryCSVTool(Logger logger, ShapeFileReader shapeFileReader, String nameColumn) {
        this.logger = logger;
        this.shapeFileReader = shapeFileReader;
        this.filenameDateExtractor = new FilenameDateExtractor();
        this.statisticsDatabase = new StatisticsDatabase(nameColumn);
    }

    private static void printHelp(Options options) {
        HelpFormatter helpFormatter = new HelpFormatter();
        helpFormatter.setWidth(120);
        System.out.println(SummaryCSVTool.getHeader());
        helpFormatter.printHelp(EXE_NAME, options, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void putOutSummerizedData(File outputDir) throws IOException {
        ObservationYear[] years;
        for (ObservationYear year : years = this.statisticsDatabase.getYears()) {
            ParameterName[] parameterNames;
            for (ParameterName parameterName : parameterNames = this.statisticsDatabase.getParameterNames(year)) {
                File outputFile = new File(outputDir, "WFD_stat_" + year + "_" + parameterName + ".txt");
                FileOutputStream fileOutputStream = null;
                PrintWriter writer = null;
                try {
                    fileOutputStream = new FileOutputStream(outputFile);
                    writer = new PrintWriter(fileOutputStream);
                    DatabaseRecord[] databaseRecords = this.statisticsDatabase.getData(year, parameterName);
                    this.printHeader(databaseRecords, writer, parameterName);
                    this.printData(databaseRecords, writer);
                }
                finally {
                    if (writer != null) {
                        writer.close();
                    }
                    if (fileOutputStream != null) {
                        fileOutputStream.close();
                    }
                }
            }
        }
    }

    private void printHeader(DatabaseRecord[] records, PrintWriter pw, ParameterName parameterName) {
        pw.print("Bod.ID");
        pw.print("\t");
        pw.print("Bod.Name");
        TreeMap<Date, TreeSet> dateColNamesMap = new TreeMap<Date, TreeSet>();
        for (DatabaseRecord record : records) {
            Set<Date> dataDates = record.getDataDates();
            for (Date date : dataDates) {
                TreeSet colNames;
                if (dateColNamesMap.containsKey(date)) {
                    colNames = (TreeSet)dateColNamesMap.get(date);
                } else {
                    colNames = new TreeSet();
                    dateColNamesMap.put(date, colNames);
                }
                Set<String> statDataColumns = record.getStatDataColumns(date);
                colNames.addAll(statDataColumns);
            }
        }
        Calendar calendar = Calendar.getInstance();
        for (Map.Entry dateColNamesEntry : dateColNamesMap.entrySet()) {
            Date date = (Date)dateColNamesEntry.getKey();
            calendar.setTime(date);
            String year = Integer.toString(calendar.get(1));
            String month = Integer.toString(calendar.get(2) + 1);
            String day = Integer.toString(calendar.get(5));
            String monthPart = month.length() == 1 ? "0" + month : month;
            String dayPart = day.length() == 1 ? "0" + day : day;
            String datePart = "" + year + monthPart + dayPart;
            TreeSet colNames = (TreeSet)dateColNamesEntry.getValue();
            for (String colName : colNames) {
                pw.print("\t");
                pw.print(parameterName + "_" + colName + "_" + datePart);
            }
        }
        pw.println();
    }

    private void printData(DatabaseRecord[] databaseRecords, PrintWriter writer) {
        TreeMap<Date, Integer> columnsPerDay = new TreeMap<Date, Integer>();
        for (DatabaseRecord record : databaseRecords) {
            Set<Date> dataDates = record.getDataDates();
            for (Date dataDate : dataDates) {
                Set<String> statDataColumns = record.getStatDataColumns(dataDate);
                if (columnsPerDay.containsKey(dataDate)) {
                    Integer numCols = (Integer)columnsPerDay.get(dataDate);
                    columnsPerDay.put(dataDate, Math.max(statDataColumns.size(), numCols));
                    continue;
                }
                columnsPerDay.put(dataDate, statDataColumns.size());
            }
        }
        for (DatabaseRecord record : databaseRecords) {
            this.print(record, writer, columnsPerDay);
        }
    }

    private void print(DatabaseRecord record, PrintWriter pw, Map<Date, Integer> columnsPerDay) {
        pw.print(record.geomId);
        pw.print("\t");
        pw.print(record.geomName);
        for (Date date : columnsPerDay.keySet()) {
            Set<String> statDataColumns = record.getStatDataColumns(date);
            if (statDataColumns != null) {
                for (String statDataColumn : statDataColumns) {
                    pw.print("\t");
                    pw.print(record.getValue(date, statDataColumn));
                }
                continue;
            }
            Integer numCols = columnsPerDay.get(date);
            for (int i = 0; i < numCols; ++i) {
                pw.print("\t");
            }
        }
        pw.println();
    }

    void summarize(File inputDir) {
        File[] shapeFiles;
        for (File shapeFile : shapeFiles = inputDir.listFiles((dir, name) -> name.toLowerCase().endsWith(".shp"))) {
            File mappingFile;
            if (!this.isValidDateExtractableFileName(shapeFile) || !(mappingFile = new File(inputDir, FileUtils.getFilenameWithoutExtension((File)shapeFile).concat("_band_mapping.txt"))).isFile()) continue;
            try (FileReader fileReader = new FileReader(mappingFile);){
                ProductData.UTC date = this.filenameDateExtractor.getDate(shapeFile);
                FeatureCollection<SimpleFeatureType, SimpleFeature> featureCollection = this.shapeFileReader.read(shapeFile);
                Properties properties = new Properties();
                properties.load(fileReader);
                this.statisticsDatabase.append(date, featureCollection, properties);
            }
            catch (IOException e) {
                this.logger.log(Level.WARNING, e.getMessage());
                e.printStackTrace();
            }
        }
    }

    private boolean isValidDateExtractableFileName(File shapeFile) {
        boolean validFilename = this.filenameDateExtractor.isValidFilename(shapeFile);
        if (!validFilename) {
            this.logger.log(Level.WARNING, "The filename '" + shapeFile.getName() + "' does not match the pattern " + FILENAME_PATTERN_SHAPEFILE + ".");
            this.logger.log(Level.INFO, "Continuing with next ESRI shapefile.");
        }
        return validFilename;
    }

    private static String getHeader() {
        StringWriter stringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stringWriter);
        printWriter.println("SummaryCSVTool version 1.0");
        printWriter.println("    The tool reads statistical data from ESRI shapefiles and summarizes it in ");
        printWriter.println("    *.csv-files. One csv file will be generated per year and parameter.");
        printWriter.println("    An example for a file name is WFD_stat_2009_CHL.csv. ");
        printWriter.close();
        return stringWriter.toString();
    }

    public static interface ShapeFileReader {
        public FeatureCollection<SimpleFeatureType, SimpleFeature> read(File var1) throws IOException;
    }
}

