/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.core.datamodel;

import java.awt.Dimension;
import java.awt.Rectangle;
import org.esa.snap.core.dataio.ProductSubsetDef;
import org.esa.snap.core.datamodel.AbstractGeoCoding;
import org.esa.snap.core.datamodel.GeoPos;
import org.esa.snap.core.datamodel.PixelPos;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.Scene;
import org.esa.snap.core.datamodel.TiePointGrid;
import org.esa.snap.core.dataop.maptransf.Datum;
import org.esa.snap.core.util.Debug;
import org.esa.snap.core.util.Guardian;
import org.esa.snap.core.util.math.FXYSum;
import org.esa.snap.core.util.math.MathUtils;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

public class TiePointGeoCoding
extends AbstractGeoCoding {
    private static final double ABS_ERROR_LIMIT = 0.5;
    private static final int MAX_NUM_POINTS_PER_TILE = 1000;
    private final TiePointGrid latGrid;
    private final TiePointGrid lonGrid;
    private final Datum datum;
    private boolean approximationsComputed;
    private boolean normalized;
    private double normalizedLonMin;
    private double normalizedLonMax;
    private double latMin;
    private double latMax;
    private double overlapStart;
    private double overlapEnd;
    private Approximation[] approximations;

    public TiePointGeoCoding(TiePointGrid latGrid, TiePointGrid lonGrid) {
        this(latGrid, lonGrid, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84);
    }

    public TiePointGeoCoding(TiePointGrid latGrid, TiePointGrid lonGrid, CoordinateReferenceSystem geoCRS) {
        super(geoCRS);
        Guardian.assertNotNull("latGrid", latGrid);
        Guardian.assertNotNull("lonGrid", lonGrid);
        Guardian.assertNotNull("geoCRS", geoCRS);
        if (latGrid.getGridWidth() != lonGrid.getGridWidth() || latGrid.getGridHeight() != lonGrid.getGridHeight() || latGrid.getOffsetX() != lonGrid.getOffsetX() || latGrid.getOffsetY() != lonGrid.getOffsetY() || latGrid.getSubSamplingX() != lonGrid.getSubSamplingX() || latGrid.getSubSamplingY() != lonGrid.getSubSamplingY()) {
            throw new IllegalArgumentException("latGrid is not compatible with lonGrid");
        }
        this.latGrid = latGrid;
        this.lonGrid = lonGrid;
        this.datum = Datum.WGS_84;
        this.approximationsComputed = false;
    }

    @Deprecated
    public TiePointGeoCoding(TiePointGrid latGrid, TiePointGrid lonGrid, Datum datum) {
        Guardian.assertNotNull("latGrid", latGrid);
        Guardian.assertNotNull("lonGrid", lonGrid);
        Guardian.assertNotNull("datum", datum);
        if (latGrid.getGridWidth() != lonGrid.getGridWidth() || latGrid.getGridHeight() != lonGrid.getGridHeight() || latGrid.getOffsetX() != lonGrid.getOffsetX() || latGrid.getOffsetY() != lonGrid.getOffsetY() || latGrid.getSubSamplingX() != lonGrid.getSubSamplingX() || latGrid.getSubSamplingY() != lonGrid.getSubSamplingY()) {
            throw new IllegalArgumentException("latGrid is not compatible with lonGrid");
        }
        this.latGrid = latGrid;
        this.lonGrid = lonGrid;
        this.datum = datum;
        this.approximationsComputed = false;
    }

    private synchronized void computeApproximations() {
        if (!this.approximationsComputed) {
            TiePointGrid normalizedLonGrid = this.initNormalizedLonGrid();
            this.initLatLonMinMax(normalizedLonGrid);
            this.approximations = this.initApproximations(normalizedLonGrid);
            this.approximationsComputed = true;
        }
    }

    @Override
    public Datum getDatum() {
        return this.datum;
    }

    @Override
    public boolean isCrossingMeridianAt180() {
        return this.normalized;
    }

    public int getNumApproximations() {
        if (!this.approximationsComputed) {
            this.computeApproximations();
        }
        return this.approximations != null ? this.approximations.length : 0;
    }

    public Approximation getApproximation(int index) {
        return this.approximations[index];
    }

    @Override
    public boolean canGetGeoPos() {
        return true;
    }

    @Override
    public boolean canGetPixelPos() {
        if (!this.approximationsComputed) {
            this.computeApproximations();
        }
        return this.approximations != null;
    }

    public TiePointGrid getLatGrid() {
        return this.latGrid;
    }

    public TiePointGrid getLonGrid() {
        return this.lonGrid;
    }

    @Override
    public GeoPos getGeoPos(PixelPos pixelPos, GeoPos geoPos) {
        if (geoPos == null) {
            geoPos = new GeoPos();
        }
        if (pixelPos.x < 0.0 || pixelPos.x > (double)this.latGrid.getRasterWidth() || pixelPos.y < 0.0 || pixelPos.y > (double)this.latGrid.getRasterHeight()) {
            geoPos.setInvalid();
        } else {
            geoPos.lat = this.latGrid.getPixelDouble(pixelPos.x, pixelPos.y);
            geoPos.lon = this.lonGrid.getPixelDouble(pixelPos.x, pixelPos.y);
        }
        return geoPos;
    }

    @Override
    public PixelPos getPixelPos(GeoPos geoPos, PixelPos pixelPos) {
        if (!this.approximationsComputed) {
            this.computeApproximations();
        }
        if (this.approximations != null) {
            double lat = TiePointGeoCoding.normalizeLat(geoPos.lat);
            double lon = this.normalizeLon(geoPos.lon);
            if (pixelPos == null) {
                pixelPos = new PixelPos();
            }
            if (this.isValidGeoPos(lat, lon)) {
                double squareDistance;
                double tempLon;
                Approximation renormalizedApproximation;
                Approximation approximation = TiePointGeoCoding.getBestApproximation(this.approximations, lat, lon);
                if (lon >= this.overlapStart && lon <= this.overlapEnd && (renormalizedApproximation = this.findRenormalizedApproximation(lat, tempLon = lon + 360.0, squareDistance = approximation != null ? approximation.getSquareDistance(lat, lon) : Double.MAX_VALUE)) != null) {
                    approximation = renormalizedApproximation;
                    lon = tempLon;
                }
                if (approximation != null) {
                    lat = TiePointGeoCoding.rescaleLatitude(lat);
                    lon = TiePointGeoCoding.rescaleLongitude(lon, approximation.getCenterLon());
                    pixelPos.x = approximation.getFX().computeZ(lat, lon);
                    pixelPos.y = approximation.getFY().computeZ(lat, lon);
                } else {
                    pixelPos.setInvalid();
                }
            } else {
                pixelPos.setInvalid();
            }
        }
        return pixelPos;
    }

    private boolean isValidGeoPos(double lat, double lon) {
        return !Double.isNaN(lat) && !Double.isNaN(lon);
    }

    public static double normalizeLat(double lat) {
        if (lat < -90.0 || lat > 90.0) {
            return Double.NaN;
        }
        return lat;
    }

    public final double normalizeLon(double lon) {
        if (lon < -180.0 || lon > 180.0) {
            return Double.NaN;
        }
        double normalizedLon = lon;
        if (normalizedLon < this.normalizedLonMin) {
            normalizedLon += 360.0;
        }
        if (normalizedLon < this.normalizedLonMin || normalizedLon > this.normalizedLonMax) {
            return Double.NaN;
        }
        return normalizedLon;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TiePointGeoCoding that = (TiePointGeoCoding)o;
        if (!this.latGrid.equals(that.latGrid)) {
            return false;
        }
        return this.lonGrid.equals(that.lonGrid);
    }

    public int hashCode() {
        int result = this.latGrid.hashCode();
        result = 31 * result + this.lonGrid.hashCode();
        return result;
    }

    @Override
    public void dispose() {
    }

    private TiePointGrid initNormalizedLonGrid() {
        int w = this.lonGrid.getGridWidth();
        int h = this.lonGrid.getGridHeight();
        boolean westNormalized = false;
        boolean eastNormalized = false;
        float[] longitudes = this.lonGrid.getTiePoints();
        int numValues = longitudes.length;
        float[] normalizedLongitudes = new float[numValues];
        System.arraycopy(longitudes, 0, normalizedLongitudes, 0, numValues);
        double lonDeltaMax = 0.0;
        for (int y = 0; y < h; ++y) {
            for (int x = 0; x < w; ++x) {
                int index = x + y * w;
                double p2 = normalizedLongitudes[index];
                double p1 = x == 0 && y == 0 ? (double)normalizedLongitudes[index] : (x == 0 ? (double)normalizedLongitudes[x + (y - 1) * w] : (double)normalizedLongitudes[index - 1]);
                double lonDelta = p2 - p1;
                if (lonDelta > 180.0) {
                    westNormalized = true;
                    normalizedLongitudes[index] = (float)(p2 -= 360.0);
                    continue;
                }
                if (lonDelta < -180.0) {
                    eastNormalized = true;
                    normalizedLongitudes[index] = (float)(p2 += 360.0);
                    continue;
                }
                lonDeltaMax = Math.max(lonDeltaMax, Math.abs(lonDelta));
            }
        }
        if (westNormalized) {
            int i = 0;
            while (i < numValues) {
                int n = i++;
                normalizedLongitudes[n] = normalizedLongitudes[n] + 360.0f;
            }
        }
        this.normalized = westNormalized || eastNormalized;
        TiePointGrid normalizedLonGrid = this.normalized ? new TiePointGrid(this.lonGrid.getName(), this.lonGrid.getGridWidth(), this.lonGrid.getGridHeight(), this.lonGrid.getOffsetX(), this.lonGrid.getOffsetY(), this.lonGrid.getSubSamplingX(), this.lonGrid.getSubSamplingY(), normalizedLongitudes, this.lonGrid.getDiscontinuity()) : this.lonGrid;
        Debug.trace("TiePointGeoCoding.westNormalized = " + westNormalized);
        Debug.trace("TiePointGeoCoding.eastNormalized = " + eastNormalized);
        Debug.trace("TiePointGeoCoding.normalized = " + this.normalized);
        Debug.trace("TiePointGeoCoding.lonDeltaMax = " + lonDeltaMax);
        return normalizedLonGrid;
    }

    private void initLatLonMinMax(TiePointGrid normalizedLonGrid) {
        int n;
        float[] latPoints = this.getLatGrid().getTiePoints();
        float[] lonPoints = normalizedLonGrid.getTiePoints();
        this.normalizedLonMin = Double.MAX_VALUE;
        this.normalizedLonMax = -1.7976931348623157E308;
        this.latMin = Double.MAX_VALUE;
        this.latMax = -1.7976931348623157E308;
        float[] fArray = lonPoints;
        int n2 = fArray.length;
        for (n = 0; n < n2; ++n) {
            double lonPoint = fArray[n];
            this.normalizedLonMin = Math.min(this.normalizedLonMin, lonPoint);
            this.normalizedLonMax = Math.max(this.normalizedLonMax, lonPoint);
        }
        fArray = latPoints;
        n2 = fArray.length;
        for (n = 0; n < n2; ++n) {
            double latPoint = fArray[n];
            this.latMin = Math.min(this.latMin, latPoint);
            this.latMax = Math.max(this.latMax, latPoint);
        }
        this.overlapStart = this.normalizedLonMin;
        if (this.overlapStart < -180.0) {
            this.overlapStart += 360.0;
        }
        this.overlapEnd = this.normalizedLonMax;
        if (this.overlapEnd > 180.0) {
            this.overlapEnd -= 360.0;
        }
        Debug.trace("TiePointGeoCoding.normalizedLonMin = " + this.normalizedLonMin);
        Debug.trace("TiePointGeoCoding.normalizedLonMax = " + this.normalizedLonMax);
        Debug.trace("TiePointGeoCoding.latMin = " + this.latMin);
        Debug.trace("TiePointGeoCoding.latMax = " + this.latMax);
        Debug.trace("TiePointGeoCoding.overlapRange = " + this.overlapStart + " - " + this.overlapEnd);
    }

    private Approximation[] initApproximations(TiePointGrid normalizedLonGrid) {
        int numTiles;
        int numPoints = this.latGrid.getGridData().getNumElems();
        int w = this.latGrid.getGridWidth();
        int h = this.latGrid.getGridHeight();
        double subSamplingX = this.latGrid.getSubSamplingX();
        double subSamplingY = this.latGrid.getSubSamplingY();
        if (h > 2) {
            double lonSpan = this.normalizedLonMax - this.normalizedLonMin;
            double latSpan = this.latMax - this.latMin;
            double angleSpan = Math.max(lonSpan, latSpan);
            numTiles = (int)Math.round(angleSpan / 10.0);
            if (numTiles < 1) {
                numTiles = 1;
            }
        } else {
            numTiles = 30;
        }
        int numTilesI = 1;
        int numTilesJ = 1;
        while (numTiles > 1) {
            Dimension tileDim = MathUtils.fitDimension(numTiles, (double)w * subSamplingX, (double)h * subSamplingY);
            int newNumTilesI = tileDim.width;
            int newNumTilesJ = tileDim.height;
            int newNumTiles = newNumTilesI * newNumTilesJ;
            if (numPoints / newNumTiles >= 10) {
                numTiles = newNumTiles;
                numTilesI = newNumTilesI;
                numTilesJ = newNumTilesJ;
                break;
            }
            --numTiles;
        }
        Debug.trace("TiePointGeoCoding.numTiles =  " + numTiles);
        Debug.trace("TiePointGeoCoding.numTilesI = " + numTilesI);
        Debug.trace("TiePointGeoCoding.numTilesJ = " + numTilesJ);
        Approximation[] approximations = new Approximation[numTiles];
        Rectangle[] rectangles = MathUtils.subdivideRectangle(w, h, numTilesI, numTilesJ, 1);
        for (int i = 0; i < rectangles.length; ++i) {
            Approximation approximation = this.createApproximation(normalizedLonGrid, rectangles[i]);
            if (approximation == null) {
                return null;
            }
            approximations[i] = approximation;
        }
        return approximations;
    }

    private static FXYSum getBestPolynomial(double[][] data, int[] indices) {
        FXYSum[] potentialPolynomials = new FXYSum[]{new FXYSum.Linear(), new FXYSum.BiLinear(), new FXYSum.Quadric(), new FXYSum.BiQuadric(), new FXYSum.Cubic(), new FXYSum.BiCubic(), new FXYSum(FXYSum.FXY_4TH, 4), new FXYSum(FXYSum.FXY_BI_4TH, 8)};
        double rmseMin = Double.MAX_VALUE;
        int index = -1;
        for (int i = 0; i < potentialPolynomials.length; ++i) {
            FXYSum potentialPolynomial = potentialPolynomials[i];
            int order = potentialPolynomial.getOrder();
            int numPointsRequired = order >= 0 ? (order + 2) * (order + 1) / 2 : 2 * potentialPolynomial.getNumTerms();
            if (data.length < numPointsRequired) continue;
            try {
                potentialPolynomial.approximate(data, indices);
                double rmse = potentialPolynomial.getRootMeanSquareError();
                double maxError = potentialPolynomial.getMaxError();
                if (rmse < rmseMin) {
                    index = i;
                    rmseMin = rmse;
                }
                if (!(maxError < 0.5)) continue;
                index = i;
                break;
            }
            catch (ArithmeticException e) {
                Debug.trace("Polynomial cannot be constructed due to a numerically singular or degenerate matrix:");
                Debug.trace(e);
            }
        }
        return index >= 0 ? potentialPolynomials[index] : null;
    }

    private double[][] createWarpPoints(TiePointGrid lonGrid, Rectangle subsetRect) {
        TiePointGrid latGrid = this.getLatGrid();
        int w = latGrid.getGridWidth();
        int sw = subsetRect.width;
        int sh = subsetRect.height;
        int i1 = subsetRect.x;
        int i2 = i1 + sw - 1;
        int j1 = subsetRect.y;
        int j2 = j1 + sh - 1;
        Debug.trace("Selecting warp points for X/Y approximations");
        Debug.trace("  subset rectangle (in tie point coordinates): " + subsetRect);
        Debug.trace("  index i: " + i1 + " to " + i2);
        Debug.trace("  index j: " + j1 + " to " + j2);
        int[] warpParameters = TiePointGeoCoding.determineWarpParameters(sw, sh);
        int numU = warpParameters[0];
        int numV = warpParameters[1];
        int stepI = warpParameters[2];
        int stepJ = warpParameters[3];
        int m = numU * numV;
        double[][] data = new double[m][4];
        int k = 0;
        for (int v = 0; v < numV; ++v) {
            int j = j1 + v * stepJ;
            if (j > j2) {
                j = j2;
            }
            for (int u = 0; u < numU; ++u) {
                int i = i1 + u * stepI;
                if (i > i2) {
                    i = i2;
                }
                double lat = latGrid.getGridData().getElemDoubleAt(j * w + i);
                double lon = lonGrid.getGridData().getElemDoubleAt(j * w + i);
                double x = latGrid.getOffsetX() + (double)i * latGrid.getSubSamplingX();
                double y = latGrid.getOffsetY() + (double)j * latGrid.getSubSamplingY();
                data[k][0] = lat;
                data[k][1] = lon;
                data[k][2] = x;
                data[k][3] = y;
                ++k;
            }
        }
        Debug.assertTrue(k == m);
        Debug.trace("TiePointGeoCoding: numU=" + numU + ", stepI=" + stepI);
        Debug.trace("TiePointGeoCoding: numV=" + numV + ", stepJ=" + stepJ);
        return data;
    }

    static int[] determineWarpParameters(int sw, int sh) {
        boolean adjustStepI;
        int numU = sw;
        int numV = sh;
        int stepI = 1;
        int stepJ = 1;
        boolean bl = adjustStepI = numU >= numV;
        while (numU * numV > 1000) {
            if (adjustStepI) {
                numU = sw / ++stepI;
                while (numU * stepI < sw) {
                    ++numU;
                }
            } else {
                numV = sh / ++stepJ;
                while (numV * stepJ < sh) {
                    ++numV;
                }
            }
            adjustStepI = numU >= numV;
        }
        return new int[]{numU, numV, stepI, stepJ};
    }

    private Approximation createApproximation(TiePointGrid normalizedLonGrid, Rectangle subsetRect) {
        double[][] data = this.createWarpPoints(normalizedLonGrid, subsetRect);
        double sumLat = 0.0;
        double sumLon = 0.0;
        for (double[] point : data) {
            sumLat += point[0];
            sumLon += point[1];
        }
        double centerLon = sumLon / (double)data.length;
        double centerLat = sumLat / (double)data.length;
        double maxSquareDistance = TiePointGeoCoding.getMaxSquareDistance(data, centerLat, centerLon);
        for (int i = 0; i < data.length; ++i) {
            data[i][0] = TiePointGeoCoding.rescaleLatitude(data[i][0]);
            data[i][1] = TiePointGeoCoding.rescaleLongitude(data[i][1], centerLon);
        }
        int[] xIndices = new int[]{0, 1, 2};
        int[] yIndices = new int[]{0, 1, 3};
        FXYSum fX = TiePointGeoCoding.getBestPolynomial(data, xIndices);
        FXYSum fY = TiePointGeoCoding.getBestPolynomial(data, yIndices);
        if (fX == null || fY == null) {
            return null;
        }
        double rmseX = fX.getRootMeanSquareError();
        double rmseY = fY.getRootMeanSquareError();
        double maxErrorX = fX.getMaxError();
        double maxErrorY = fY.getMaxError();
        Debug.trace("TiePointGeoCoding: RMSE X      = " + rmseX + ", " + (rmseX < 0.5 ? "OK" : "too large"));
        Debug.trace("TiePointGeoCoding: RMSE Y      = " + rmseY + ", " + (rmseY < 0.5 ? "OK" : "too large"));
        Debug.trace("TiePointGeoCoding: Max.error X = " + maxErrorX + ", " + (maxErrorX < 0.5 ? "OK" : "too large"));
        Debug.trace("TiePointGeoCoding: Max.error Y = " + maxErrorY + ", " + (maxErrorY < 0.5 ? "OK" : "too large"));
        return new Approximation(fX, fY, centerLat, centerLon, maxSquareDistance * 1.1);
    }

    private static double getMaxSquareDistance(double[][] data, double centerLat, double centerLon) {
        double maxSquareDistance = 0.0;
        for (double[] point : data) {
            double dLat = point[0] - centerLat;
            double dLon = point[1] - centerLon;
            double squareDistance = dLat * dLat + dLon * dLon;
            if (!(squareDistance > maxSquareDistance)) continue;
            maxSquareDistance = squareDistance;
        }
        return maxSquareDistance;
    }

    private static Approximation getBestApproximation(Approximation[] approximations, double lat, double lon) {
        Approximation approximation = null;
        if (approximations.length == 1) {
            Approximation a = approximations[0];
            double squareDistance = a.getSquareDistance(lat, lon);
            if (squareDistance < a.getMinSquareDistance()) {
                approximation = a;
            }
        } else {
            double minSquareDistance = Double.MAX_VALUE;
            for (Approximation a : approximations) {
                double squareDistance = a.getSquareDistance(lat, lon);
                if (!(squareDistance < minSquareDistance) || !(squareDistance < a.getMinSquareDistance())) continue;
                minSquareDistance = squareDistance;
                approximation = a;
            }
        }
        return approximation;
    }

    private Approximation findRenormalizedApproximation(double lat, double renormalizedLon, double distance) {
        double renormalizedDistance;
        Approximation renormalizedApproximation = TiePointGeoCoding.getBestApproximation(this.approximations, lat, renormalizedLon);
        if (renormalizedApproximation != null && (renormalizedDistance = renormalizedApproximation.getSquareDistance(lat, renormalizedLon)) < distance) {
            return renormalizedApproximation;
        }
        return null;
    }

    double getNormalizedLonMin() {
        return this.normalizedLonMin;
    }

    private static double rescaleLongitude(double lon, double centerLon) {
        return (lon - centerLon) / 90.0;
    }

    private static double rescaleLatitude(double lat) {
        return lat / 90.0;
    }

    @Override
    public boolean transferGeoCoding(Scene srcScene, Scene destScene, ProductSubsetDef subsetDef) {
        TiePointGrid lonGrid;
        String latGridName = this.getLatGrid().getName();
        String lonGridName = this.getLonGrid().getName();
        Product destProduct = destScene.getProduct();
        TiePointGrid latGrid = destProduct.getTiePointGrid(latGridName);
        if (latGrid == null) {
            latGrid = TiePointGrid.createSubset(this.getLatGrid(), subsetDef);
            destProduct.addTiePointGrid(latGrid);
        }
        if ((lonGrid = destProduct.getTiePointGrid(lonGridName)) == null) {
            lonGrid = TiePointGrid.createSubset(this.getLonGrid(), subsetDef);
            destProduct.addTiePointGrid(lonGrid);
        }
        if (latGrid != null && lonGrid != null) {
            destScene.setGeoCoding(new TiePointGeoCoding(latGrid, lonGrid));
            return true;
        }
        return false;
    }

    public static final class Approximation {
        private final FXYSum _fX;
        private final FXYSum _fY;
        private final double _centerLat;
        private final double _centerLon;
        private final double _minSquareDistance;

        public Approximation(FXYSum fX, FXYSum fY, double centerLat, double centerLon, double minSquareDistance) {
            this._fX = fX;
            this._fY = fY;
            this._centerLat = centerLat;
            this._centerLon = centerLon;
            this._minSquareDistance = minSquareDistance;
        }

        public final FXYSum getFX() {
            return this._fX;
        }

        public final FXYSum getFY() {
            return this._fY;
        }

        public double getCenterLat() {
            return this._centerLat;
        }

        public double getCenterLon() {
            return this._centerLon;
        }

        public double getMinSquareDistance() {
            return this._minSquareDistance;
        }

        public final double getSquareDistance(double lat, double lon) {
            double dx = lon - this._centerLon;
            double dy = lat - this._centerLat;
            return dx * dx + dy * dy;
        }
    }
}

