/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.worldwind.util.measure;

import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.MeasurableArea;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.globes.Globe;
import gov.nasa.worldwind.util.GeometryBuilder;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.WWMath;
import gov.nasa.worldwind.util.measure.LengthMeasurer;
import java.util.ArrayList;

public class AreaMeasurer
extends LengthMeasurer
implements MeasurableArea {
    private static final double DEFAULT_AREA_SAMPLING_STEPS = 32.0;
    private ArrayList<? extends Position> subdividedPositions;
    private Cell[][] sectorCells;
    private Double[][] sectorElevations;
    private double areaTerrainSamplingSteps = 32.0;
    protected double surfaceArea = -1.0;
    protected double projectedArea = -1.0;

    public AreaMeasurer() {
    }

    public AreaMeasurer(ArrayList<? extends Position> arrayList) {
        super(arrayList);
    }

    @Override
    protected void clearCachedValues() {
        super.clearCachedValues();
        this.subdividedPositions = null;
        this.projectedArea = -1.0;
        this.surfaceArea = -1.0;
    }

    @Override
    public void setPositions(ArrayList<? extends Position> arrayList) {
        Sector sector = this.getBoundingSector();
        super.setPositions(arrayList);
        if (this.getBoundingSector() == null || !this.getBoundingSector().equals(sector)) {
            this.sectorCells = null;
            this.sectorElevations = null;
        }
    }

    public double getAreaTerrainSamplingSteps() {
        return this.areaTerrainSamplingSteps;
    }

    public void setAreaTerrainSamplingSteps(double d) {
        if (d < 1.0) {
            String string = Logging.getMessage("generic.ArgumentOutOfRange", d);
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (this.areaTerrainSamplingSteps != d) {
            this.areaTerrainSamplingSteps = d;
            this.surfaceArea = -1.0;
            this.projectedArea = -1.0;
            this.sectorCells = null;
            this.sectorElevations = null;
        }
    }

    @Override
    public double getArea(Globe globe) {
        return this.isFollowTerrain() ? this.getSurfaceArea(globe) : this.getProjectedArea(globe);
    }

    public double getSurfaceArea(Globe globe) {
        if (globe == null) {
            String string = Logging.getMessage("nullValue.GlobeIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (this.surfaceArea < 0.0) {
            this.surfaceArea = this.computeSurfaceAreaSampling(globe, this.areaTerrainSamplingSteps);
        }
        return this.surfaceArea;
    }

    public double getProjectedArea(Globe globe) {
        if (globe == null) {
            String string = Logging.getMessage("nullValue.GlobeIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (this.projectedArea < 0.0) {
            this.projectedArea = this.computeProjectedAreaGeometry(globe);
        }
        return this.projectedArea;
    }

    @Override
    public double getPerimeter(Globe globe) {
        return this.getLength(globe);
    }

    @Override
    public double getWidth(Globe globe) {
        if (globe == null) {
            String string = Logging.getMessage("nullValue.GlobeIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        Sector sector = this.getBoundingSector();
        if (sector != null) {
            return globe.getRadiusAt(sector.getCentroid()) * sector.getDeltaLon().radians * Math.cos(sector.getCentroid().getLatitude().radians);
        }
        return -1.0;
    }

    @Override
    public double getHeight(Globe globe) {
        if (globe == null) {
            String string = Logging.getMessage("nullValue.GlobeIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        Sector sector = this.getBoundingSector();
        if (sector != null) {
            return globe.getRadiusAt(sector.getCentroid()) * sector.getDeltaLat().radians;
        }
        return -1.0;
    }

    protected double computeProjectedAreaGeometry(Globe globe) {
        Sector sector = this.getBoundingSector();
        if (sector != null && this.isClosedShape()) {
            if (this.subdividedPositions == null) {
                this.subdividedPositions = AreaMeasurer.subdividePositions(globe, this.getPositions(), this.getMaxSegmentLength(), this.isFollowTerrain(), this.getPathType());
            }
            int n = this.subdividedPositions.size() - 1;
            float[] fArray = new float[n * 3];
            int n2 = 0;
            for (int i = 0; i < n; ++i) {
                fArray[n2++] = (float)this.subdividedPositions.get((int)i).getLongitude().radians;
                fArray[n2++] = (float)this.subdividedPositions.get((int)i).getLatitude().radians;
                fArray[n2++] = 0.0f;
            }
            GeometryBuilder geometryBuilder = new GeometryBuilder();
            GeometryBuilder.IndexedTriangleArray indexedTriangleArray = geometryBuilder.tessellatePolygon2(0, n, fArray);
            double d = 0.0;
            int[] nArray = indexedTriangleArray.getIndices();
            int n3 = indexedTriangleArray.getIndexCount() / 3;
            for (int i = 0; i < n3; ++i) {
                n2 = i * 3;
                d += this.computeTriangleProjectedArea(globe, indexedTriangleArray.getVertices(), nArray[n2] * 3, nArray[n2 + 1] * 3, nArray[n2 + 2] * 3);
            }
            return d;
        }
        return -1.0;
    }

    protected double computeTriangleProjectedArea(Globe globe, float[] fArray, int n, int n2, int n3) {
        double d = Math.abs(fArray[n] * (fArray[n2 + 1] - fArray[n3 + 1]) + fArray[n2] * (fArray[n3 + 1] - fArray[n + 1]) + fArray[n3] * (fArray[n + 1] - fArray[n2 + 1])) / 2.0f;
        double d2 = (fArray[n + 1] + fArray[n2 + 1] + fArray[n3 + 1]) / 3.0f;
        double d3 = (fArray[n] + fArray[n2] + fArray[n3]) / 3.0f;
        double d4 = globe.getRadiusAt(Angle.fromRadians(d2), Angle.fromRadians(d3));
        return d *= Math.cos(d2) * d4 * d4;
    }

    protected double computeSurfaceAreaSampling(Globe globe, double d) {
        Sector sector = this.getBoundingSector();
        if (sector != null && this.isClosedShape()) {
            if (this.subdividedPositions == null) {
                this.subdividedPositions = AreaMeasurer.subdividePositions(globe, this.getPositions(), this.getMaxSegmentLength(), true, this.getPathType());
            }
            double d2 = Math.max(sector.getDeltaLatRadians() / d, sector.getDeltaLonRadians() / d);
            int n = (int)Math.round(sector.getDeltaLatRadians() / d2);
            int n2 = (int)Math.round(sector.getDeltaLonRadians() / d2 * Math.cos(sector.getCentroid().getLatitude().radians));
            double d3 = sector.getDeltaLatRadians() / (double)n;
            double d4 = sector.getDeltaLonRadians() / (double)n2;
            if (this.sectorCells == null) {
                this.sectorCells = new Cell[n][n2];
            }
            if (this.sectorElevations == null) {
                this.sectorElevations = new Double[n + 1][n2 + 1];
            }
            double d5 = 0.0;
            for (int i = 0; i < n; ++i) {
                double d6 = sector.getMinLatitude().radians + d3 * (double)i;
                double d7 = globe.getRadiusAt(Angle.fromRadians(d6 + d3 / 2.0), sector.getCentroid().getLongitude());
                double d8 = d4 * d7 * Math.cos(d6 + d3 / 2.0);
                double d9 = d3 * d7;
                double d10 = d8 * d9;
                for (int j = 0; j < n2; ++j) {
                    double d11 = sector.getMinLongitude().radians + d4 * (double)j;
                    Sector sector2 = Sector.fromRadians(d6, d6 + d3, d11, d11 + d4);
                    if (!WWMath.isLocationInside(sector2.getCentroid(), this.subdividedPositions)) continue;
                    Cell cell = this.sectorCells[i][j];
                    if (cell == null || cell.surfaceArea == -1.0) {
                        double d12 = this.sectorElevations[i][j] != null ? this.sectorElevations[i][j].doubleValue() : globe.getElevation(Angle.fromRadians(d6), Angle.fromRadians(d11));
                        double d13 = this.sectorElevations[i][j + 1] != null ? this.sectorElevations[i][j + 1].doubleValue() : globe.getElevation(Angle.fromRadians(d6), Angle.fromRadians(d11 + d4));
                        double d14 = this.sectorElevations[i + 1][j] != null ? this.sectorElevations[i + 1][j].doubleValue() : globe.getElevation(Angle.fromRadians(d6 + d3), Angle.fromRadians(d11));
                        this.sectorElevations[i][j] = d12;
                        this.sectorElevations[i][j + 1] = d13;
                        this.sectorElevations[i + 1][j] = d14;
                        Vec4 vec4 = new Vec4(d8, 0.0, d13 - d12).normalize3();
                        Vec4 vec42 = new Vec4(0.0, d9, d14 - d12).normalize3();
                        Vec4 vec43 = vec4.cross3(vec42).normalize3();
                        double d15 = Math.tan(Vec4.UNIT_Z.angleBetween3((Vec4)vec43).radians);
                        double d16 = Math.sqrt(1.0 + d15 * d15);
                        this.sectorCells[i][j] = cell = new Cell(sector2, d10, d10 * d16);
                    }
                    d5 += cell.surfaceArea;
                }
            }
            return d5;
        }
        return -1.0;
    }

    protected class Cell {
        Sector sector;
        double projectedArea;
        double surfaceArea;

        public Cell(Sector sector, double d, double d2) {
            this.sector = sector;
            this.projectedArea = d;
            this.surfaceArea = d2;
        }
    }
}

