/*
 * Decompiled with CFR 0.152.
 */
package org.esa.s3tbx.olci.radiometry.rayleigh;

import com.bc.ceres.core.ProgressMonitor;
import java.awt.Rectangle;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.esa.s3tbx.olci.radiometry.Sensor;
import org.esa.s3tbx.olci.radiometry.gasabsorption.GaseousAbsorptionAux;
import org.esa.s3tbx.olci.radiometry.rayleigh.RayleighAux;
import org.esa.s3tbx.olci.radiometry.rayleigh.RayleighCorrAlgorithm;
import org.esa.s3tbx.olci.radiometry.smilecorr.SmileCorrectionUtils;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.RasterDataNode;
import org.esa.snap.core.gpf.Operator;
import org.esa.snap.core.gpf.OperatorException;
import org.esa.snap.core.gpf.OperatorSpi;
import org.esa.snap.core.gpf.Tile;
import org.esa.snap.core.gpf.annotations.OperatorMetadata;
import org.esa.snap.core.gpf.annotations.Parameter;
import org.esa.snap.core.gpf.annotations.SourceProduct;
import org.esa.snap.core.util.ProductUtils;

@OperatorMetadata(alias="RayleighCorrection", description="Performs radiometric corrections on OLCI and MERIS L1b data products.", authors="Marco Peters, Muhammad Bala (Brockmann Consult)", copyright="(c) 2016 by Brockmann Consult", category="Optical/Pre-Processing", version="1.2")
public class RayleighCorrectionOp
extends Operator {
    private static final String ALTITUDE = "altitude";
    private static final String R_BRR_PATTERN = "rBRR_\\d{2}";
    private static final String AUTO_GROUPING = "rtoa:taur:rtoa_ng:rtoaRay:rBRR";
    private static final int WV_709_FOR_GASEOUS_ABSORPTION_CALCULATION = 709;
    private static final String SOLAR_FLUX_BAND_PATTERN = "solar_flux_band_%d";
    private static final String LAMBDA0_BAND_PATTERN = "lambda0_band_%d";
    private static final String MERIS_SUN_AZIMUTH = "sun_azimuth";
    private static final String MERIS_SUN_ZENITH = "sun_zenith";
    private static final String MERIS_VIEW_ZENITH = "view_zenith";
    private static final String MERIS_VIEW_AZIMUTH = "view_azimuth";
    private static final String MERIS_ATM_PRESS = "atm_press";
    private static final String MERIS_OZONE = "ozone";
    private static final String MERIS_LATITUDE = "latitude";
    private static final String MERIS_LONGITUDE = "longitude";
    private static final String SZA = "SZA";
    private static final String SAA = "SAA";
    private static final String OZA = "OZA";
    private static final String OAA = "OAA";
    private static final String SEA_LEVEL_PRESSURE = "sea_level_pressure";
    private static final String TOTAL_OZONE = "total_ozone";
    private static final String TP_LATITUDE = "TP_latitude";
    private static final String TP_LONGITUDE = "TP_longitude";
    private static final String[] BAND_CATEGORIES = new String[]{"taur_%02d", "rBRR_%02d", "rtoa_ng_%02d", "rtoa_%02d"};
    private static final String AIRMASS = "airmass";
    private static final String ALTITUDE_DEM = "dem_alt";
    private static final String RTOA_PATTERN = "rtoa_\\d{2}";
    private static final String TAUR_PATTERN = "taur_\\d{2}";
    private static final String RTOA_NG_PATTERN = "rtoa_ng_\\d{2}";
    @SourceProduct
    Product sourceProduct;
    @Parameter(defaultValue="true", label="Compute Rayleigh optical thickness bands")
    private boolean computeTaur;
    @Parameter(defaultValue="true", label="Compute bottom of Rayleigh reflectance bands")
    private boolean computeRBrr;
    @Parameter(defaultValue="false", label="Compute gaseous absorption corrected TOA reflectance bands")
    private boolean computeRtoaNg;
    @Parameter(defaultValue="false", label="Compute TOA reflectance bands")
    private boolean computeRtoa;
    @Parameter(defaultValue="false", label="Add air mass")
    private boolean addAirMass;
    private RayleighCorrAlgorithm algorithm;
    private Sensor sensor;
    private double[] absorpOzone;
    private double[] crossSectionSigma;

    public void initialize() throws OperatorException {
        this.sensor = SmileCorrectionUtils.getSensorType(this.sourceProduct);
        this.algorithm = new RayleighCorrAlgorithm();
        this.absorpOzone = GaseousAbsorptionAux.getInstance().absorptionOzone(this.sensor.toString());
        this.crossSectionSigma = this.getCrossSectionSigma(this.sourceProduct, this.sensor.getNumBands(), this.sensor.getNamePattern());
        Product targetProduct = new Product(this.sourceProduct.getName() + "_rayleigh", this.sourceProduct.getProductType(), this.sourceProduct.getSceneRasterWidth(), this.sourceProduct.getSceneRasterHeight());
        RayleighAux.initDefaultAuxiliary();
        this.addTargetBands(targetProduct);
        ProductUtils.copyProductNodes((Product)this.sourceProduct, (Product)targetProduct);
        ProductUtils.copyFlagBands((Product)this.sourceProduct, (Product)targetProduct, (boolean)true);
        targetProduct.setAutoGrouping(AUTO_GROUPING);
        this.setTargetProduct(targetProduct);
    }

    public void computeTileStack(Map<Band, Tile> targetTiles, Rectangle targetRectangle, ProgressMonitor pm) throws OperatorException {
        this.checkForCancellation();
        RayleighAux rayleighAux = this.createAuxiliary(this.sensor, targetRectangle);
        Set<Map.Entry<Band, Tile>> entries = targetTiles.entrySet();
        entries.forEach(targetTileStream -> {
            Tile targetTile = (Tile)targetTileStream.getValue();
            Band targetBand = (Band)targetTileStream.getKey();
            String targetBandName = targetBand.getName();
            double[] rayleighOpticalThickness = null;
            int sourceBandIndex = SmileCorrectionUtils.getSourceBandIndex(targetBand.getName());
            if (targetBandName.equals(AIRMASS) && this.addAirMass) {
                double[] massAirs = rayleighAux.getAirMass();
                targetTile.setSamples(massAirs);
                return;
            }
            if (sourceBandIndex == -1) {
                return;
            }
            this.addAuxiliaryData(rayleighAux, targetRectangle, sourceBandIndex);
            if (targetBandName.matches(RTOA_PATTERN) && this.computeRtoa) {
                targetTile.setSamples(this.getReflectance(rayleighAux));
            } else if (targetBandName.matches(TAUR_PATTERN) && this.computeTaur) {
                rayleighOpticalThickness = this.algorithm.getRayleighThickness(rayleighAux, this.crossSectionSigma, sourceBandIndex);
                targetTile.setSamples(rayleighOpticalThickness);
            } else if (this.computeRBrr || this.computeRtoaNg) {
                double[] reflectance = this.getReflectance(rayleighAux);
                if (Math.ceil(rayleighAux.getWaveLength()) == 709.0) {
                    reflectance = this.waterVaporCorrection709(reflectance, targetRectangle, this.sensor);
                }
                double[] corrOzoneRefl = this.getCorrectOzone(rayleighAux, reflectance, sourceBandIndex);
                if (targetBandName.matches(RTOA_NG_PATTERN) && this.computeRtoaNg) {
                    targetTile.setSamples(corrOzoneRefl);
                }
                if (targetBandName.matches(R_BRR_PATTERN) && this.computeRBrr) {
                    if (Objects.isNull(rayleighOpticalThickness)) {
                        rayleighOpticalThickness = this.algorithm.getRayleighThickness(rayleighAux, this.crossSectionSigma, sourceBandIndex);
                    }
                    double[] rhoBrr = this.getRhoBrr(rayleighAux, rayleighOpticalThickness, corrOzoneRefl);
                    targetTile.setSamples(rhoBrr);
                }
            }
        });
    }

    private double[] waterVaporCorrection709(double[] reflectances, Rectangle targetRectangle, Sensor sensor) {
        String bandNamePattern = sensor.getNamePattern();
        int[] upperLowerBounds = sensor.getBounds();
        double[] bWVRefTile = SmileCorrectionUtils.getSampleDoubles(this.getSourceTile((RasterDataNode)this.sourceProduct.getBand(String.format(bandNamePattern, upperLowerBounds[1])), targetRectangle));
        double[] bWVTile = SmileCorrectionUtils.getSampleDoubles(this.getSourceTile((RasterDataNode)this.sourceProduct.getBand(String.format(bandNamePattern, upperLowerBounds[0])), targetRectangle));
        return this.algorithm.waterVaporCorrection709(reflectances, bWVRefTile, bWVTile);
    }

    private double[] getRhoBrr(RayleighAux rayleighAux, double[] rayleighOpticalThickness, double[] corrOzoneRefl) {
        return this.algorithm.getRhoBrr(rayleighAux, rayleighOpticalThickness, corrOzoneRefl);
    }

    private double[] getCorrectOzone(RayleighAux rayleighAux, double[] reflectance, int sourceBandIndex) {
        double absorpO = this.absorpOzone[sourceBandIndex - 1];
        double[] totalOzones = rayleighAux.getTotalOzones();
        double[] cosOZARads = rayleighAux.getCosOZARads();
        double[] cosSZARads = rayleighAux.getCosSZARads();
        return this.algorithm.getCorrOzone(reflectance, absorpO, totalOzones, cosOZARads, cosSZARads);
    }

    private double[] getReflectance(RayleighAux rayleighAux) {
        double[] sourceSampleRad = rayleighAux.getSourceSampleRad();
        double[] solarFluxs = rayleighAux.getSolarFluxs();
        double[] sunZenithAngles = rayleighAux.getSunZenithAngles();
        return this.algorithm.convertRadsToRefls(sourceSampleRad, solarFluxs, sunZenithAngles);
    }

    private void addTargetBands(Product targetProduct) {
        if (this.computeTaur) {
            this.addTargetBands(targetProduct, BAND_CATEGORIES[0]);
        }
        if (this.computeRBrr) {
            this.addTargetBands(targetProduct, BAND_CATEGORIES[1]);
        }
        if (this.computeRtoaNg) {
            this.addTargetBands(targetProduct, BAND_CATEGORIES[2]);
        }
        if (this.computeRtoa) {
            this.addTargetBands(targetProduct, BAND_CATEGORIES[3]);
        }
        if (this.addAirMass) {
            targetProduct.addBand(AIRMASS, 30);
        }
    }

    private void addTargetBands(Product targetProduct, String bandCategory) {
        for (int i = 1; i <= this.sensor.getNumBands(); ++i) {
            Band sourceBand = this.sourceProduct.getBand(String.format(this.sensor.getNamePattern(), i));
            Band targetBand = targetProduct.addBand(String.format(bandCategory, i), 30);
            ProductUtils.copySpectralBandProperties((Band)sourceBand, (Band)targetBand);
        }
    }

    private int addAuxiliaryData(RayleighAux rayleighAux, Rectangle rectangle, int sourceBandRefIndex) {
        String format = String.format(this.sensor.getNamePattern(), sourceBandRefIndex);
        Band band = this.getSourceProduct().getBand(format);
        String sourceBandName = band.getName();
        rayleighAux.setWavelength(band.getSpectralWavelength());
        rayleighAux.setSourceBandIndex(sourceBandRefIndex);
        rayleighAux.setSourceBandName(sourceBandName);
        if (this.sensor.equals((Object)Sensor.OLCI)) {
            rayleighAux.setSolarFluxs(this.getSourceTile((RasterDataNode)this.sourceProduct.getBand(String.format(SOLAR_FLUX_BAND_PATTERN, sourceBandRefIndex)), rectangle));
            rayleighAux.setLambdaSource(this.getSourceTile((RasterDataNode)this.sourceProduct.getBand(String.format(LAMBDA0_BAND_PATTERN, sourceBandRefIndex)), rectangle));
            rayleighAux.setSourceSampleRad(this.getSourceTile((RasterDataNode)this.sourceProduct.getBand(sourceBandName), rectangle));
        } else if (this.sensor.equals((Object)Sensor.MERIS)) {
            Band sourceBand = this.sourceProduct.getBand(sourceBandName);
            rayleighAux.setSourceSampleRad(this.getSourceTile((RasterDataNode)sourceBand, rectangle));
            int length = rectangle.width * rectangle.height;
            double[] solarFlux = this.fillDefaultArray(length, sourceBand.getSolarFlux());
            double[] lambdaSource = this.fillDefaultArray(length, sourceBand.getSpectralWavelength());
            rayleighAux.setSolarFluxs(solarFlux);
            rayleighAux.setLambdaSource(lambdaSource);
        }
        return sourceBandRefIndex;
    }

    private double[] fillDefaultArray(int length, double value) {
        double[] createArray = new double[length];
        Arrays.fill(createArray, value);
        return createArray;
    }

    private RayleighAux createAuxiliary(Sensor sensor, Rectangle rectangle) {
        RayleighAux rayleighAux = new RayleighAux();
        if (sensor.equals((Object)Sensor.MERIS)) {
            rayleighAux.setSunAzimuthAngles(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(MERIS_SUN_AZIMUTH), rectangle));
            rayleighAux.setSunZenithAngles(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(MERIS_SUN_ZENITH), rectangle));
            rayleighAux.setViewZenithAngles(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(MERIS_VIEW_ZENITH), rectangle));
            rayleighAux.setViewAzimuthAngles(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(MERIS_VIEW_AZIMUTH), rectangle));
            rayleighAux.setSeaLevels(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(MERIS_ATM_PRESS), rectangle));
            rayleighAux.setTotalOzones(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(MERIS_OZONE), rectangle));
            rayleighAux.setLatitudes(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(MERIS_LATITUDE), rectangle));
            rayleighAux.setLongitude(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(MERIS_LONGITUDE), rectangle));
            rayleighAux.setAltitudes(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(ALTITUDE_DEM), rectangle));
        } else if (sensor.equals((Object)Sensor.OLCI)) {
            rayleighAux.setSunZenithAngles(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(SZA), rectangle));
            rayleighAux.setViewZenithAngles(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(OZA), rectangle));
            rayleighAux.setSunAzimuthAngles(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(SAA), rectangle));
            rayleighAux.setViewAzimuthAngles(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(OAA), rectangle));
            rayleighAux.setSeaLevels(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(SEA_LEVEL_PRESSURE), rectangle));
            rayleighAux.setTotalOzones(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(TOTAL_OZONE), rectangle));
            rayleighAux.setLatitudes(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(TP_LATITUDE), rectangle));
            rayleighAux.setLongitude(this.getSourceTile((RasterDataNode)this.sourceProduct.getTiePointGrid(TP_LONGITUDE), rectangle));
            rayleighAux.setAltitudes(this.getSourceTile((RasterDataNode)this.sourceProduct.getBand(ALTITUDE), rectangle));
        }
        return rayleighAux;
    }

    private double[] getCrossSectionSigma(Product sourceProduct, int numBands, String getBandNamePattern) {
        return this.algorithm.getCrossSectionSigma(sourceProduct, numBands, getBandNamePattern);
    }

    public static class Spi
    extends OperatorSpi {
        public Spi() {
            super(RayleighCorrectionOp.class);
        }
    }
}

