/*
 * Decompiled with CFR 0.152.
 */
package org.esa.s3tbx.smac;

import com.bc.ceres.core.Assert;
import com.bc.ceres.core.ProgressMonitor;
import java.awt.Color;
import java.awt.Rectangle;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.esa.s3tbx.smac.AEROSOL_TYPE;
import org.esa.s3tbx.smac.SensorCoefficientFile;
import org.esa.s3tbx.smac.SensorCoefficientManager;
import org.esa.s3tbx.smac.SmacAlgorithm;
import org.esa.s3tbx.smac.SmacSensorCoefficients;
import org.esa.s3tbx.smac.SmacUtils;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.Mask;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.RasterDataNode;
import org.esa.snap.core.datamodel.TiePointGrid;
import org.esa.snap.core.gpf.Operator;
import org.esa.snap.core.gpf.OperatorException;
import org.esa.snap.core.gpf.OperatorSpi;
import org.esa.snap.core.gpf.Tile;
import org.esa.snap.core.gpf.annotations.OperatorMetadata;
import org.esa.snap.core.gpf.annotations.Parameter;
import org.esa.snap.core.gpf.annotations.SourceProduct;
import org.esa.snap.core.gpf.annotations.TargetProduct;
import org.esa.snap.core.util.ObjectUtils;
import org.esa.snap.core.util.ProductUtils;
import org.esa.snap.core.util.ResourceInstaller;
import org.esa.snap.core.util.SystemUtils;
import org.esa.snap.core.util.converters.BooleanExpressionConverter;
import org.esa.snap.core.util.math.RsMathUtils;
import org.esa.snap.dataio.envisat.EnvisatConstants;

@OperatorMetadata(alias="SmacOp", category="Optical/Thematic Land Processing", version="1.5.205", authors="H. Rahman, G. Dedieu (Algorithm); T. Block, T. Storm (Implementation)", copyright="Copyright (C) 2002-2014 by Brockmann Consult (info@brockmann-consult.de)", description="Applies the Simplified Method for Atmospheric Corrections of Envisat MERIS/(A)ATSR measurements.")
public class SmacOperator
extends Operator {
    private static final String DEFAULT_MERIS_FLAGS_VALUE = "l1_flags.LAND_OCEAN and not (l1_flags.INVALID or l1_flags.BRIGHT)";
    private static final String DEFAULT_FORWARD_FLAGS_VALUE = "cloud_flags_fward.LAND and not cloud_flags_fward.CLOUDY";
    private static final String DEFAULT_NADIR_FLAGS_VALUE = "cloud_flags_nadir.LAND and not cloud_flags_nadir.CLOUDY";
    private static final String LOG_MSG_LOADED = "Loaded ";
    private static final int merisSzaIndex = 6;
    private static final int merisSaaIndex = 7;
    private static final int merisVzaIndex = 8;
    private static final int merisVaaIndex = 9;
    private static final int merisPressIndex = 12;
    private static final int merisElevIndex = 2;
    private static final int merisO3Index = 13;
    private static final int merisWvIndex = 14;
    private static final int aatsrSzaIndex = 7;
    private static final int aatsrSzaFwdIndex = 11;
    private static final int aatsrSaaIndex = 9;
    private static final int aatsrSaaFwdIndex = 13;
    private static final int aatsrVzaIndex = 8;
    private static final int aatsrVzaFwdIndex = 12;
    private static final int aatsrVaaIndex = 10;
    private static final int aatsrVaaFwdIndex = 14;
    private static final float duToCmAtm = 0.001f;
    private static final float relHumTogcm = 0.07f;
    private static final String merisBandPrefix = "reflec";
    private static final String SMAC_MASK = "smac_mask";
    private static final String SMAC_MASK_FORWARD = "smac_mask_forward";
    private final List<Band> inputBandList;
    private final Logger logger;
    private String sensorType;
    private Path auxdataInstallDir;
    private Map<String, String> bandNameMapping;
    private HashMap<String, SmacSensorCoefficients> coefficients;
    private TiePointGrid szaBand;
    private TiePointGrid saaBand;
    private TiePointGrid vzaBand;
    private TiePointGrid vaaBand;
    private TiePointGrid wvBand;
    private TiePointGrid o3Band;
    private TiePointGrid pressBand;
    private TiePointGrid elevBand;
    private TiePointGrid szaFwdBand;
    private TiePointGrid saaFwdBand;
    private TiePointGrid vzaFwdBand;
    private TiePointGrid vaaFwdBand;
    @Parameter(description="Aerosol optical depth", label="Aerosol optical depth", defaultValue="0.2")
    private Float tauAero550 = Float.valueOf(0.2f);
    @Parameter(description="Relative humidity", label="Relative humidity", defaultValue="3.0", unit="g/cm\u00b2")
    private Float uH2o = Float.valueOf(3.0f);
    @Parameter(description="Ozone content", label="Ozone content", defaultValue="0.15", unit="cm * atm")
    private Float uO3 = Float.valueOf(0.15f);
    @Parameter(description="Surface pressure", label="Surface pressure", defaultValue="1013.0", unit="hPa")
    private Float surfPress = Float.valueOf(1013.0f);
    @Parameter(description="Use ECMWF data in the MERIS ADS", label="Use MERIS ECMWF data", defaultValue="false")
    private Boolean useMerisADS = true;
    @Parameter(description="Aerosol type", label="Aerosol type", notNull=true, defaultValue="CONTINENTAL")
    private AEROSOL_TYPE aerosolType;
    @Parameter(description="Default reflectance for invalid pixel", label="Default reflectance for invalid pixel", defaultValue="0.0")
    Float invalidPixel = Float.valueOf(0.0f);
    @Parameter(description="Mask expression for the whole view (MERIS) or the nadir view (AATSR)", label="Mask expression for the whole view (MERIS) or the nadir view (AATSR)", converter=BooleanExpressionConverter.class)
    private String maskExpression = "";
    @Parameter(description="Mask expression for the forward view (AATSR only)", label="Mask expression for the forward view (AATSR only)", converter=BooleanExpressionConverter.class)
    private String maskExpressionForward = "";
    @Parameter(description="Bands to process", label="Bands to process", notNull=true, rasterDataNodeType=Band.class)
    private String[] bandNames;
    @SourceProduct(alias="source", label="Source product", description="The source product.")
    private Product sourceProduct;
    @TargetProduct(label="SMAC product")
    private Product targetProduct;

    public SmacOperator() {
        this.inputBandList = new ArrayList<Band>();
        this.logger = this.getLogger();
        this.bandNameMapping = new HashMap<String, String>();
        this.coefficients = new HashMap();
    }

    public void initialize() throws OperatorException {
        try {
            this.prepareProcessing();
            this.createOutputProduct();
        }
        catch (IOException e) {
            throw new OperatorException((Throwable)e);
        }
    }

    public void computeTileStack(Map<Band, Tile> targetTiles, Rectangle targetRectangle, ProgressMonitor pm) throws OperatorException {
        int width = targetRectangle.width;
        int height = targetRectangle.height;
        int x = targetRectangle.x;
        int y = targetRectangle.y;
        SourceData sourceData = new SourceData();
        sourceData.taup550 = new float[width * height];
        sourceData.process = new boolean[width * height];
        sourceData.uh2o = new float[width * height];
        sourceData.uo3 = new float[width * height];
        sourceData.press = new float[width * height];
        sourceData.toa = new float[width * height];
        for (int i = 0; i < width * height; ++i) {
            sourceData.taup550[i] = this.tauAero550.floatValue();
            sourceData.uh2o[i] = this.uH2o.floatValue();
            sourceData.uo3[i] = this.uO3.floatValue();
            sourceData.press[i] = this.surfPress.floatValue();
            sourceData.process[i] = true;
        }
        sourceData.sza = this.getSourceTile((RasterDataNode)this.szaBand, targetRectangle).getSamplesFloat();
        sourceData.saa = this.getSourceTile((RasterDataNode)this.saaBand, targetRectangle).getSamplesFloat();
        sourceData.vza = this.getSourceTile((RasterDataNode)this.vzaBand, targetRectangle).getSamplesFloat();
        sourceData.vaa = this.getSourceTile((RasterDataNode)this.vaaBand, targetRectangle).getSamplesFloat();
        if (ObjectUtils.equalObjects((Object)this.sensorType, (Object)"AATSR")) {
            sourceData.szaFwd = this.getSourceTile((RasterDataNode)this.szaFwdBand, targetRectangle).getSamplesFloat();
            sourceData.saaFwd = this.getSourceTile((RasterDataNode)this.saaFwdBand, targetRectangle).getSamplesFloat();
            sourceData.vzaFwd = this.getSourceTile((RasterDataNode)this.vzaFwdBand, targetRectangle).getSamplesFloat();
            sourceData.vaaFwd = this.getSourceTile((RasterDataNode)this.vaaFwdBand, targetRectangle).getSamplesFloat();
        } else if (this.useMerisADS.booleanValue()) {
            sourceData.uh2o = this.getSourceTile((RasterDataNode)this.wvBand, targetRectangle).getSamplesFloat();
            sourceData.uo3 = this.getSourceTile((RasterDataNode)this.o3Band, targetRectangle).getSamplesFloat();
            sourceData.press = this.getSourceTile((RasterDataNode)this.pressBand, targetRectangle).getSamplesFloat();
            sourceData.elev = this.getSourceTile((RasterDataNode)this.elevBand, targetRectangle).getSamplesFloat();
        }
        for (Map.Entry<Band, Tile> bandTileEntry : targetTiles.entrySet()) {
            Band sourceBand = this.sourceProduct.getBand(SmacOperator.revertMerisBandName(bandTileEntry.getKey().getName(), this.bandNameMapping));
            sourceBand.setValidPixelExpression("");
            Tile sourceTile = this.getSourceTile((RasterDataNode)sourceBand, targetRectangle);
            sourceData.toa = sourceTile.getSamplesFloat();
            Tile targetTile = bandTileEntry.getValue();
            try {
                if (ObjectUtils.equalObjects((Object)this.sensorType, (Object)"MERIS")) {
                    if (this.useMerisADS.booleanValue()) {
                        this.processMerisWithADS(sourceBand, sourceData, targetTile, targetRectangle, new SmacAlgorithm());
                        continue;
                    }
                    this.processMeris(sourceBand, sourceData, targetTile, targetRectangle, new SmacAlgorithm());
                    continue;
                }
                if (!ObjectUtils.equalObjects((Object)this.sensorType, (Object)"AATSR")) continue;
                this.processAatsr(sourceBand.getName(), sourceData, targetTile, targetRectangle, new SmacAlgorithm());
            }
            catch (IOException e) {
                this.logger.severe("An error occurred during processing: ");
                this.logger.severe(e.getMessage());
            }
        }
    }

    void installAuxdata() {
        this.auxdataInstallDir = this.initAuxdataInstallDir();
        try {
            Path sourceDirPath = ResourceInstaller.findModuleCodeBasePath(((Object)((Object)this)).getClass()).resolve("auxdata");
            new ResourceInstaller(sourceDirPath, this.auxdataInstallDir).install(".*", ProgressMonitor.NULL);
        }
        catch (IOException e) {
            throw new OperatorException("Failed to install auxdata into " + this.auxdataInstallDir.toString(), (Throwable)e);
        }
    }

    Path getAuxdataInstallDir() {
        return this.auxdataInstallDir;
    }

    private void prepareProcessing() throws IOException {
        this.logger.info("Preparing SMAC processing");
        this.loadInputProduct();
        this.createMask();
        this.installAuxdata();
    }

    private void loadInputProduct() throws IOException {
        this.sensorType = SmacUtils.getSensorType(this.sourceProduct.getProductType());
        if (ObjectUtils.equalObjects((Object)this.sensorType, (Object)"MERIS")) {
            this.loadMERIS_ADS(this.sourceProduct);
        } else if (ObjectUtils.equalObjects((Object)this.sensorType, (Object)"AATSR")) {
            this.loadAATSR_ADS(this.sourceProduct);
            this.useMerisADS = false;
        } else {
            throw new OperatorException(String.format("Unsupported input product of type '%s'.\nSMAC processes AATSR and MERIS L1b products.", this.sourceProduct.getProductType()));
        }
        if (this.bandNames == null || this.bandNames.length == 0) {
            throw new OperatorException("No input bands defined, processing cannot be performed");
        }
        for (String bandName : this.bandNames) {
            Band band = this.sourceProduct.getBand(bandName);
            if (band == null) {
                this.logger.warning("The requested band '" + bandName + "' is not contained in the input product!");
                continue;
            }
            if (band.getSpectralBandIndex() != -1) {
                this.inputBandList.add(band);
                continue;
            }
            this.logger.warning("The requested band '" + bandName + "' is not a spectral band and will be excluded from processing");
        }
    }

    private void loadMERIS_ADS(Product product) {
        this.logger.info("Loading MERIS ADS");
        String gridName = EnvisatConstants.MERIS_TIE_POINT_GRID_NAMES[6];
        this.szaBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.szaBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.MERIS_TIE_POINT_GRID_NAMES[7];
        this.saaBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.saaBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.MERIS_TIE_POINT_GRID_NAMES[8];
        this.vzaBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.vzaBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.MERIS_TIE_POINT_GRID_NAMES[9];
        this.vaaBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.vaaBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        if (this.useMerisADS.booleanValue()) {
            gridName = EnvisatConstants.MERIS_TIE_POINT_GRID_NAMES[14];
            this.wvBand = product.getTiePointGrid(gridName);
            SmacOperator.checkForNull(gridName, this.wvBand);
            this.logger.fine(LOG_MSG_LOADED + gridName);
            gridName = EnvisatConstants.MERIS_TIE_POINT_GRID_NAMES[13];
            this.o3Band = product.getTiePointGrid(gridName);
            SmacOperator.checkForNull(gridName, this.o3Band);
            this.logger.fine(LOG_MSG_LOADED + gridName);
            gridName = EnvisatConstants.MERIS_TIE_POINT_GRID_NAMES[12];
            this.pressBand = product.getTiePointGrid(gridName);
            SmacOperator.checkForNull(gridName, this.pressBand);
            this.logger.fine(LOG_MSG_LOADED + gridName);
            gridName = EnvisatConstants.MERIS_TIE_POINT_GRID_NAMES[2];
            this.elevBand = product.getTiePointGrid(gridName);
            SmacOperator.checkForNull(gridName, this.elevBand);
            this.logger.fine(LOG_MSG_LOADED + gridName);
        }
        this.logger.info("... success");
    }

    private static void checkForNull(String gridName, TiePointGrid tiePointGrid) {
        if (tiePointGrid == null) {
            throw new OperatorException("Tie-point grid '" + gridName + "' must be present in product.");
        }
    }

    private void loadAATSR_ADS(Product product) {
        this.logger.info("Loading AATSR ADS");
        String gridName = EnvisatConstants.AATSR_TIE_POINT_GRID_NAMES[7];
        this.szaBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.szaBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.AATSR_TIE_POINT_GRID_NAMES[11];
        this.szaFwdBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.szaFwdBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.AATSR_TIE_POINT_GRID_NAMES[9];
        this.saaBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.saaBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.AATSR_TIE_POINT_GRID_NAMES[13];
        this.saaFwdBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.saaFwdBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.AATSR_TIE_POINT_GRID_NAMES[8];
        this.vzaBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.vzaBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.AATSR_TIE_POINT_GRID_NAMES[12];
        this.vzaFwdBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.vzaFwdBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.AATSR_TIE_POINT_GRID_NAMES[10];
        this.vaaBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.vaaBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        gridName = EnvisatConstants.AATSR_TIE_POINT_GRID_NAMES[14];
        this.vaaFwdBand = product.getTiePointGrid(gridName);
        SmacOperator.checkForNull(gridName, this.vaaFwdBand);
        this.logger.fine(LOG_MSG_LOADED + gridName);
        this.logger.info("... success");
    }

    private void createMask() {
        if (ObjectUtils.equalObjects((Object)this.sensorType, (Object)"MERIS")) {
            this.createMerisMask();
        } else {
            this.createAatsrMask();
        }
    }

    private void createMerisMask() {
        if ("".equalsIgnoreCase(this.maskExpression)) {
            this.maskExpression = DEFAULT_MERIS_FLAGS_VALUE;
            this.logger.warning("No mask expression defined");
            this.logger.warning("Using default mask expression: l1_flags.LAND_OCEAN and not (l1_flags.INVALID or l1_flags.BRIGHT)");
        } else {
            this.logger.info("Using mask expression: " + this.maskExpression);
        }
        Mask mask = this.sourceProduct.addMask(SMAC_MASK, this.maskExpression, "", Color.BLACK, 0.0);
        mask.setValidPixelExpression(this.maskExpression);
    }

    private void createAatsrMask() {
        Mask forwardMask;
        Mask mask;
        if ("".equalsIgnoreCase(this.maskExpression)) {
            mask = this.sourceProduct.addMask(SMAC_MASK, DEFAULT_NADIR_FLAGS_VALUE, "", Color.BLACK, 0.0);
            forwardMask = this.sourceProduct.addMask(SMAC_MASK_FORWARD, DEFAULT_FORWARD_FLAGS_VALUE, "", Color.BLACK, 0.0);
            this.logger.warning("No mask expression defined");
            this.logger.warning("Using default nadir mask expression: cloud_flags_nadir.LAND and not cloud_flags_nadir.CLOUDY");
            this.logger.warning("Using default forward mask expression: cloud_flags_fward.LAND and not cloud_flags_fward.CLOUDY");
        } else {
            mask = this.sourceProduct.addMask(SMAC_MASK, this.maskExpression, "", Color.BLACK, 0.0);
            forwardMask = this.sourceProduct.addMask(SMAC_MASK_FORWARD, this.maskExpressionForward, "", Color.BLACK, 0.0);
            this.logger.info("Using nadir mask expression: " + this.maskExpression);
            this.logger.info("Using forward mask expression: " + this.maskExpressionForward);
        }
        mask.setValidPixelExpression(this.maskExpression);
        forwardMask.setValidPixelExpression(this.maskExpressionForward);
    }

    private Path initAuxdataInstallDir() {
        return SystemUtils.getAuxDataPath().resolve("smac").toAbsolutePath();
    }

    private void createOutputProduct() throws IOException {
        String productType = this.sourceProduct.getProductType() + "_SMAC";
        String productName = this.sourceProduct.getName() + "_SMAC";
        int sceneWidth = this.sourceProduct.getSceneRasterWidth();
        int sceneHeight = this.sourceProduct.getSceneRasterHeight();
        this.targetProduct = new Product(productName, productType, sceneWidth, sceneHeight);
        if (ObjectUtils.equalObjects((Object)this.sensorType, (Object)"MERIS")) {
            this.addBandsToOutput("Atmosphere corrected MERIS band ", true);
        } else {
            this.addBandsToOutput("Atmosphere corrected band ", false);
        }
        ProductUtils.copyTiePointGrids((Product)this.sourceProduct, (Product)this.targetProduct);
        ProductUtils.copyFlagBands((Product)this.sourceProduct, (Product)this.targetProduct, (boolean)true);
        ProductUtils.copyBand((String)"corr_latitude", (Product)this.sourceProduct, (Product)this.targetProduct, (boolean)true);
        ProductUtils.copyBand((String)"corr_longitude", (Product)this.sourceProduct, (Product)this.targetProduct, (boolean)true);
        ProductUtils.copyBand((String)"altitude", (Product)this.sourceProduct, (Product)this.targetProduct, (boolean)true);
        ProductUtils.copyGeoCoding((Product)this.sourceProduct, (Product)this.targetProduct);
    }

    private void addBandsToOutput(String description, boolean convertMerisName) {
        for (Band inBand : this.inputBandList) {
            String bandUnit;
            String newBandName;
            if (convertMerisName) {
                newBandName = SmacOperator.convertMerisBandName(inBand.getName(), this.bandNameMapping);
                bandUnit = "dl";
            } else {
                newBandName = inBand.getName();
                bandUnit = inBand.getUnit();
            }
            Band outBand = new Band(newBandName, inBand.getGeophysicalDataType(), inBand.getRasterWidth(), inBand.getRasterHeight());
            outBand.setUnit(bandUnit);
            outBand.setDescription(description + inBand.getName());
            ProductUtils.copySpectralBandProperties((Band)inBand, (Band)outBand);
            this.targetProduct.addBand(outBand);
        }
    }

    static String convertMerisBandName(String bandName, Map<String, String> bandNameMapping) {
        String outBandName = merisBandPrefix;
        int blankIndex = bandName.indexOf(95);
        if (blankIndex > 0) {
            outBandName = outBandName + bandName.substring(blankIndex, bandName.length());
        }
        bandNameMapping.put(outBandName, bandName);
        return outBandName;
    }

    static String revertMerisBandName(String targetBandName, Map<String, String> targetToSourceMapping) {
        for (String targetName : targetToSourceMapping.keySet()) {
            if (!targetName.equals(targetBandName)) continue;
            return targetToSourceMapping.get(targetName);
        }
        return targetBandName;
    }

    private void processMerisWithADS(Band spectralBand, SourceData sourceData, Tile targetTile, Rectangle targetRectangle, SmacAlgorithm algorithm) throws IOException {
        if (!this.setBandCoefficients(spectralBand.getName(), algorithm)) {
            this.logger.severe(String.format("Sensor coefficient file for spectral band '%s' not found!", spectralBand.getName()));
            return;
        }
        float[] toa = RsMathUtils.radianceToReflectance((float[])sourceData.toa, (float[])sourceData.sza, (float)spectralBand.getSolarFlux(), null);
        float[] press = RsMathUtils.simpleBarometric((float[])sourceData.press, (float[])sourceData.elev, null);
        float[] uo3 = SmacOperator.dobsonToCmAtm(sourceData.uo3);
        float[] uh2o = SmacOperator.relativeHumidityTogcm2(sourceData.uh2o);
        Mask mask = (Mask)this.sourceProduct.getMaskGroup().get(SMAC_MASK);
        int i = 0;
        for (int absY = targetRectangle.y; absY < targetRectangle.y + targetRectangle.height; ++absY) {
            this.checkForCancellation();
            for (int absX = targetRectangle.x; absX < targetRectangle.x + targetRectangle.width; ++absX) {
                sourceData.process[i] = mask.getSampleInt(absX, absY) != 0;
                ++i;
            }
        }
        float[] toa_corr = new float[toa.length];
        toa_corr = algorithm.run(sourceData.sza, sourceData.saa, sourceData.vza, sourceData.vaa, sourceData.taup550, uh2o, uo3, press, sourceData.process, this.invalidPixel.floatValue(), toa, toa_corr);
        targetTile.setSamples(toa_corr);
    }

    private void processMeris(Band spectralBand, SourceData sourceData, Tile targetTile, Rectangle targetRectangle, SmacAlgorithm algorithm) throws IOException {
        if (!this.setBandCoefficients(spectralBand.getName(), algorithm)) {
            this.logger.severe("Sensor coefficient file for spectral band '" + spectralBand.getName() + "' not found!");
            return;
        }
        float[] reflectances = RsMathUtils.radianceToReflectance((float[])sourceData.toa, (float[])sourceData.sza, (float)spectralBand.getSolarFlux(), null);
        Mask mask = (Mask)this.sourceProduct.getMaskGroup().get(SMAC_MASK);
        int i = 0;
        for (int absY = targetRectangle.y; absY < targetRectangle.y + targetRectangle.height; ++absY) {
            this.checkForCancellation();
            for (int absX = targetRectangle.x; absX < targetRectangle.x + targetRectangle.width; ++absX) {
                sourceData.process[i] = mask.getSampleInt(absX, absY) != 0;
                ++i;
            }
        }
        float[] toa_corr = new float[reflectances.length];
        toa_corr = algorithm.run(sourceData.sza, sourceData.saa, sourceData.vza, sourceData.vaa, sourceData.taup550, sourceData.uh2o, sourceData.uo3, sourceData.press, sourceData.process, this.invalidPixel.floatValue(), reflectances, toa_corr);
        targetTile.setSamples(toa_corr);
    }

    private void processAatsr(String bandName, SourceData sourceData, Tile targetTile, Rectangle targetRectangle, SmacAlgorithm algorithm) throws IOException {
        Mask mask;
        float[] sza;
        float[] vza;
        if (!this.setBandCoefficients(bandName, algorithm)) {
            this.logger.severe("Sensor coefficient file for spectral band '" + bandName + "' not found!");
            return;
        }
        boolean isForwardBand = bandName.contains("fward");
        if (isForwardBand) {
            vza = RsMathUtils.elevationToZenith((float[])sourceData.vzaFwd, null);
            sza = RsMathUtils.elevationToZenith((float[])sourceData.szaFwd, null);
            mask = (Mask)this.sourceProduct.getMaskGroup().get(SMAC_MASK_FORWARD);
        } else {
            vza = RsMathUtils.elevationToZenith((float[])sourceData.vza, null);
            sza = RsMathUtils.elevationToZenith((float[])sourceData.sza, null);
            mask = (Mask)this.sourceProduct.getMaskGroup().get(SMAC_MASK);
        }
        int i = 0;
        for (int absY = targetRectangle.y; absY < targetRectangle.y + targetRectangle.height; ++absY) {
            this.checkForCancellation();
            for (int absX = targetRectangle.x; absX < targetRectangle.x + targetRectangle.width; ++absX) {
                sourceData.process[i] = mask.getSampleInt(absX, absY) != 0;
                ++i;
            }
        }
        float[] toa_corr = new float[sourceData.toa.length];
        toa_corr = algorithm.run(sourceData.sza, sourceData.saa, vza, sza, sourceData.taup550, sourceData.uh2o, sourceData.uo3, sourceData.press, sourceData.process, this.invalidPixel.floatValue(), sourceData.toa, toa_corr);
        targetTile.setSamples(toa_corr);
    }

    private boolean setBandCoefficients(String bandName, SmacAlgorithm algorithm) {
        if (this.coefficients.containsKey(bandName)) {
            algorithm.setSensorCoefficients(this.coefficients.get(bandName));
            return true;
        }
        SensorCoefficientFile coeff = new SensorCoefficientFile();
        boolean handleError = false;
        boolean success = false;
        try {
            URL url = this.getSensorCoefficientManager().getCoefficientFile(this.sensorType, bandName, this.aerosolType);
            if (url == null) {
                handleError = true;
            } else {
                coeff.readFile(new File(url.toURI()).getAbsolutePath());
                this.logger.info("Loaded sensor coefficient file " + url.getFile());
                algorithm.setSensorCoefficients(coeff);
                this.coefficients.put(bandName, coeff);
                success = true;
            }
        }
        catch (IOException | URISyntaxException e) {
            handleError = true;
            this.logger.severe(e.getMessage());
        }
        if (handleError) {
            this.logger.severe("Unable to load sensor coefficients for band " + bandName);
        }
        return success;
    }

    private SensorCoefficientManager getSensorCoefficientManager() {
        SensorCoefficientManager coeffMgr = null;
        try {
            coeffMgr = new SensorCoefficientManager(this.auxdataInstallDir.toUri().toURL());
            this.logger.fine("Using auxiliary data path: " + this.auxdataInstallDir.toString());
        }
        catch (IOException e) {
            this.logger.severe("Error reading coefficients from: " + this.auxdataInstallDir.toString());
            this.logger.severe(e.getMessage());
            this.logger.log(Level.FINE, e.getMessage(), e);
        }
        return coeffMgr;
    }

    private static float[] dobsonToCmAtm(float[] du) {
        Assert.notNull((Object)du, (String)"du");
        for (int n = 0; n < du.length; ++n) {
            du[n] = du[n] * 0.001f;
        }
        return du;
    }

    private static float[] relativeHumidityTogcm2(float[] relHum) {
        Assert.notNull((Object)relHum, (String)"relHum");
        for (int n = 0; n < relHum.length; ++n) {
            relHum[n] = 0.07f * relHum[n];
        }
        return relHum;
    }

    private class SourceData {
        float[] sza;
        float[] saa;
        float[] vza;
        float[] vaa;
        float[] szaFwd;
        float[] saaFwd;
        float[] vzaFwd;
        float[] vaaFwd;
        float[] uh2o;
        float[] uo3;
        float[] press;
        float[] elev;
        float[] taup550;
        boolean[] process;
        float[] toa;

        private SourceData() {
        }
    }

    public static class Spi
    extends OperatorSpi {
        public Spi() {
            super(SmacOperator.class);
        }
    }
}

