/*
 * Decompiled with CFR 0.152.
 */
package org.esa.s3tbx.meris.radiometry;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.TreeMap;
import org.esa.s3tbx.meris.radiometry.calibration.CalibrationAlgorithm;
import org.esa.s3tbx.meris.radiometry.calibration.Resolution;
import org.esa.s3tbx.meris.radiometry.equalization.EqualizationAlgorithm;
import org.esa.s3tbx.meris.radiometry.equalization.ReprocessingVersion;
import org.esa.s3tbx.meris.radiometry.smilecorr.SmileCorrectionAlgorithm;
import org.esa.s3tbx.meris.radiometry.smilecorr.SmileCorrectionAuxdata;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.gpf.OperatorException;
import org.esa.snap.core.gpf.OperatorSpi;
import org.esa.snap.core.gpf.annotations.OperatorMetadata;
import org.esa.snap.core.gpf.annotations.Parameter;
import org.esa.snap.core.gpf.annotations.SourceProduct;
import org.esa.snap.core.gpf.pointop.ProductConfigurer;
import org.esa.snap.core.gpf.pointop.Sample;
import org.esa.snap.core.gpf.pointop.SampleOperator;
import org.esa.snap.core.gpf.pointop.SourceSampleConfigurer;
import org.esa.snap.core.gpf.pointop.TargetSampleConfigurer;
import org.esa.snap.core.gpf.pointop.WritableSample;
import org.esa.snap.core.util.ProductUtils;
import org.esa.snap.core.util.math.RsMathUtils;
import org.esa.snap.dataio.envisat.EnvisatConstants;

@OperatorMetadata(alias="Meris.CorrectRadiometry", description="Performs radiometric corrections on MERIS L1b data products.", authors="Marc Bouvet (ESTEC); Marco Peters, Ralf Quast, Thomas Storm, Marco Zuehlke (Brockmann Consult)", copyright="(c) 2015 by Brockmann Consult", category="Optical/Pre-Processing", version="1.2")
public class MerisRadiometryCorrectionOp
extends SampleOperator {
    private static final String UNIT_DL = "dl";
    private static final double RAW_SATURATION_THRESHOLD = 65435.0;
    private static final String DEFAULT_SOURCE_RAC_RESOURCE = "MER_RAC_AXVIEC20050708_135553_20021224_121445_20041213_220000";
    private static final String DEFAULT_TARGET_RAC_RESOURCE = "MER_RAC_AXVACR20091016_154511_20021224_121445_20041213_220000";
    private static final int INVALID_BIT_INDEX = 7;
    private static final int LAND_BIT_INDEX = 4;
    @Parameter(defaultValue="true", label="Perform calibration", description="Whether to perform the calibration.")
    private boolean doCalibration;
    @Parameter(label="Source radiometric correction file (optional)", description="The radiometric correction auxiliary file for the source product. The default 'MER_RAC_AXVIEC20050708_135553_20021224_121445_20041213_220000'")
    private File sourceRacFile;
    @Parameter(label="Target radiometric correction file (optional)", description="The radiometric correction auxiliary file for the target product. The default 'MER_RAC_AXVACR20091016_154511_20021224_121445_20041213_220000'")
    private File targetRacFile;
    @Parameter(defaultValue="true", label="Perform Smile-effect correction", description="Whether to perform Smile-effect correction.")
    private boolean doSmile;
    @Parameter(defaultValue="true", label="Perform equalisation", description="Perform removal of detector-to-detector systematic radiometric differences in MERIS L1b data products.")
    private boolean doEqualization;
    @Parameter(label="Reprocessing version", valueSet={"AUTO_DETECT", "REPROCESSING_2", "REPROCESSING_3"}, defaultValue="AUTO_DETECT", description="The version of the reprocessing the product comes from. Is only used if equalisation is enabled.")
    private ReprocessingVersion reproVersion;
    @Parameter(defaultValue="false", label="Perform radiance-to-reflectance conversion", description="Whether to perform radiance-to-reflectance conversion. When selecting ENVISAT as target format, the radiance to reflectance conversion can not be performed.")
    private boolean doRadToRefl;
    @SourceProduct(alias="source", label="Name", description="The source product.", bands={"l1_flags", "detector_index", "radiance_1", "radiance_2", "radiance_3", "radiance_4", "radiance_5", "radiance_6", "radiance_7", "radiance_8", "radiance_9", "radiance_10", "radiance_11", "radiance_12", "radiance_13", "radiance_14", "radiance_15"})
    private Product sourceProduct;
    private transient CalibrationAlgorithm calibrationAlgorithm;
    private transient EqualizationAlgorithm equalizationAlgorithm;
    private transient SmileCorrectionAlgorithm smileCorrAlgorithm;
    private transient int detectorIndexSampleIndex;
    private transient int sunZenithAngleSampleIndex;
    private transient int flagBandIndex;
    private transient int currentPixel = 0;
    private Map<Integer, Double> bandIndexToMaxValueMap;

    protected void prepareInputs() throws OperatorException {
        super.prepareInputs();
        this.validateSourceProduct();
        this.initAlgorithms();
    }

    protected void configureSourceSamples(SourceSampleConfigurer sampleConfigurer) {
        int i = -1;
        for (Band band : this.getSourceProduct().getBands()) {
            int spectralBandIndex = band.getSpectralBandIndex();
            if (spectralBandIndex == -1) continue;
            sampleConfigurer.defineSample(spectralBandIndex, band.getName());
            if (spectralBandIndex <= i) continue;
            i = spectralBandIndex;
        }
        this.detectorIndexSampleIndex = i + 1;
        if (this.doCalibration || this.doSmile || this.doEqualization) {
            sampleConfigurer.defineSample(this.detectorIndexSampleIndex, "detector_index");
        }
        this.sunZenithAngleSampleIndex = i + 2;
        if (this.doRadToRefl) {
            sampleConfigurer.defineSample(this.sunZenithAngleSampleIndex, "sun_zenith");
        }
        this.flagBandIndex = i + 3;
        if (this.doSmile) {
            sampleConfigurer.defineSample(this.flagBandIndex, "l1_flags");
        }
    }

    protected void configureTargetSamples(TargetSampleConfigurer sampleConfigurer) {
        for (Band band : this.getTargetProduct().getBands()) {
            int spectralBandIndex = band.getSpectralBandIndex();
            if (spectralBandIndex == -1) continue;
            sampleConfigurer.defineSample(spectralBandIndex, band.getName());
        }
    }

    protected void configureTargetProduct(ProductConfigurer productConfigurer) {
        productConfigurer.copyMetadata();
        productConfigurer.copyTimeCoding();
        Product targetProduct = productConfigurer.getTargetProduct();
        targetProduct.setName(this.getSourceProduct().getName());
        if (this.doRadToRefl) {
            targetProduct.setProductType(String.format("%s_REFL", this.getSourceProduct().getProductType()));
            targetProduct.setAutoGrouping("reflec");
        } else {
            targetProduct.setProductType(this.getSourceProduct().getProductType());
            targetProduct.setAutoGrouping("radiance");
        }
        targetProduct.setDescription("MERIS L1b Radiometric Correction");
        this.bandIndexToMaxValueMap = new TreeMap<Integer, Double>();
        Band[] bands = this.getSourceProduct().getBands();
        int bandIndex = 0;
        for (Band sourceBand : bands) {
            double scalingOffset;
            double scalingFactor;
            String unit;
            int dataType;
            String targetBandDescription;
            String targetBandName;
            if (sourceBand.getSpectralBandIndex() == -1) continue;
            if (this.doRadToRefl) {
                targetBandName = sourceBand.getName().replace("radiance", "reflec");
                targetBandDescription = "Radiometry-corrected TOA reflectance";
                dataType = 30;
                unit = UNIT_DL;
                scalingFactor = 1.0;
                scalingOffset = 0.0;
            } else {
                targetBandName = sourceBand.getName();
                targetBandDescription = "Radiometry-corrected TOA radiance";
                dataType = sourceBand.getDataType();
                unit = sourceBand.getUnit();
                scalingFactor = sourceBand.getScalingFactor();
                scalingOffset = sourceBand.getScalingOffset();
            }
            Band targetBand = targetProduct.addBand(targetBandName, dataType);
            targetBand.setScalingFactor(scalingFactor);
            targetBand.setScalingOffset(scalingOffset);
            this.bandIndexToMaxValueMap.put(bandIndex++, targetBand.scale(65535.0));
            targetBand.setDescription(targetBandDescription);
            targetBand.setUnit(unit);
            targetBand.setValidPixelExpression(sourceBand.getValidPixelExpression());
            ProductUtils.copySpectralBandProperties((Band)sourceBand, (Band)targetBand);
        }
        productConfigurer.copyTiePointGrids(new String[0]);
        productConfigurer.copyGeoCoding();
        for (Band sourceBand : this.getSourceProduct().getBands()) {
            if (sourceBand.getSpectralBandIndex() != -1 || targetProduct.containsBand(sourceBand.getName())) continue;
            productConfigurer.copyBands(new String[]{sourceBand.getName()});
        }
        productConfigurer.copyMasks();
    }

    protected void computeSample(int x, int y, Sample[] sourceSamples, WritableSample targetSample) {
        Sample flagSample;
        boolean invalid;
        boolean isValidDetectorIndex;
        this.checkCancellation();
        int bandIndex = targetSample.getIndex();
        Sample sourceRadiance = sourceSamples[bandIndex];
        int detectorIndex = -1;
        if (this.doCalibration || this.doSmile || this.doEqualization) {
            detectorIndex = sourceSamples[this.detectorIndexSampleIndex].getInt();
        }
        double value = sourceRadiance.getDouble();
        boolean bl = isValidDetectorIndex = detectorIndex >= 0;
        if (this.doCalibration && isValidDetectorIndex && value < sourceRadiance.getNode().scale(65435.0)) {
            value = this.calibrationAlgorithm.calibrate(bandIndex, detectorIndex, value);
        }
        if (this.doSmile && !(invalid = (flagSample = sourceSamples[this.flagBandIndex]).getBit(7)) && detectorIndex != -1) {
            boolean land = flagSample.getBit(4);
            double[] sourceValues = new double[15];
            for (int i = 0; i < sourceValues.length; ++i) {
                sourceValues[i] = sourceSamples[i].getDouble();
            }
            value = this.smileCorrAlgorithm.correct(bandIndex, detectorIndex, sourceValues, land);
        }
        if (this.doRadToRefl) {
            float solarFlux = ((Band)sourceRadiance.getNode()).getSolarFlux();
            float sunZenithSample = sourceSamples[this.sunZenithAngleSampleIndex].getFloat();
            value = RsMathUtils.radianceToReflectance((float)((float)value), (float)sunZenithSample, (float)solarFlux);
        }
        if (this.doEqualization && isValidDetectorIndex) {
            value = this.equalizationAlgorithm.performEqualization(value, bandIndex, detectorIndex);
        }
        double croppedValue = Math.min(this.bandIndexToMaxValueMap.get(bandIndex), value);
        targetSample.set(croppedValue);
    }

    private void initAlgorithms() {
        String productType = this.getSourceProduct().getProductType();
        if (this.doCalibration) {
            InputStream sourceRacStream = null;
            InputStream targetRacStream = null;
            try {
                sourceRacStream = MerisRadiometryCorrectionOp.openStream(this.sourceRacFile, DEFAULT_SOURCE_RAC_RESOURCE);
                targetRacStream = MerisRadiometryCorrectionOp.openStream(this.targetRacFile, DEFAULT_TARGET_RAC_RESOURCE);
                double cntJD = 0.5 * (this.getSourceProduct().getStartTime().getMJD() + this.getSourceProduct().getEndTime().getMJD());
                Resolution resolution = productType.contains("RR") ? Resolution.RR : Resolution.FR;
                this.calibrationAlgorithm = new CalibrationAlgorithm(resolution, cntJD, sourceRacStream, targetRacStream);
            }
            catch (IOException e) {
                throw new OperatorException((Throwable)e);
            }
            finally {
                try {
                    if (sourceRacStream != null) {
                        sourceRacStream.close();
                    }
                    if (targetRacStream != null) {
                        targetRacStream.close();
                    }
                }
                catch (IOException iOException) {}
            }
            this.reproVersion = ReprocessingVersion.REPROCESSING_3;
        }
        if (this.doSmile) {
            try {
                this.smileCorrAlgorithm = new SmileCorrectionAlgorithm(SmileCorrectionAuxdata.loadAuxdata(productType));
            }
            catch (Exception e) {
                throw new OperatorException((Throwable)e);
            }
        }
        if (this.doEqualization) {
            try {
                this.equalizationAlgorithm = new EqualizationAlgorithm(this.getSourceProduct(), this.reproVersion);
            }
            catch (Exception e) {
                throw new OperatorException((Throwable)e);
            }
        }
    }

    private static InputStream openStream(File racFile, String defaultRacResource) throws FileNotFoundException {
        if (racFile == null) {
            return CalibrationAlgorithm.class.getResourceAsStream(defaultRacResource);
        }
        return new FileInputStream(racFile);
    }

    private void validateSourceProduct() throws OperatorException {
        boolean isReprocessing2;
        if (!EnvisatConstants.MERIS_L1_TYPE_PATTERN.matcher(this.getSourceProduct().getProductType()).matches()) {
            String msg = String.format("Source product must be of type MERIS Level 1b. Product type is: '%s'", this.getSourceProduct().getProductType());
            this.getLogger().warning(msg);
        }
        boolean bl = isReprocessing2 = this.reproVersion == ReprocessingVersion.REPROCESSING_2 || ReprocessingVersion.autoDetect(this.getSourceProduct()) == ReprocessingVersion.REPROCESSING_2;
        if (!isReprocessing2 && this.doCalibration) {
            this.getLogger().warning("Skipping calibration. Source product is already of 3rd reprocessing.");
            this.doCalibration = false;
        }
        if ((this.doCalibration || this.doEqualization) && this.getSourceProduct().getStartTime() == null) {
            throw new OperatorException("Source product must have a start time");
        }
        if (this.doCalibration && this.getSourceProduct().getEndTime() == null) {
            throw new OperatorException("Source product must have an end time");
        }
        String msgPatternMissingBand = "Source product must contain '%s'.";
        if (this.doSmile) {
            if (!this.getSourceProduct().containsBand("detector_index")) {
                throw new OperatorException(String.format("Source product must contain '%s'.", "detector_index"));
            }
            if (!this.getSourceProduct().containsBand("l1_flags")) {
                throw new OperatorException(String.format("Source product must contain '%s'.", "l1_flags"));
            }
            if (!this.getSourceProduct().getBand("l1_flags").isFlagBand()) {
                throw new OperatorException(String.format("Flag-coding is missing for band '%s' ", "l1_flags"));
            }
        }
        if (this.doEqualization && !this.getSourceProduct().containsBand("detector_index")) {
            throw new OperatorException(String.format("Source product must contain '%s'.", "detector_index"));
        }
        if (this.doRadToRefl && !this.getSourceProduct().containsRasterDataNode("sun_zenith")) {
            throw new OperatorException(String.format("Source product must contain '%s'.", "sun_zenith"));
        }
    }

    private void checkCancellation() {
        if (this.currentPixel % 1000 == 0) {
            this.checkForCancellation();
            this.currentPixel = 0;
        }
        ++this.currentPixel;
    }

    public static class Spi
    extends OperatorSpi {
        public Spi() {
            super(MerisRadiometryCorrectionOp.class);
        }
    }
}

