/*
 * Decompiled with CFR 0.152.
 */
package org.esa.s3tbx.dataio.ceos.avnir2;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import javax.imageio.stream.FileImageInputStream;
import javax.imageio.stream.ImageInputStream;
import org.esa.s3tbx.dataio.ceos.CeosHelper;
import org.esa.s3tbx.dataio.ceos.IllegalCeosFormatException;
import org.esa.s3tbx.dataio.ceos.avnir2.Avnir2ImageFile;
import org.esa.s3tbx.dataio.ceos.avnir2.Avnir2LeaderFile;
import org.esa.s3tbx.dataio.ceos.avnir2.Avnir2SupplementalFile;
import org.esa.s3tbx.dataio.ceos.avnir2.Avnir2TrailerFile;
import org.esa.s3tbx.dataio.ceos.avnir2.Avnir2VolumeDirectoryFile;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.FXYGeoCoding;
import org.esa.snap.core.datamodel.GeoCoding;
import org.esa.snap.core.datamodel.GeoPos;
import org.esa.snap.core.datamodel.MapGeoCoding;
import org.esa.snap.core.datamodel.MetadataAttribute;
import org.esa.snap.core.datamodel.MetadataElement;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.core.dataop.maptransf.Datum;
import org.esa.snap.core.dataop.maptransf.Ellipsoid;
import org.esa.snap.core.dataop.maptransf.MapInfo;
import org.esa.snap.core.dataop.maptransf.MapProjection;
import org.esa.snap.core.dataop.maptransf.MapTransform;
import org.esa.snap.core.dataop.maptransf.MapTransformFactory;
import org.esa.snap.core.dataop.maptransf.StereographicDescriptor;
import org.esa.snap.core.dataop.maptransf.UTM;
import org.esa.snap.core.util.Debug;
import org.esa.snap.core.util.Guardian;
import org.esa.snap.core.util.StringUtils;
import org.esa.snap.core.util.TreeNode;
import org.esa.snap.core.util.math.FXYSum;

class Avnir2ProductDirectory {
    private static final int METER_PER_KILOMETER = 1000;
    private final File baseDir;
    private Avnir2VolumeDirectoryFile volumeDirectoryFile;
    private Avnir2ImageFile[] imageFiles;
    private Map<Band, Avnir2ImageFile> imageFileMap;
    private Avnir2LeaderFile leaderFile;
    private Avnir2TrailerFile trailerFile;
    private Avnir2SupplementalFile supplementalFile = null;
    private final int sceneWidth;
    private final int sceneHeight;

    Avnir2ProductDirectory(File dir) throws IOException, IllegalCeosFormatException {
        File supplementalFile;
        Guardian.assertNotNull((String)"dir", (Object)dir);
        this.baseDir = dir;
        this.volumeDirectoryFile = new Avnir2VolumeDirectoryFile(this.baseDir);
        this.leaderFile = new Avnir2LeaderFile(this.createInputStream(this.volumeDirectoryFile.getLeaderFileName()));
        this.trailerFile = new Avnir2TrailerFile(this.createInputStream(this.volumeDirectoryFile.getTrailerFileName()));
        if (!this.leaderFile.getProductLevel().equalsIgnoreCase("1B2") && (supplementalFile = new File(this.baseDir, this.volumeDirectoryFile.getSupplementalFileName())).exists()) {
            this.supplementalFile = new Avnir2SupplementalFile(this.createInputStream(this.volumeDirectoryFile.getSupplementalFileName()));
        }
        String[] imageFileNames = this.volumeDirectoryFile.getImageFileNames();
        this.imageFiles = new Avnir2ImageFile[imageFileNames.length];
        for (int i = 0; i < this.imageFiles.length; ++i) {
            this.imageFiles[i] = new Avnir2ImageFile(this.createInputStream(imageFileNames[i]));
        }
        this.sceneWidth = this.imageFiles[0].getRasterWidth();
        this.sceneHeight = this.imageFiles[0].getRasterHeight();
        this.assertSameWidthAndHeightForAllImages();
    }

    Product createProduct() throws IOException, IllegalCeosFormatException {
        Product product = new Product(this.volumeDirectoryFile.getProductName(), this.getProductType(), this.sceneWidth, this.sceneHeight);
        product.setFileLocation(this.baseDir);
        this.imageFileMap = new HashMap<Band, Avnir2ImageFile>(this.imageFiles.length);
        for (Avnir2ImageFile avnir2ImageFile : this.imageFiles) {
            Band band = this.createBand(avnir2ImageFile);
            product.addBand(band);
            this.imageFileMap.put(band, avnir2ImageFile);
        }
        product.setStartTime(this.getScanTimeUTC(0));
        product.setEndTime(this.getScanTimeUTC(this.sceneHeight - 1));
        product.setDescription(this.getProductDescription());
        this.addGeoCoding(product);
        this.addMetaData(product);
        return product;
    }

    TreeNode<File> getProductComponents() {
        try {
            String[] imageFileNames;
            File parentDir = this.baseDir.getParentFile();
            if (parentDir == null) {
                throw new IllegalStateException("Could not retrieve the parent directory of '" + this.baseDir.getAbsolutePath() + "'.");
            }
            TreeNode root = new TreeNode(parentDir.getCanonicalPath());
            root.setContent((Object)parentDir);
            TreeNode dir = new TreeNode(this.baseDir.getName());
            dir.setContent((Object)this.baseDir);
            root.addChild(dir);
            File volumeFile = CeosHelper.getVolumeFile(this.baseDir);
            this.addFileToDir((TreeNode<File>)dir, volumeFile);
            this.addFileToDir((TreeNode<File>)dir, new File(this.volumeDirectoryFile.getLeaderFileName()));
            this.addFileToDir((TreeNode<File>)dir, new File(this.volumeDirectoryFile.getTrailerFileName()));
            String supplemental = this.volumeDirectoryFile.getSupplementalFileName();
            if (StringUtils.isNotNullAndNotEmpty((String)supplemental)) {
                this.addFileToDir((TreeNode<File>)dir, new File(supplemental));
            }
            for (String imageFileName : imageFileNames = this.volumeDirectoryFile.getImageFileNames()) {
                this.addFileToDir((TreeNode<File>)dir, new File(imageFileName));
            }
            return root;
        }
        catch (IOException e) {
            return null;
        }
        catch (IllegalCeosFormatException e) {
            return null;
        }
    }

    private void addFileToDir(TreeNode<File> dir, File file) {
        TreeNode fileNode = new TreeNode(file.getName());
        fileNode.setContent((Object)file);
        dir.addChild(fileNode);
    }

    private String getProductType() throws IOException, IllegalCeosFormatException {
        return "AV2_" + this.leaderFile.getProductLevel();
    }

    private void addGeoCoding(Product product) throws IllegalCeosFormatException, IOException {
        String usedProjection = this.leaderFile.getUsedProjection();
        if ("NNNNN".equalsIgnoreCase(usedProjection)) {
            Band[] bands;
            for (Band band : bands = product.getBands()) {
                Avnir2ImageFile imageFile = this.getImageFile(band);
                int bandIndex = imageFile.getBandIndex();
                double[][] uncorrectedCoeffs = this.leaderFile.getUncorrectedTransformationCoeffs(bandIndex);
                FXYSum.Cubic funcLat = new FXYSum.Cubic(CeosHelper.sortToFXYSumOrder(uncorrectedCoeffs[0]));
                FXYSum.Cubic funcLon = new FXYSum.Cubic(CeosHelper.sortToFXYSumOrder(uncorrectedCoeffs[1]));
                FXYSum.Cubic funcX = new FXYSum.Cubic(CeosHelper.sortToFXYSumOrder(uncorrectedCoeffs[2]));
                FXYSum.Cubic funcY = new FXYSum.Cubic(CeosHelper.sortToFXYSumOrder(uncorrectedCoeffs[3]));
                FXYGeoCoding gc = new FXYGeoCoding(0.0f, 0.0f, 1.0f, 1.0f, (FXYSum)funcX, (FXYSum)funcY, (FXYSum)funcLat, (FXYSum)funcLon, Datum.ITRF_97);
                band.setGeoCoding((GeoCoding)gc);
            }
        } else if ("YNNNN".equalsIgnoreCase(usedProjection)) {
            int zoneIndex = (int)this.leaderFile.getUTMZoneIndex();
            boolean isSouth = this.leaderFile.isUTMSouthHemisphere();
            double easting = this.leaderFile.getUTMEasting() * 1000.0;
            double northing = this.leaderFile.getUTMNorthing() * 1000.0;
            double pixelSizeX = this.leaderFile.getNominalInterPixelDistance();
            double pixelSizeY = this.leaderFile.getNominalInterLineDistance();
            float orientationAngle = (float)this.leaderFile.getUTMOrientationAngle();
            MapInfo mapInfo = new MapInfo((MapProjection)UTM.createProjection((int)(zoneIndex - 1), (boolean)isSouth), (float)this.sceneWidth * 0.5f, (float)this.sceneHeight * 0.5f, (float)easting, (float)northing, (float)pixelSizeX, (float)pixelSizeY, Datum.ITRF_97);
            mapInfo.setOrientation(-orientationAngle);
            mapInfo.setSceneWidth(this.sceneWidth);
            mapInfo.setSceneHeight(this.sceneHeight);
            product.setSceneGeoCoding((GeoCoding)new MapGeoCoding(mapInfo));
        } else if ("NNNNY".equalsIgnoreCase(usedProjection)) {
            double[] parameterValues = StereographicDescriptor.PARAMETER_DEFAULT_VALUES;
            parameterValues[0] = Ellipsoid.GRS_80.getSemiMajor();
            parameterValues[1] = Ellipsoid.GRS_80.getSemiMinor();
            GeoPos psReferencePoint = this.leaderFile.getPSReferencePoint();
            GeoPos psProjectionOrigin = this.leaderFile.getPSProjectionOrigin();
            parameterValues[2] = psProjectionOrigin.getLat();
            parameterValues[3] = psReferencePoint.getLon();
            MapTransform transform = MapTransformFactory.createTransform((String)"Stereographic", (double[])parameterValues);
            MapProjection projection = new MapProjection("Stereographic", transform);
            double pixelSizeX = this.leaderFile.getNominalInterPixelDistance();
            double pixelSizeY = this.leaderFile.getNominalInterLineDistance();
            double easting = this.leaderFile.getPSXCoordinate() * 1000.0;
            double northing = this.leaderFile.getPSYCoordinate() * 1000.0;
            int sceneRasterWidth = product.getSceneRasterWidth();
            int sceneRasterHeight = product.getSceneRasterHeight();
            MapInfo mapInfo = new MapInfo(projection, (float)sceneRasterWidth * 0.5f, (float)sceneRasterHeight * 0.5f, (float)easting, (float)northing, (float)pixelSizeX, (float)pixelSizeY, Datum.ITRF_97);
            mapInfo.setOrientation((float)this.leaderFile.getPSOrientationAngle());
            mapInfo.setSceneWidth(sceneRasterWidth);
            mapInfo.setSceneHeight(sceneRasterHeight);
            product.setSceneGeoCoding((GeoCoding)new MapGeoCoding(mapInfo));
        } else {
            Debug.trace((String)"Unknown map projection method. Could not create geo-coding.");
        }
    }

    Avnir2ImageFile getImageFile(Band band) throws IOException, IllegalCeosFormatException {
        return this.imageFileMap.get(band);
    }

    void close() throws IOException {
        for (int i = 0; i < this.imageFiles.length; ++i) {
            this.imageFiles[i].close();
            this.imageFiles[i] = null;
        }
        this.imageFiles = null;
        this.imageFileMap.clear();
        this.volumeDirectoryFile.close();
        this.volumeDirectoryFile = null;
        this.leaderFile.close();
        this.leaderFile = null;
        this.trailerFile.close();
        this.trailerFile = null;
        if (this.supplementalFile != null) {
            this.supplementalFile.close();
            this.supplementalFile = null;
        }
    }

    private Band createBand(Avnir2ImageFile avnir2ImageFile) throws IOException, IllegalCeosFormatException {
        Band band = new Band(avnir2ImageFile.getBandName(), 20, this.sceneWidth, this.sceneHeight);
        int bandIndex = avnir2ImageFile.getBandIndex();
        band.setSpectralBandIndex(bandIndex - 1);
        band.setSpectralWavelength(avnir2ImageFile.getSpectralWavelength());
        band.setSpectralBandwidth(avnir2ImageFile.getSpectralBandwidth());
        band.setUnit(avnir2ImageFile.getGeophysicalUnit());
        double scalingFactor = this.leaderFile.getAbsoluteCalibrationGain(bandIndex);
        double scalingOffset = this.leaderFile.getAbsoluteCalibrationOffset(bandIndex);
        band.setScalingFactor(scalingFactor);
        band.setScalingOffset(scalingOffset);
        band.setNoDataValueUsed(false);
        band.setDescription("Radiance band " + avnir2ImageFile.getBandIndex());
        return band;
    }

    private void addMetaData(Product product) throws IOException, IllegalCeosFormatException {
        MetadataElement metadata = new MetadataElement("SPH");
        metadata.addElement(this.leaderFile.getMapProjectionMetadata());
        metadata.addElement(this.leaderFile.getRadiometricMetadata());
        metadata.addElement(this.leaderFile.getPlatformMetadata());
        this.addSummaryMetadata(metadata);
        product.getMetadataRoot().addElement(metadata);
        MetadataElement volumeDescriptor = new MetadataElement("VOLUME_DESCRIPTOR");
        this.volumeDirectoryFile.assignMetadataTo(volumeDescriptor);
        product.getMetadataRoot().addElement(volumeDescriptor);
    }

    private void addSummaryMetadata(MetadataElement parent) throws IOException {
        MetadataElement summaryMetadata = new MetadataElement("Summary Information");
        Properties properties = new Properties();
        File file = new File(this.baseDir, "summary.txt");
        if (!file.exists()) {
            return;
        }
        properties.load(new FileInputStream(file));
        Set<Map.Entry<Object, Object>> unsortedEntries = properties.entrySet();
        TreeSet<Map.Entry<Object, Object>> sortedEntries = new TreeSet<Map.Entry<Object, Object>>(new Comparator(){

            public int compare(Object a, Object b) {
                Map.Entry entryA = (Map.Entry)a;
                Map.Entry entryB = (Map.Entry)b;
                return ((String)entryA.getKey()).compareTo((String)entryB.getKey());
            }
        });
        sortedEntries.addAll(unsortedEntries);
        for (Object e : sortedEntries) {
            Map.Entry entry = (Map.Entry)e;
            String data = (String)entry.getValue();
            String strippedData = data.substring(1, data.length() - 1);
            MetadataAttribute attribute = new MetadataAttribute((String)entry.getKey(), (ProductData)new ProductData.ASCII(strippedData), true);
            summaryMetadata.addAttribute(attribute);
        }
        parent.addElement(summaryMetadata);
    }

    private String getProductDescription() throws IOException, IllegalCeosFormatException {
        return "AVNIR-2 product Level " + this.leaderFile.getProductLevel();
    }

    private void assertSameWidthAndHeightForAllImages() throws IOException, IllegalCeosFormatException {
        for (int i = 0; i < this.imageFiles.length; ++i) {
            Avnir2ImageFile imageFile = this.imageFiles[i];
            Guardian.assertTrue((String)("_sceneWidth == imageFile[" + i + "].getRasterWidth()"), (this.sceneWidth == imageFile.getRasterWidth() ? 1 : 0) != 0);
            Guardian.assertTrue((String)("_sceneHeight == imageFile[" + i + "].getRasterHeight()"), (this.sceneHeight == imageFile.getRasterHeight() ? 1 : 0) != 0);
        }
    }

    private ProductData.UTC getScanTimeUTC(int lineNumber) throws IOException, IllegalCeosFormatException {
        Calendar imageEndDate = this.leaderFile.getDateImageWasTaken();
        Avnir2ImageFile firstImageFile = this.imageFiles[0];
        imageEndDate.set(14, firstImageFile.getTotalMillisInDayOfLine(lineNumber));
        int remainingMillis = imageEndDate.get(14);
        imageEndDate.set(14, 0);
        return ProductData.UTC.create((Date)imageEndDate.getTime(), (long)(firstImageFile.getMicrosecondsOfLine(lineNumber) + remainingMillis * 1000));
    }

    private ImageInputStream createInputStream(String fileName) throws IOException {
        return new FileImageInputStream(new File(this.baseDir, fileName));
    }
}

