/*
 * Decompiled with CFR 0.152.
 */
package org.esa.s1tbx.io.gamma;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Calendar;
import org.esa.s1tbx.io.gamma.GammaProductWriter;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.GeoPos;
import org.esa.snap.core.datamodel.MetadataElement;
import org.esa.snap.core.datamodel.PixelPos;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.core.datamodel.ProductNode;
import org.esa.snap.core.util.SystemUtils;
import org.esa.snap.core.util.io.FileUtils;
import org.esa.snap.engine_utilities.datamodel.AbstractMetadata;
import org.esa.snap.engine_utilities.datamodel.OrbitStateVector;
import org.esa.snap.engine_utilities.eo.GeoUtils;

class HeaderWriter {
    private final File outputFile;
    private final Product srcProduct;
    private final MetadataElement absRoot;
    private String baseFileName;
    private boolean isComplex;
    private boolean isCoregistered;
    private final GammaProductWriter writer;
    private ProductData.UTC dateDay;
    private static final String sep = ":\t";
    private static final String tab = "\t";
    private static final double daysToSeconds = 43200.0;

    HeaderWriter(GammaProductWriter writer, Product srcProduct, File userOutputFile) {
        this.writer = writer;
        this.srcProduct = srcProduct;
        this.isComplex = false;
        this.isCoregistered = false;
        this.absRoot = AbstractMetadata.getAbstractedMetadata((Product)srcProduct);
        if (this.absRoot != null) {
            try {
                this.isComplex = this.absRoot.getAttributeString("SAMPLE_TYPE").equals("COMPLEX");
                this.isCoregistered = AbstractMetadata.getAttributeBoolean((MetadataElement)this.absRoot, (String)"coregistered_stack");
            }
            catch (Exception e) {
                SystemUtils.LOG.severe("Unable to read metadata " + e.getMessage());
            }
        }
        this.outputFile = this.createParFile(userOutputFile);
        this.baseFileName = FileUtils.getFilenameWithoutExtension((File)this.outputFile);
        Calendar cal = srcProduct.getStartTime().getAsCalendar();
        String dateStr = "" + cal.get(5) + '-' + (cal.get(2) + 1) + '-' + cal.get(1);
        try {
            this.dateDay = ProductData.UTC.parse((String)dateStr, (String)"dd-MM-yyyy");
        }
        catch (Exception e) {
            this.dateDay = srcProduct.getStartTime();
        }
    }

    String getBaseFileName() {
        return this.baseFileName;
    }

    void writeParFile() throws IOException {
        String oldEOL = System.getProperty("line.separator");
        System.setProperty("line.separator", "\n");
        FileOutputStream out = new FileOutputStream(this.outputFile);
        try (PrintStream p = new PrintStream(out);){
            p.println("title:\t" + this.srcProduct.getName());
            p.println("sensor:\t" + this.absRoot.getAttributeString("MISSION"));
            p.println("date:\t" + this.writeDate());
            p.println("start_time:\t" + this.writeStartTime());
            p.println("center_time:\t" + this.writeCenterTime());
            p.println("end_time:\t" + this.writeEndTime());
            p.println("azimuth_line_time:\t" + this.absRoot.getAttributeString("line_time_interval"));
            p.println("range_samples:\t" + this.srcProduct.getSceneRasterWidth());
            p.println("azimuth_lines:\t" + this.srcProduct.getSceneRasterHeight());
            p.println("range_looks:\t" + this.absRoot.getAttributeInt("range_looks"));
            p.println("azimuth_looks:\t" + this.absRoot.getAttributeInt("azimuth_looks"));
            p.println("image_format:\t" + this.getDataType());
            p.println("image_geometry:\t" + this.writeImageGeometry());
            this.writeCenterLatLon(p);
            p.println("range_pixel_spacing:\t" + this.absRoot.getAttributeInt("range_spacing") + tab + "m");
            p.println("azimuth_pixel_spacing:\t" + this.absRoot.getAttributeInt("azimuth_spacing") + tab + "m");
            p.println("radar_frequency:\t" + this.absRoot.getAttributeString("radar_frequency") + tab + "Hz");
            p.println("prf:\t" + this.absRoot.getAttributeString("pulse_repetition_frequency") + tab + "Hz");
            p.println("azimuth_proc_bandwidth:\t" + this.absRoot.getAttributeString("azimuth_bandwidth") + tab + "Hz");
            this.writeEarthParams(p);
            p.println("near_range_slc:\t" + this.absRoot.getAttributeString("slant_range_to_first_pixel") + tab + "m");
            p.println("center_range_slc:\t" + this.absRoot.getAttributeString("slant_range_to_first_pixel") + tab + "m");
            p.println("far_range_slc:\t" + this.absRoot.getAttributeString("slant_range_to_first_pixel") + tab + "m");
            this.writeOrbitStateVectors(p);
            p.flush();
        }
        catch (Exception e) {
            throw new IOException("GammaWriter unable to write par file " + e.getMessage());
        }
        finally {
            System.setProperty("line.separator", oldEOL);
        }
    }

    private String writeDate() {
        Calendar cal = this.srcProduct.getStartTime().getAsCalendar();
        return cal.get(1) + "  " + (cal.get(2) + 1) + "  " + cal.get(5);
    }

    private String writeStartTime() {
        double diff = this.srcProduct.getStartTime().getMJD() - this.dateDay.getMJD();
        double seconds = diff * 43200.0;
        return seconds + tab + "s";
    }

    private String writeCenterTime() {
        double center = this.srcProduct.getStartTime().getMJD() + (this.srcProduct.getEndTime().getMJD() - this.srcProduct.getStartTime().getMJD()) / 2.0;
        double seconds = (center - this.dateDay.getMJD()) * 43200.0;
        return seconds + tab + "s";
    }

    private String writeEndTime() {
        double diff = this.srcProduct.getEndTime().getMJD() - this.dateDay.getMJD();
        double seconds = diff * 43200.0;
        return seconds + tab + "s";
    }

    private String writeImageGeometry() {
        if (this.absRoot.getAttributeString("sample_type").equals("COMPLEX") || this.absRoot.getAttributeInt("srgr_flag", 0) == 0) {
            return "SLANT_RANGE";
        }
        return "GROUND_RANGE";
    }

    private void writeCenterLatLon(PrintStream p) {
        GeoPos geoPos = this.srcProduct.getSceneGeoCoding().getGeoPos(new PixelPos((double)(this.srcProduct.getSceneRasterWidth() / 2), (double)(this.srcProduct.getSceneRasterHeight() / 2)), null);
        p.println("center_latitude:\t" + geoPos.getLat() + tab + "degrees");
        p.println("center_longitude:\t" + geoPos.getLon() + tab + "degrees");
        GeoPos geoPos2 = this.srcProduct.getSceneGeoCoding().getGeoPos(new PixelPos((double)(this.srcProduct.getSceneRasterWidth() / 2), (double)(this.srcProduct.getSceneRasterHeight() / 2 + 100)), null);
        GeoUtils.DistanceHeading heading = GeoUtils.vincenty_inverse((double)geoPos.lat, (double)geoPos.lon, (double)geoPos2.lat, (double)geoPos2.lon);
        p.println("heading:\t" + heading.heading1 + tab + "degrees");
    }

    private void writeOrbitStateVectors(PrintStream p) {
        OrbitStateVector[] osvList = AbstractMetadata.getOrbitStateVectors((MetadataElement)this.absRoot);
        if (osvList != null && osvList.length > 0) {
            double seconds = (osvList[0].time_mjd - this.dateDay.getMJD()) * 43200.0;
            double seconds2 = (osvList[1].time_mjd - this.dateDay.getMJD()) * 43200.0;
            double interval = seconds2 - seconds;
            p.println("number_of_state_vectors:\t" + osvList.length);
            p.println("time_of_first_state_vector:\t" + seconds + tab + "s");
            p.println("state_vector_interval:\t" + interval + tab + "s");
            int num = 1;
            for (OrbitStateVector osv : osvList) {
                p.println("state_vector_position_" + num + sep + osv.x_pos + tab + osv.y_pos + tab + osv.z_pos + tab + "m   m   m");
                p.println("state_vector_velocity_" + num + sep + osv.x_vel + tab + osv.y_vel + tab + osv.z_vel + tab + "m/s m/s m/s");
                ++num;
            }
        }
    }

    private void writeEarthParams(PrintStream p) {
        double startTime = this.srcProduct.getStartTime().getMJD();
        OrbitStateVector[] osvList = AbstractMetadata.getOrbitStateVectors((MetadataElement)this.absRoot);
        double sensorToEarth = 0.0;
        if (osvList != null && osvList.length > 0) {
            double dtMin = Double.MAX_VALUE;
            int idx = 0;
            for (int i = 0; i < osvList.length; ++i) {
                double dt = Math.abs(startTime - osvList[i].time_mjd);
                if (!(dt < dtMin)) continue;
                dtMin = dt;
                idx = i;
            }
            sensorToEarth = Math.sqrt(osvList[idx].x_pos * osvList[idx].x_pos + osvList[idx].y_pos * osvList[idx].y_pos + osvList[idx].z_pos * osvList[idx].z_pos);
        }
        p.println("sar_to_earth_center:\t" + String.valueOf(sensorToEarth) + tab + "m");
        GeoPos geoPos = this.srcProduct.getSceneGeoCoding().getGeoPos(new PixelPos((double)(this.srcProduct.getSceneRasterWidth() / 2), (double)(this.srcProduct.getSceneRasterHeight() / 2)), null);
        double lat = geoPos.getLat();
        double tmp1 = 4.0680631590769E13 * Math.cos(lat);
        double tmp2 = 4.0408299984661445E13 * Math.sin(lat);
        double r = Math.sqrt((tmp1 * tmp1 + tmp2 * tmp2) / (tmp1 * Math.cos(lat) + tmp2 * Math.sin(lat)));
        p.println("earth_radius_below_sensor:\t" + String.valueOf(r) + tab + "m");
        p.println("earth_semi_major_axis:\t" + String.valueOf(6378137.0) + tab + "m");
        p.println("earth_semi_minor_axis:\t" + String.valueOf(6356752.314245179) + tab + "m");
    }

    int getHighestElemSize() {
        int highestElemSize = 0;
        for (Band band : this.srcProduct.getBands()) {
            int elemSize;
            if (!this.writer.shouldWrite((ProductNode)band) || (elemSize = ProductData.getElemSize((int)band.getDataType())) <= highestElemSize) continue;
            highestElemSize = elemSize;
        }
        return highestElemSize;
    }

    private String getDataType() {
        int highestElemSize = this.getHighestElemSize();
        if (highestElemSize >= 4) {
            return "FCOMPLEX";
        }
        return "SCOMPLEX";
    }

    private File createParFile(File file) {
        String name = FileUtils.getFilenameWithoutExtension((File)file);
        String ext = FileUtils.getExtension((String)name);
        String newExt = ".par";
        if (ext == null && this.isComplex) {
            newExt = this.isCoregistered ? ".rslc" + newExt : ".slc" + newExt;
        }
        name = name + newExt;
        return new File(file.getParent(), name);
    }
}

