/*
 * Decompiled with CFR 0.152.
 */
package org.esa.s1tbx.io.ceos.alos;

import Jama.Matrix;
import com.bc.ceres.core.ProgressMonitor;
import com.bc.ceres.core.VirtualDir;
import java.io.IOException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.math3.util.FastMath;
import org.esa.s1tbx.io.SARReader;
import org.esa.s1tbx.io.binary.BinaryRecord;
import org.esa.s1tbx.io.binary.IllegalBinaryFormatException;
import org.esa.s1tbx.io.ceos.CEOSImageFile;
import org.esa.s1tbx.io.ceos.CEOSProductDirectory;
import org.esa.s1tbx.io.ceos.alos.AlosPalsarConstants;
import org.esa.s1tbx.io.ceos.alos.AlosPalsarImageFile;
import org.esa.s1tbx.io.ceos.alos.AlosPalsarLeaderFile;
import org.esa.s1tbx.io.ceos.alos.AlosPalsarTrailerFile;
import org.esa.snap.core.datamodel.Band;
import org.esa.snap.core.datamodel.GeoCoding;
import org.esa.snap.core.datamodel.GeoPos;
import org.esa.snap.core.datamodel.MetadataAttribute;
import org.esa.snap.core.datamodel.MetadataElement;
import org.esa.snap.core.datamodel.PixelPos;
import org.esa.snap.core.datamodel.Product;
import org.esa.snap.core.datamodel.ProductData;
import org.esa.snap.core.datamodel.TiePointGeoCoding;
import org.esa.snap.core.datamodel.TiePointGrid;
import org.esa.snap.core.util.Debug;
import org.esa.snap.core.util.Guardian;
import org.esa.snap.engine_utilities.datamodel.AbstractMetadata;
import org.esa.snap.engine_utilities.datamodel.OrbitStateVector;
import org.esa.snap.engine_utilities.datamodel.Orbits;
import org.esa.snap.engine_utilities.eo.GeoUtils;
import org.esa.snap.engine_utilities.gpf.ReaderUtils;
import org.esa.snap.engine_utilities.util.Maths;

public class AlosPalsarProductDirectory
extends CEOSProductDirectory {
    protected AlosPalsarImageFile[] imageFiles = null;
    protected AlosPalsarLeaderFile leaderFile = null;
    protected AlosPalsarTrailerFile trailerFile = null;
    protected final transient Map<String, AlosPalsarImageFile> bandImageFileMap = new HashMap<String, AlosPalsarImageFile>(1);
    public static final DateFormat dateFormat1 = ProductData.UTC.createDateFormat((String)"yyyyMMddHHmmssSSS");
    public static final DateFormat dateFormat2 = ProductData.UTC.createDateFormat((String)"yyyyMMdd HH:mm:ss");
    public static final DateFormat dateFormat3 = ProductData.UTC.createDateFormat((String)"yyyyDDDSSSSSSSS");
    private float[] rangeDist;

    public AlosPalsarProductDirectory(VirtualDir dir) {
        Guardian.assertNotNull((String)"dir", (Object)dir);
        this.constants = new AlosPalsarConstants();
        this.productDir = dir;
    }

    @Override
    protected void readProductDirectory() throws IOException, IllegalBinaryFormatException {
        this.readVolumeDirectoryFileStream();
        this.updateProductType();
        this.leaderFile = new AlosPalsarLeaderFile(this.getCEOSFile((String[])this.constants.getLeaderFilePrefix())[0].imgInputStream);
        CEOSProductDirectory.CeosFile[] trlFile = this.getCEOSFile(this.constants.getTrailerFilePrefix());
        if (trlFile.length > 0) {
            this.trailerFile = new AlosPalsarTrailerFile(trlFile[0].imgInputStream);
        }
        CEOSProductDirectory.CeosFile[] ceosFiles = this.getCEOSFile(this.constants.getImageFilePrefix());
        ArrayList<AlosPalsarImageFile> imgArray = new ArrayList<AlosPalsarImageFile>(ceosFiles.length);
        for (CEOSProductDirectory.CeosFile imageFile : ceosFiles) {
            try {
                AlosPalsarImageFile imgFile = new AlosPalsarImageFile(imageFile.imgInputStream, this.getProductLevel(), imageFile.fileName);
                imgArray.add(imgFile);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        this.imageFiles = imgArray.toArray(new AlosPalsarImageFile[imgArray.size()]);
        this.sceneWidth = this.imageFiles[0].getRasterWidth();
        this.sceneHeight = this.imageFiles[0].getRasterHeight();
        AlosPalsarProductDirectory.assertSameWidthAndHeightForAllImages(this.imageFiles, this.sceneWidth, this.sceneHeight);
        if (this.leaderFile.getProductLevel() == 0 || this.leaderFile.getProductLevel() == 1) {
            this.isProductSLC = true;
        }
    }

    protected void updateProductType() {
        String prodType = this.productType.toUpperCase();
        while (prodType.endsWith("A") || prodType.endsWith("D") || prodType.endsWith("U") || prodType.endsWith("_")) {
            prodType = prodType.substring(0, prodType.length() - 1);
        }
        this.productType = prodType;
    }

    public boolean isALOS() throws IOException {
        String volumeId = this.getVolumeId().toUpperCase();
        String logicalVolumeId = this.getLogicalVolumeId().toUpperCase();
        return volumeId.contains("ALOS") || logicalVolumeId.contains("ALOS");
    }

    protected String getMission() {
        return "ALOS";
    }

    public int getProductLevel() {
        return this.leaderFile.getProductLevel();
    }

    @Override
    public Product createProduct() throws IOException {
        Product product = new Product(this.getProductName(), this.productType, this.sceneWidth, this.sceneHeight);
        for (AlosPalsarImageFile imageFile : this.imageFiles) {
            String pol = imageFile.getPolarization();
            if (this.isProductSLC) {
                Band bandI = this.createBand(product, "i_" + pol, "real", imageFile);
                Band bandQ = this.createBand(product, "q_" + pol, "imaginary", imageFile);
                ReaderUtils.createVirtualIntensityBand((Product)product, (Band)bandI, (Band)bandQ, (String)('_' + pol));
                continue;
            }
            Band band = this.createBand(product, "Amplitude_" + pol, "amplitude", imageFile);
            SARReader.createVirtualIntensityBand(product, band, '_' + pol);
        }
        product.setStartTime(AlosPalsarProductDirectory.getUTCScanStartTime(this.leaderFile.getSceneRecord(), null));
        product.setEndTime(AlosPalsarProductDirectory.getUTCScanStopTime(this.leaderFile.getSceneRecord(), null));
        product.setDescription(this.getProductDescription());
        if (this.isSLC()) {
            AlosPalsarProductDirectory.addGeoCodingFromPixelToLatLonCoefficients(product, this.leaderFile.getFacilityRecord());
        }
        if (product.getSceneGeoCoding() == null) {
            ReaderUtils.addGeoCoding((Product)product, (float[])AlosPalsarLeaderFile.getLatCorners(this.leaderFile.getMapProjRecord()), (float[])AlosPalsarLeaderFile.getLonCorners(this.leaderFile.getMapProjRecord()));
        }
        this.addTiePointGrids(product);
        this.addMetaData(product);
        if (product.getSceneGeoCoding() == null) {
            AlosPalsarProductDirectory.addGeoCodingFromWorkReport(product);
        }
        if (product.getSceneGeoCoding() == null && this.leaderFile.getSceneRecord() != null && this.leaderFile.getFacilityRecord() != null) {
            Double refLat = this.leaderFile.getSceneRecord().getAttributeDouble("scene centre geodetic latitude");
            Double refLon = this.leaderFile.getSceneRecord().getAttributeDouble("scene centre geodetic longitude");
            if (refLat == null || refLat == 0.0 || refLon == null || refLon == 0.0) {
                refLat = this.leaderFile.getFacilityRecord().getAttributeDouble("Origin Latitude");
                refLon = this.leaderFile.getFacilityRecord().getAttributeDouble("Origin Longitude");
            }
            if (refLat != null && refLon != null) {
                AlosPalsarProductDirectory.addTPGGeoCoding(product, refLat, refLon);
            }
        }
        AlosPalsarProductDirectory.updateMetadata(product);
        return product;
    }

    private static void addGeoCodingFromPixelToLatLonCoefficients(Product product, BinaryRecord facilityRecord) {
        if (facilityRecord == null || facilityRecord.getAttributeDouble("Origin Line") == null) {
            return;
        }
        int originLine = (int)Math.floor(facilityRecord.getAttributeDouble("Origin Line"));
        int originPixel = (int)Math.floor(facilityRecord.getAttributeDouble("Origin Pixel"));
        int numCoefficients = 50;
        double[] a = new double[25];
        double[] b = new double[25];
        boolean coeffNonZero = false;
        for (int i = 0; i < 50; ++i) {
            double c = facilityRecord.getAttributeDouble("Pixel to Lat Lon coefficients " + (i + 1));
            if (!coeffNonZero && c != 0.0) {
                coeffNonZero = true;
            }
            if (i < 25) {
                a[i] = c;
                continue;
            }
            b[i - 25] = c;
        }
        if (!coeffNonZero) {
            return;
        }
        int gridWidth = 11;
        int gridHeight = 11;
        float[] targetLatTiePoints = new float[121];
        float[] targetLonTiePoints = new float[121];
        int sourceImageWidth = product.getSceneRasterWidth();
        int sourceImageHeight = product.getSceneRasterHeight();
        double subSamplingX = (double)sourceImageWidth / 10.0;
        double subSamplingY = (double)sourceImageHeight / 10.0;
        int k = 0;
        for (int r = 0; r < 11; ++r) {
            double y = (double)r * subSamplingY - (double)originLine;
            double y2 = y * y;
            double y3 = y2 * y;
            double y4 = y2 * y2;
            for (int c = 0; c < 11; ++c) {
                double x = (double)c * subSamplingX - (double)originPixel;
                double x2 = x * x;
                double x3 = x2 * x;
                double x4 = x2 * x2;
                double lat = a[0] * x4 * y4 + a[1] * x4 * y3 + a[2] * x4 * y2 + a[3] * x4 * y + a[4] * x4 + a[5] * x3 * y4 + a[6] * x3 * y3 + a[7] * x3 * y2 + a[8] * x3 * y + a[9] * x3 + a[10] * x2 * y4 + a[11] * x2 * y3 + a[12] * x2 * y2 + a[13] * x2 * y + a[14] * x2 + a[15] * x * y4 + a[16] * x * y3 + a[17] * x * y2 + a[18] * x * y + a[19] * x + a[20] * y4 + a[21] * y3 + a[22] * y2 + a[23] * y + a[24];
                double lon = b[0] * x4 * y4 + b[1] * x4 * y3 + b[2] * x4 * y2 + b[3] * x4 * y + b[4] * x4 + b[5] * x3 * y4 + b[6] * x3 * y3 + b[7] * x3 * y2 + b[8] * x3 * y + b[9] * x3 + b[10] * x2 * y4 + b[11] * x2 * y3 + b[12] * x2 * y2 + b[13] * x2 * y + b[14] * x2 + b[15] * x * y4 + b[16] * x * y3 + b[17] * x * y2 + b[18] * x * y + b[19] * x + b[20] * y4 + b[21] * y3 + b[22] * y2 + b[23] * y + b[24];
                targetLatTiePoints[k] = (float)lat;
                targetLonTiePoints[k] = (float)lon;
                ++k;
            }
        }
        TiePointGrid latGrid = new TiePointGrid("latitude", 11, 11, 0.0, 0.0, (double)((int)subSamplingX), (double)((int)subSamplingY), targetLatTiePoints);
        TiePointGrid lonGrid = new TiePointGrid("longitude", 11, 11, 0.0, 0.0, (double)((int)subSamplingX), (double)((int)subSamplingY), targetLonTiePoints, 180);
        TiePointGeoCoding tpGeoCoding = new TiePointGeoCoding(latGrid, lonGrid);
        product.addTiePointGrid(latGrid);
        product.addTiePointGrid(lonGrid);
        product.setSceneGeoCoding((GeoCoding)tpGeoCoding);
    }

    private static void updateMetadata(Product product) {
        GeoCoding geoCoding = product.getSceneGeoCoding();
        if (geoCoding == null) {
            return;
        }
        MetadataElement absRoot = AbstractMetadata.getAbstractedMetadata((Product)product);
        int w = product.getSceneRasterWidth();
        int h = product.getSceneRasterHeight();
        GeoPos geo00 = geoCoding.getGeoPos(new PixelPos(0.0, 0.0), null);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_near_lat", (double)geo00.getLat());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_near_long", (double)geo00.getLon());
        GeoPos geo01 = geoCoding.getGeoPos(new PixelPos((double)w, 0.0), null);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_far_lat", (double)geo01.getLat());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_far_long", (double)geo01.getLon());
        GeoPos geo10 = geoCoding.getGeoPos(new PixelPos(0.0, (double)h), null);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_near_lat", (double)geo10.getLat());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_near_long", (double)geo10.getLon());
        GeoPos geo11 = geoCoding.getGeoPos(new PixelPos((double)w, (double)h), null);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_far_lat", (double)geo11.getLat());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_far_long", (double)geo11.getLon());
    }

    private void addTiePointGrids(Product product) {
        int gridWidth = 11;
        int gridHeight = 11;
        int subSamplingX = product.getSceneRasterWidth() / 10;
        int subSamplingY = product.getSceneRasterHeight() / 10;
        BinaryRecord sceneRec = this.leaderFile.getSceneRecord();
        TiePointGrid slantRangeGrid = new TiePointGrid("slant_range_time", 11, 11, 0.0, 0.0, (double)subSamplingX, (double)subSamplingY);
        slantRangeGrid.setUnit("ns");
        product.addTiePointGrid(slantRangeGrid);
        if (sceneRec != null) {
            TiePointGrid incidentAngleGrid = new TiePointGrid("incident_angle", 11, 11, 0.0, 0.0, (double)subSamplingX, (double)subSamplingY);
            incidentAngleGrid.setDiscontinuity(-1);
            incidentAngleGrid.setUnit("deg");
            product.addTiePointGrid(incidentAngleGrid);
        }
    }

    public void readTiePointGridRasterData(TiePointGrid tpg, ProductData destBuffer, ProgressMonitor pm) throws IOException {
        int gridWidth = 11;
        int gridHeight = 11;
        int subSamplingX = (int)tpg.getSubSamplingX();
        float[] rangeTime = new float[121];
        BinaryRecord sceneRec = this.leaderFile.getSceneRecord();
        if (this.rangeDist == null) {
            int j;
            int k;
            this.rangeDist = new float[121];
            if (this.leaderFile.getProductLevel() == 1 && sceneRec != null) {
                double samplingRate = sceneRec.getAttributeDouble("Range sampling rate") * 1000000.0;
                double rangePixelSpacing = (double)subSamplingX * 1.49896229E8 / samplingRate;
                double slantRangeToFirstPixel = this.imageFiles[0].getSlantRangeToFirstPixel(0);
                k = 0;
                for (j = 0; j < 11; ++j) {
                    for (int i = 0; i < 11; ++i) {
                        this.rangeDist[k++] = (float)(slantRangeToFirstPixel + (double)i * rangePixelSpacing);
                    }
                }
            } else if (this.leaderFile.getProductLevel() == 3) {
                double slantRangeToFirstPixel = this.imageFiles[0].getSlantRangeToFirstPixel(0);
                double slantRangeToMidPixel = this.imageFiles[0].getSlantRangeToMidPixel(0);
                double slantRangeToLastPixel = this.imageFiles[0].getSlantRangeToLastPixel(0);
                k = 0;
                for (j = 0; j < 11; ++j) {
                    double[] polyCoef = AlosPalsarProductDirectory.computePolynomialCoefficients(slantRangeToFirstPixel, slantRangeToMidPixel, slantRangeToLastPixel, this.sceneWidth);
                    for (int i = 0; i < 11; ++i) {
                        int x = i * subSamplingX;
                        this.rangeDist[k++] = (float)(polyCoef[0] + polyCoef[1] * (double)x + polyCoef[2] * (double)x * (double)x);
                    }
                }
            }
        }
        if (tpg.getName().equals("slant_range_time")) {
            for (int k = 0; k < this.rangeDist.length; ++k) {
                rangeTime[k] = (float)((double)this.rangeDist[k] / 1.49896229E8 * 1.0E9);
            }
            tpg.setDataElems((Object)rangeTime);
            destBuffer.setElems((Object)rangeTime);
        } else if (tpg.getName().equals("incident_angle") && sceneRec != null) {
            double a0 = sceneRec.getAttributeDouble("Incidence angle constant term");
            double a1 = sceneRec.getAttributeDouble("Incidence angle linear term");
            double a2 = sceneRec.getAttributeDouble("Incidence angle quadratic term");
            double a3 = sceneRec.getAttributeDouble("Incidence angle cubic term");
            double a4 = sceneRec.getAttributeDouble("Incidence angle fourth term");
            double a5 = sceneRec.getAttributeDouble("Incidence angle fifth term");
            float[] angles = new float[121];
            int k = 0;
            for (int j = 0; j < 11; ++j) {
                for (int i = 0; i < 11; ++i) {
                    angles[k] = (float)((a0 + a1 * (double)this.rangeDist[k] / 1000.0 + a2 * FastMath.pow((double)((double)this.rangeDist[k] / 1000.0), (double)2.0) + a3 * FastMath.pow((double)((double)this.rangeDist[k] / 1000.0), (double)3.0) + a4 * FastMath.pow((double)((double)this.rangeDist[k] / 1000.0), (double)4.0) + a5 * FastMath.pow((double)((double)this.rangeDist[k] / 1000.0), (double)5.0)) * 57.29577951308232);
                    ++k;
                }
            }
            tpg.setDataElems((Object)angles);
            destBuffer.setElems((Object)angles);
        }
    }

    private static double[] computePolynomialCoefficients(double slantRangeToFirstPixel, double slantRangeToMidPixel, double slantRangeToLastPixel, int imageWidth) {
        boolean firstPixel = false;
        int midPixel = imageWidth / 2;
        int lastPixel = imageWidth - 1;
        double[] idxArray = new double[]{0.0, midPixel, lastPixel};
        double[] rangeArray = new double[]{slantRangeToFirstPixel, slantRangeToMidPixel, slantRangeToLastPixel};
        Matrix A = Maths.createVandermondeMatrix((double[])idxArray, (int)2);
        Matrix b = new Matrix(rangeArray, 3);
        Matrix x = A.solve(b);
        return x.getColumnPackedCopy();
    }

    private static void addGeoCodingFromWorkReport(Product product) {
        MetadataElement absRoot = AbstractMetadata.getAbstractedMetadata((Product)product);
        MetadataElement workReportElem = AbstractMetadata.getOriginalProductMetadata((Product)product).getElement("Work Report");
        if (workReportElem != null) {
            try {
                float latUL = Float.parseFloat(workReportElem.getAttributeString("Brs_ImageSceneLeftTopLatitude", "0"));
                float latUR = Float.parseFloat(workReportElem.getAttributeString("Brs_ImageSceneRightTopLatitude", "0"));
                float latLL = Float.parseFloat(workReportElem.getAttributeString("Brs_ImageSceneLeftBottomLatitude", "0"));
                float latLR = Float.parseFloat(workReportElem.getAttributeString("Brs_ImageSceneRightBottomLatitude", "0"));
                float lonUL = Float.parseFloat(workReportElem.getAttributeString("Brs_ImageSceneLeftTopLongitude", "0"));
                float lonUR = Float.parseFloat(workReportElem.getAttributeString("Brs_ImageSceneRightTopLongitude", "0"));
                float lonLL = Float.parseFloat(workReportElem.getAttributeString("Brs_ImageSceneLeftBottomLongitude", "0"));
                float lonLR = Float.parseFloat(workReportElem.getAttributeString("Brs_ImageSceneRightBottomLongitude", "0"));
                String pass = absRoot.getAttributeString("PASS");
                String prodType = absRoot.getAttributeString("PRODUCT_TYPE");
                if (prodType.contains("1.1")) {
                    float temp;
                    if (pass.equals("ASCENDING")) {
                        temp = latUL;
                        latUL = latLL;
                        latLL = temp;
                        temp = latUR;
                        latUR = latLR;
                        latLR = temp;
                        temp = lonUL;
                        lonUL = lonLL;
                        lonLL = temp;
                        temp = lonUR;
                        lonUR = lonLR;
                        lonLR = temp;
                    } else {
                        temp = latUL;
                        latUL = latUR;
                        latUR = temp;
                        temp = latLL;
                        latLL = latLR;
                        latLR = temp;
                        temp = lonUL;
                        lonUL = lonUR;
                        lonUR = temp;
                        temp = lonLL;
                        lonLL = lonLR;
                        lonLR = temp;
                    }
                }
                float[] latCorners = new float[]{latUL, latUR, latLL, latLR};
                float[] lonCorners = new float[]{lonUL, lonUR, lonLL, lonLR};
                absRoot.setAttributeDouble("first_near_lat", (double)latUL);
                absRoot.setAttributeDouble("first_near_long", (double)lonUL);
                absRoot.setAttributeDouble("first_far_lat", (double)latUR);
                absRoot.setAttributeDouble("first_far_long", (double)lonUR);
                absRoot.setAttributeDouble("last_near_lat", (double)latLL);
                absRoot.setAttributeDouble("last_near_long", (double)lonLL);
                absRoot.setAttributeDouble("last_far_lat", (double)latLR);
                absRoot.setAttributeDouble("last_far_long", (double)lonLR);
                ReaderUtils.addGeoCoding((Product)product, (float[])latCorners, (float[])lonCorners);
            }
            catch (Exception e) {
                Debug.trace((String)e.toString());
            }
        }
    }

    @Override
    public CEOSImageFile getImageFile(Band band) {
        return this.bandImageFileMap.get(band.getName());
    }

    @Override
    public void close() throws IOException {
        for (int i = 0; i < this.imageFiles.length; ++i) {
            this.imageFiles[i].close();
            this.imageFiles[i] = null;
        }
        this.imageFiles = null;
    }

    private Band createBand(Product product, String name, String unit, AlosPalsarImageFile imageFile) {
        Band band = this.createBand(product, name, unit, imageFile.getBitsPerSample());
        this.bandImageFileMap.put(name, imageFile);
        return band;
    }

    private void addMetaData(Product product) throws IOException {
        MetadataElement root = AbstractMetadata.addOriginalProductMetadata((MetadataElement)product.getMetadataRoot());
        if (this.leaderFile != null) {
            MetadataElement leadMetadata = new MetadataElement("Leader");
            this.leaderFile.addMetadata(leadMetadata);
            root.addElement(leadMetadata);
        }
        if (this.trailerFile != null) {
            MetadataElement trailMetadata = new MetadataElement("Trailer");
            this.trailerFile.addMetadata(trailMetadata);
            root.addElement(trailMetadata);
        }
        MetadataElement volMetadata = new MetadataElement("Volume");
        this.volumeDirectoryFile.assignMetadataTo(volMetadata);
        root.addElement(volMetadata);
        int c = 1;
        for (AlosPalsarImageFile imageFile : this.imageFiles) {
            imageFile.assignMetadataTo(root, c++);
        }
        AlosPalsarProductDirectory.addSummaryMetadata(this.findFile("summary.txt"), "Summary Information", root);
        AlosPalsarProductDirectory.addSummaryMetadata(this.findFile("workreport"), "Work Report", root);
        this.addAbstractedMetadataHeader(product);
    }

    private void addAbstractedMetadataHeader(Product product) {
        MetadataElement absRoot = AbstractMetadata.addAbstractedMetadataHeader((MetadataElement)product.getMetadataRoot());
        MetadataElement origProductMetadata = AbstractMetadata.getOriginalProductMetadata((Product)product);
        BinaryRecord sceneRec = this.leaderFile.getSceneRecord();
        BinaryRecord mapProjRec = this.leaderFile.getMapProjRecord();
        BinaryRecord radiometricRec = this.leaderFile.getRadiometricRecord();
        if (sceneRec == null) {
            return;
        }
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"PRODUCT", (String)this.getProductName());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"PRODUCT_TYPE", (String)this.getProductType());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"SPH_DESCRIPTOR", (String)sceneRec.getAttributeString("Product type descriptor"));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"MISSION", (String)this.getMission());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"antenna_pointing", (String)"right");
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"PROC_TIME", (ProductData.UTC)AlosPalsarProductDirectory.getProcTime(this.volumeDirectoryFile.getVolumeDescriptorRecord()));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"Processing_system_identifier", (String)sceneRec.getAttributeString("Processing system identifier").trim());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"ABS_ORBIT", (int)Integer.parseInt(sceneRec.getAttributeString("Orbit number").trim()));
        ProductData.UTC startTime = AlosPalsarProductDirectory.getStartTime(sceneRec, origProductMetadata, "StartDateTime");
        product.setStartTime(startTime);
        ProductData.UTC endTime = AlosPalsarProductDirectory.getEndTime(sceneRec, origProductMetadata, "EndDateTime", startTime);
        product.setEndTime(endTime);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_line_time", (ProductData.UTC)startTime);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_line_time", (ProductData.UTC)endTime);
        if (mapProjRec != null) {
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_near_lat", (Double)mapProjRec.getAttributeDouble("1st line 1st pixel geodetic latitude"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_near_long", (Double)mapProjRec.getAttributeDouble("1st line 1st pixel geodetic longitude"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_far_lat", (Double)mapProjRec.getAttributeDouble("1st line last valid pixel geodetic latitude"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"first_far_long", (Double)mapProjRec.getAttributeDouble("1st line last valid pixel geodetic longitude"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_near_lat", (Double)mapProjRec.getAttributeDouble("Last line 1st pixel geodetic latitude"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_near_long", (Double)mapProjRec.getAttributeDouble("Last line 1st pixel geodetic longitude"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_far_lat", (Double)mapProjRec.getAttributeDouble("Last line last valid pixel geodetic latitude"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"last_far_long", (Double)mapProjRec.getAttributeDouble("Last line last valid pixel geodetic longitude"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"PASS", (String)AlosPalsarProductDirectory.getPass(mapProjRec, sceneRec));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"range_spacing", (Double)mapProjRec.getAttributeDouble("Nominal inter-pixel distance in output scene"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"azimuth_spacing", (Double)mapProjRec.getAttributeDouble("Nominal inter-line distance in output scene"));
        } else if (sceneRec != null) {
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"range_spacing", (Double)sceneRec.getAttributeDouble("Pixel spacing"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"azimuth_spacing", (Double)sceneRec.getAttributeDouble("Line spacing"));
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"PASS", (String)AlosPalsarProductDirectory.getPass(mapProjRec, sceneRec));
        }
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"SAMPLE_TYPE", (String)this.getSampleType());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"algorithm", (String)sceneRec.getAttributeString("Processing algorithm identifier"));
        for (int i = 0; i < this.imageFiles.length; ++i) {
            if (this.imageFiles[i] == null) continue;
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)AbstractMetadata.polarTags[i], (String)this.imageFiles[i].getPolarization());
        }
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"azimuth_looks", (Double)sceneRec.getAttributeDouble("Nominal number of looks processed in azimuth"));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"range_looks", (Double)sceneRec.getAttributeDouble("Nominal number of looks processed in range"));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"pulse_repetition_frequency", (double)(sceneRec.getAttributeDouble("Pulse Repetition Frequency") / 1000.0));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"radar_frequency", (double)AlosPalsarProductDirectory.getRadarFrequency(sceneRec));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"slant_range_to_first_pixel", (double)this.imageFiles[0].getSlantRangeToFirstPixel(0));
        double rangeBW = sceneRec.getAttributeDouble("Total processor bandwidth in range");
        double azimuthBW = sceneRec.getAttributeDouble("Total processor bandwidth in azimuth");
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"range_bandwidth", (double)rangeBW);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"azimuth_bandwidth", (double)azimuthBW);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"line_time_interval", (double)ReaderUtils.getLineTimeInterval((ProductData.UTC)startTime, (ProductData.UTC)endTime, (int)this.sceneHeight));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"num_output_lines", (int)product.getSceneRasterHeight());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"num_samples_per_line", (int)product.getSceneRasterWidth());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"total_size", (int)ReaderUtils.getTotalSize((Product)product));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"srgr_flag", (int)this.isGroundRange());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"map_projection", (String)this.getMapProjection());
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"geo_ref_system", (String)sceneRec.getAttributeString("Ellipsoid designator"));
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"ant_elev_corr_flag", (int)1);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"range_spread_comp_flag", (int)1);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"replica_power_corr_flag", (int)0);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"abs_calibration_flag", (int)0);
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"coregistered_stack", (int)0);
        if (radiometricRec != null) {
            AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"calibration_factor", (Double)radiometricRec.getAttributeDouble("Calibration factor"));
            absRoot.getAttribute("calibration_factor").setUnit("dB");
        }
        AbstractMetadata.setAttribute((MetadataElement)absRoot, (String)"range_sampling_rate", (Double)sceneRec.getAttributeDouble("Range sampling rate"));
        AlosPalsarProductDirectory.addOrbitStateVectors(absRoot, this.leaderFile.getPlatformPositionRecord());
        AlosPalsarProductDirectory.addDopplerCentroidCoefficients(absRoot, sceneRec);
    }

    private int isGroundRange() {
        if (this.leaderFile.getMapProjRecord() == null) {
            return this.isProductSLC ? 0 : 1;
        }
        String projDesc = this.leaderFile.getMapProjRecord().getAttributeString("Map projection descriptor").toLowerCase();
        if (projDesc.contains("slant")) {
            return 0;
        }
        return 1;
    }

    private String getMapProjection() {
        if (this.leaderFile.getMapProjRecord() == null) {
            return " ";
        }
        String projDesc = this.leaderFile.getMapProjRecord().getAttributeString("Map projection descriptor").toLowerCase();
        if (projDesc.contains("geo") || this.getProductType().contains("1.5G")) {
            return "Geocoded";
        }
        return " ";
    }

    private static ProductData.UTC getStartTime(BinaryRecord sceneRec, MetadataElement origProductMetadata, String tagInSummary) {
        ProductData.UTC time = AlosPalsarProductDirectory.getUTCScanStartTime(sceneRec, null);
        if (time.equalElems((ProductData)AbstractMetadata.NO_METADATA_UTC)) {
            try {
                MetadataElement imageRecordElem;
                ProductData.UTC summaryTime = null;
                MetadataElement summaryElem = origProductMetadata.getElement("Summary Information");
                if (summaryElem != null) {
                    for (MetadataAttribute sum : summaryElem.getAttributes()) {
                        if (!sum.getName().contains(tagInSummary)) continue;
                        summaryTime = AbstractMetadata.parseUTC((String)summaryElem.getAttributeString(sum.getName().trim()), (DateFormat)dateFormat2);
                    }
                }
                ProductData.UTC workReportTime = null;
                MetadataElement workReportElem = origProductMetadata.getElement("Work Report");
                if (workReportElem != null) {
                    String valueStr = workReportElem.getAttributeString("Img_SceneStartDateTime");
                    if (valueStr != null && valueStr.length() > 0) {
                        workReportTime = AbstractMetadata.parseUTC((String)valueStr, (DateFormat)dateFormat2);
                    }
                    if (workReportTime == null && (valueStr = workReportElem.getAttributeString("Brs_SceneStartDateTime")) != null && valueStr.length() > 0) {
                        workReportTime = AbstractMetadata.parseUTC((String)valueStr, (DateFormat)dateFormat2);
                    }
                }
                ProductData.UTC imgRecTime = null;
                MetadataElement imageDescriptorElem = origProductMetadata.getElement("Image Descriptor 1");
                if (imageDescriptorElem != null && (imageRecordElem = imageDescriptorElem.getElement("Image Record")) != null) {
                    int year = imageRecordElem.getAttributeInt("Sensor acquisition year", 0);
                    int days = imageRecordElem.getAttributeInt("Sensor acquisition day of year", 0);
                    int milliseconds = imageRecordElem.getAttributeInt("Sensor acquisition milliseconds of day", 0);
                    StringBuilder sb = new StringBuilder(String.valueOf(year));
                    String dayStr = String.valueOf(days);
                    for (int i = dayStr.length(); i < 3; ++i) {
                        sb.append('0');
                    }
                    sb.append(dayStr);
                    String millisecondStr = String.valueOf(milliseconds);
                    for (int i = millisecondStr.length(); i < 8; ++i) {
                        sb.append('0');
                    }
                    sb.append(millisecondStr);
                    imgRecTime = ProductData.UTC.parse((String)sb.toString(), (DateFormat)dateFormat3);
                }
                if (summaryTime != null) {
                    return summaryTime;
                }
                if (workReportTime != null) {
                    return workReportTime;
                }
                return imgRecTime;
            }
            catch (Exception e) {
                time = AbstractMetadata.NO_METADATA_UTC;
            }
        }
        return time;
    }

    private static ProductData.UTC getEndTime(BinaryRecord sceneRec, MetadataElement origProductMetadata, String tagInSummary, ProductData.UTC startTime) {
        ProductData.UTC time = AlosPalsarProductDirectory.getUTCScanStartTime(sceneRec, null);
        if (time.equalElems((ProductData)AbstractMetadata.NO_METADATA_UTC)) {
            try {
                ProductData.UTC summaryTime = null;
                MetadataElement summaryElem = origProductMetadata.getElement("Summary Information");
                if (summaryElem != null) {
                    for (MetadataAttribute metadataAttribute : summaryElem.getAttributes()) {
                        if (!metadataAttribute.getName().contains(tagInSummary)) continue;
                        summaryTime = AbstractMetadata.parseUTC((String)summaryElem.getAttributeString(metadataAttribute.getName().trim()), (DateFormat)dateFormat2);
                    }
                }
                ProductData.UTC workReportTime = null;
                MetadataElement workReportElem = origProductMetadata.getElement("Work Report");
                if (workReportElem != null) {
                    String valueStr = workReportElem.getAttributeString("Img_SceneEndDateTime");
                    if (valueStr != null && valueStr.length() > 0) {
                        workReportTime = AbstractMetadata.parseUTC((String)valueStr, (DateFormat)dateFormat2);
                    }
                    if (workReportTime == null && (valueStr = workReportElem.getAttributeString("Brs_SceneEndDateTime")) != null && valueStr.length() > 0) {
                        workReportTime = AbstractMetadata.parseUTC((String)valueStr, (DateFormat)dateFormat2);
                    }
                    if (workReportTime == null) {
                        for (MetadataAttribute workRep : workReportElem.getAttributes()) {
                            if (!workRep.getName().contains("SceneCenterDateTime")) continue;
                            ProductData.UTC centreTime = AbstractMetadata.parseUTC((String)workReportElem.getAttributeString(workRep.getName().trim()), (DateFormat)dateFormat2);
                            double diff = centreTime.getMJD() - startTime.getMJD();
                            workReportTime = new ProductData.UTC(startTime.getMJD() + diff * 2.0);
                        }
                    }
                }
                ProductData.UTC imgRecTime = null;
                MetadataElement metadataElement = origProductMetadata.getElement("Image Descriptor 1");
                if (metadataElement != null) {
                    int numRecords = metadataElement.getAttributeInt("Number of SAR DATA records", 0);
                    MetadataElement imageRecordElem = metadataElement.getElement("Image Record");
                    if (imageRecordElem != null) {
                        int year = imageRecordElem.getAttributeInt("Sensor acquisition year", 0);
                        int days = imageRecordElem.getAttributeInt("Sensor acquisition day of year", 0);
                        int milliseconds = imageRecordElem.getAttributeInt("Sensor acquisition milliseconds of day", 0);
                        double prf = imageRecordElem.getAttributeDouble("PRF", 0.0);
                        milliseconds += (int)((double)(numRecords - 1) * 1000000.0 / prf);
                        StringBuilder sb = new StringBuilder(String.valueOf(year));
                        String dayStr = String.valueOf(days);
                        for (int i = dayStr.length(); i < 3; ++i) {
                            sb.append('0');
                        }
                        sb.append(dayStr);
                        String millisecondStr = String.valueOf(milliseconds);
                        for (int i = millisecondStr.length(); i < 8; ++i) {
                            sb.append('0');
                        }
                        sb.append(millisecondStr);
                        imgRecTime = ProductData.UTC.parse((String)sb.toString(), (DateFormat)dateFormat3);
                    }
                }
                if (summaryTime != null) {
                    return summaryTime;
                }
                if (workReportTime != null) {
                    return workReportTime;
                }
                if (imgRecTime != null) {
                    return imgRecTime;
                }
                String centreTimeStr = sceneRec.getAttributeString("Scene centre time");
                ProductData.UTC centreTime = AbstractMetadata.parseUTC((String)centreTimeStr.trim(), (DateFormat)dateFormat1);
                double diff = centreTime.getMJD() - startTime.getMJD();
                return new ProductData.UTC(startTime.getMJD() + diff * 2.0);
            }
            catch (Exception e) {
                time = AbstractMetadata.NO_METADATA_UTC;
            }
        }
        return time;
    }

    private String getProductName() {
        return this.getMission() + '-' + this.volumeDirectoryFile.getProductName();
    }

    protected String getProductDescription() {
        return "ALOS PALSAR product " + this.leaderFile.getProductLevel();
    }

    private static void addTPGGeoCoding(Product product, double refLat, double refLon) throws IOException {
        OrbitStateVector[] orbitStateVectors;
        int gridWidth = 11;
        int gridHeight = 11;
        float[] targetLatTiePoints = new float[121];
        float[] targetLonTiePoints = new float[121];
        int sourceImageWidth = product.getSceneRasterWidth();
        int sourceImageHeight = product.getSceneRasterHeight();
        double subSamplingX = (double)sourceImageWidth / 10.0;
        double subSamplingY = (double)sourceImageHeight / 10.0;
        TiePointGrid slantRangeTime = product.getTiePointGrid("slant_range_time");
        MetadataElement absRoot = AbstractMetadata.getAbstractedMetadata((Product)product);
        double firstLineUTC = absRoot.getAttributeUTC("first_line_time").getMJD();
        double lastLineUTC = absRoot.getAttributeUTC("last_line_time").getMJD();
        double lineTimeInterval = absRoot.getAttributeDouble("line_time_interval") / 86400.0;
        try {
            orbitStateVectors = AbstractMetadata.getOrbitStateVectors((MetadataElement)absRoot);
        }
        catch (Exception e) {
            throw new IOException(e.getMessage());
        }
        if (!AlosPalsarProductDirectory.checkStateVectorValidity(orbitStateVectors)) {
            return;
        }
        int numVectors = orbitStateVectors.length;
        int startIdx = 0;
        int endIdx = 0;
        double t1 = Math.min(firstLineUTC, lastLineUTC);
        double t2 = Math.max(firstLineUTC, lastLineUTC);
        for (int i = 0; i < numVectors; ++i) {
            double time = orbitStateVectors[i].time_mjd;
            if (time < t1) {
                startIdx = i;
            }
            if (!(time < t2)) continue;
            endIdx = i;
        }
        while (endIdx - startIdx + 1 < Math.min(5, numVectors)) {
            startIdx = Math.max(startIdx - 1, 0);
            endIdx = Math.min(endIdx + 1, numVectors - 1);
        }
        int numVectorsUsed = endIdx - startIdx + 1;
        double[] timeArray = new double[numVectorsUsed];
        double[] xPosArray = new double[numVectorsUsed];
        double[] yPosArray = new double[numVectorsUsed];
        double[] zPosArray = new double[numVectorsUsed];
        double[] xVelArray = new double[numVectorsUsed];
        double[] yVelArray = new double[numVectorsUsed];
        double[] zVelArray = new double[numVectorsUsed];
        for (int i = startIdx; i <= endIdx; ++i) {
            timeArray[i - startIdx] = orbitStateVectors[i].time_mjd;
            xPosArray[i - startIdx] = orbitStateVectors[i].x_pos;
            yPosArray[i - startIdx] = orbitStateVectors[i].y_pos;
            zPosArray[i - startIdx] = orbitStateVectors[i].z_pos;
            xVelArray[i - startIdx] = orbitStateVectors[i].x_vel;
            yVelArray[i - startIdx] = orbitStateVectors[i].y_vel;
            zVelArray[i - startIdx] = orbitStateVectors[i].z_vel;
        }
        int k = 0;
        for (int r = 0; r < 11; ++r) {
            int y = r == 10 ? sourceImageHeight - 1 : (int)((double)r * subSamplingY);
            double curLineUTC = firstLineUTC + (double)y * lineTimeInterval;
            Orbits.OrbitVector data = AlosPalsarProductDirectory.getOrbitData(curLineUTC, timeArray, xPosArray, yPosArray, zPosArray, xVelArray, yVelArray, zVelArray);
            for (int c = 0; c < 11; ++c) {
                int x = c == 10 ? sourceImageWidth - 1 : (int)((double)c * subSamplingX);
                double slrgTime = slantRangeTime.getPixelDouble(x, y) / 1.0E9;
                GeoPos geoPos = AlosPalsarProductDirectory.computeLatLon(refLat, refLon, slrgTime, data);
                targetLatTiePoints[k] = (float)geoPos.lat;
                targetLonTiePoints[k] = (float)geoPos.lon;
                ++k;
            }
        }
        TiePointGrid latGrid = new TiePointGrid("latitude", 11, 11, 0.0, 0.0, subSamplingX, subSamplingY, targetLatTiePoints);
        TiePointGrid lonGrid = new TiePointGrid("longitude", 11, 11, 0.0, 0.0, subSamplingX, subSamplingY, targetLonTiePoints, 180);
        TiePointGeoCoding tpGeoCoding = new TiePointGeoCoding(latGrid, lonGrid);
        product.addTiePointGrid(latGrid);
        product.addTiePointGrid(lonGrid);
        product.setSceneGeoCoding((GeoCoding)tpGeoCoding);
    }

    private static GeoPos computeLatLon(double refLat, double refLon, double slrgTime, Orbits.OrbitVector data) {
        double[] xyz = new double[3];
        GeoPos geoPos = new GeoPos(refLat, refLon);
        GeoUtils.geo2xyz((GeoPos)geoPos, (double[])xyz);
        GeoUtils.computeAccurateXYZ((Orbits.OrbitVector)data, (double[])xyz, (double)slrgTime);
        GeoUtils.xyz2geo((double[])xyz, (GeoPos)geoPos);
        return geoPos;
    }

    private static boolean checkStateVectorValidity(OrbitStateVector[] orbitStateVectors) {
        if (orbitStateVectors == null) {
            return false;
        }
        if (orbitStateVectors.length <= 1) {
            return false;
        }
        for (int i = 1; i < orbitStateVectors.length; ++i) {
            if (orbitStateVectors[i].time_mjd != orbitStateVectors[0].time_mjd) continue;
            return false;
        }
        return true;
    }

    private static Orbits.OrbitVector getOrbitData(double utc, double[] timeArray, double[] xPosArray, double[] yPosArray, double[] zPosArray, double[] xVelArray, double[] yVelArray, double[] zVelArray) {
        return new Orbits.OrbitVector(utc, Maths.lagrangeInterpolatingPolynomial((double[])timeArray, (double[])xPosArray, (double)utc), Maths.lagrangeInterpolatingPolynomial((double[])timeArray, (double[])yPosArray, (double)utc), Maths.lagrangeInterpolatingPolynomial((double[])timeArray, (double[])zPosArray, (double)utc), Maths.lagrangeInterpolatingPolynomial((double[])timeArray, (double[])xVelArray, (double)utc), Maths.lagrangeInterpolatingPolynomial((double[])timeArray, (double[])yVelArray, (double)utc), Maths.lagrangeInterpolatingPolynomial((double[])timeArray, (double[])zVelArray, (double)utc));
    }
}

