/*
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 */
package gov.nasa.worldwindx.examples;

import gov.nasa.worldwind.*;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwindx.examples.util.LabeledPath;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.layers.RenderableLayer;
import gov.nasa.worldwind.render.*;

import java.awt.*;
import java.util.*;
import java.util.List;

/**
 * Example of how to draw a path with a text annotation attached to the path. The annotation will adjust to stay in
 * view as long as part of the path is visible in the viewport.
 *
 * @author dcollins
 * @version $Id: PathsWithLabels.java 1171 2013-02-11 21:45:02Z dcollins $
 */
public class PathsWithLabels extends ApplicationTemplate
{
    public static class AppFrame extends ApplicationTemplate.AppFrame
    {
        public AppFrame()
        {
            this.makePaths();
        }

        protected void makePaths()
        {
            RenderableLayer layer = new RenderableLayer();

            ShapeAttributes pathAttrs = new BasicShapeAttributes();
            pathAttrs.setOutlineMaterial(Material.WHITE);
            pathAttrs.setOutlineOpacity(0.8);
            pathAttrs.setOutlineWidth(3);

            List<Position> positions = this.makePositionList(SAN_JUAN_COUNTY_1);
            SurfacePolyline path = new SurfacePolyline(pathAttrs, positions);
            LabeledPath label = new LabeledPath(positions, this.makeLabelAnnotation("Waldron Island"));
            label.setAltitudeMode(WorldWind.CLAMP_TO_GROUND);
            layer.addRenderable(path);
            layer.addRenderable(label);

            positions = this.makePositionList(SAN_JUAN_COUNTY_2);
            path = new SurfacePolyline(pathAttrs, positions);
            label = new LabeledPath(positions,
                this.makeLabelAnnotation("Blakley, Center, Lopez, Orcas, Shaw & San Juan Islands"));
            label.setAltitudeMode(WorldWind.CLAMP_TO_GROUND);
            layer.addRenderable(path);
            layer.addRenderable(label);

            positions = this.makePositionList(SAN_JUAN_COUNTY_3);
            path = new SurfacePolyline(pathAttrs, positions);
            label = new LabeledPath(positions, this.makeLabelAnnotation("Spieden & Stuart Islands"));
            label.setAltitudeMode(WorldWind.CLAMP_TO_GROUND);
            layer.addRenderable(path);
            layer.addRenderable(label);

            this.getWwd().getModel().getLayers().add(layer);
        }

        protected List<Position> makePositionList(double[] src)
        {
            int numCoords = src.length / 2;
            Position[] array = new Position[numCoords];

            for (int i = 0; i < numCoords; i++)
            {
                double lonDegrees = src[2 * i];
                double latDegrees = src[2 * i + 1];
                array[i] = Position.fromDegrees(latDegrees, lonDegrees, 0);
            }

            return Arrays.asList(array);
        }

        protected ScreenAnnotation makeLabelAnnotation(String text)
        {
            ScreenAnnotation ga = new ScreenAnnotation(text, new Point());
            ga.setPickEnabled(false);

            AnnotationAttributes attrs = new AnnotationAttributes();
            attrs.setAdjustWidthToText(AVKey.SIZE_FIT_TEXT);
            attrs.setFrameShape(AVKey.SHAPE_RECTANGLE);
            attrs.setDrawOffset(new Point(0, 10));
            attrs.setLeaderGapWidth(5);
            attrs.setTextColor(Color.BLACK);
            attrs.setBackgroundColor(new Color(1f, 1f, 1f, 0.8f));
            attrs.setCornerRadius(5);
            attrs.setBorderColor(new Color(0xababab));
            attrs.setFont(Font.decode("Arial-PLAIN-12"));
            attrs.setTextAlign(AVKey.CENTER);
            attrs.setInsets(new Insets(5, 5, 5, 5));

            ga.setAttributes(attrs);

            return ga;
        }
    }

    public static void main(String[] args)
    {
        Configuration.setValue(AVKey.INITIAL_LATITUDE, 48.6);
        Configuration.setValue(AVKey.INITIAL_LONGITUDE, -122.95);
        Configuration.setValue(AVKey.INITIAL_ALTITUDE, 70000);

        start("World Wind Paths with Labels", AppFrame.class);
    }

    // Boundary data for San Juan and Snohomish Counties take from
    // http://www.census.gov/geo/cob/bdy/co/co00ascii/co53_d00_ascii.zip

    protected static final double[] SAN_JUAN_COUNTY_1 =
        {
            -0.123025486000000E+03, 0.487179660000000E+02,
            -0.123019699000000E+03, 0.487213120000000E+02,
            -0.123009787000000E+03, 0.487222910000000E+02,
            -0.123007511000000E+03, 0.487188630000000E+02,
            -0.123005086000000E+03, 0.486943420000000E+02,
            -0.123014449000000E+03, 0.486849780000000E+02,
            -0.123021215000000E+03, 0.486814160000000E+02,
            -0.123042337000000E+03, 0.486756630000000E+02,
            -0.123041645000000E+03, 0.486786330000000E+02,
            -0.123035672000000E+03, 0.486853500000000E+02,
            -0.123036360000000E+03, 0.486900800000000E+02,
            -0.123047058000000E+03, 0.486957720000000E+02,
            -0.123070427000000E+03, 0.486999710000000E+02,
            -0.123040179000000E+03, 0.487172960000000E+02,
            -0.123025486000000E+03, 0.487179660000000E+02,
        };
    protected static final double[] SAN_JUAN_COUNTY_2 =
        {
            -0.122906298643435E+03, 0.487142756789313E+02,
            -0.122894599000000E+03, 0.487150300000000E+02,
            -0.122883606017278E+03, 0.487133157521183E+02,
            -0.122879724954535E+03, 0.487127105384329E+02,
            -0.122875938000000E+03, 0.487121200000000E+02,
            -0.122836802504198E+03, 0.486993713019117E+02,
            -0.122834290842953E+03, 0.486985531083446E+02,
            -0.122833124000000E+03, 0.486981730000000E+02,
            -0.122831410808922E+03, 0.486973051154063E+02,
            -0.122825683629670E+03, 0.486944037865925E+02,
            -0.122808804479515E+03, 0.486858529865323E+02,
            -0.122805788100163E+03, 0.486843249204562E+02,
            -0.122802545000000E+03, 0.486826820000000E+02,
            -0.122800267000000E+03, 0.486796200000000E+02,
            -0.122761393485795E+03, 0.486676429790988E+02,
            -0.122744791626046E+03, 0.486625279071718E+02,
            -0.122743049000000E+03, 0.486619910000000E+02,
            -0.122742776245752E+03, 0.486616237548803E+02,
            -0.122742082000000E+03, 0.486606890000000E+02,
            -0.122755031000000E+03, 0.486495120000000E+02,
            -0.122762709974974E+03, 0.486457601003223E+02,
            -0.122774165127982E+03, 0.486401631834124E+02,
            -0.122783875000000E+03, 0.486354190000000E+02,
            -0.122792147000000E+03, 0.486335020000000E+02,
            -0.122806316669653E+03, 0.486217713784910E+02,
            -0.122807583040252E+03, 0.486207229903106E+02,
            -0.122809622000000E+03, 0.486190350000000E+02,
            -0.122809583303260E+03, 0.486188448346238E+02,
            -0.122808864000000E+03, 0.486153100000000E+02,
            -0.122799010000000E+03, 0.486046830000000E+02,
            -0.122798770000000E+03, 0.486023520000000E+02,
            -0.122800217000000E+03, 0.486016900000000E+02,
            -0.122801521572817E+03, 0.486000752686420E+02,
            -0.122804869000000E+03, 0.485959320000000E+02,
            -0.122804736685678E+03, 0.485955635328447E+02,
            -0.122802617338743E+03, 0.485896616038081E+02,
            -0.122801096000000E+03, 0.485854250000000E+02,
            -0.122796913105296E+03, 0.485828999845133E+02,
            -0.122787753320193E+03, 0.485773706559316E+02,
            -0.122786586000000E+03, 0.485766660000000E+02,
            -0.122771206000000E+03, 0.485624260000000E+02,
            -0.122770349000000E+03, 0.485581060000000E+02,
            -0.122772384000000E+03, 0.485521430000000E+02,
            -0.122782618000000E+03, 0.485451910000000E+02,
            -0.122788503000000E+03, 0.485303930000000E+02,
            -0.122787596684858E+03, 0.485246062248558E+02,
            -0.122787347000000E+03, 0.485230120000000E+02,
            -0.122777467000000E+03, 0.485177990000000E+02,
            -0.122779073275586E+03, 0.485091830812273E+02,
            -0.122779124000000E+03, 0.485089110000000E+02,
            -0.122799155517055E+03, 0.484953208059023E+02,
            -0.122799759583816E+03, 0.484949109824973E+02,
            -0.122800414000000E+03, 0.484944670000000E+02,
            -0.122809344417182E+03, 0.484907496395120E+02,
            -0.122816332000000E+03, 0.484878410000000E+02,
            -0.122816392492828E+03, 0.484876896530700E+02,
            -0.122817912000000E+03, 0.484838880000000E+02,
            -0.122818113811431E+03, 0.484811078199731E+02,
            -0.122818435504082E+03, 0.484766761409594E+02,
            -0.122818482000648E+03, 0.484760355983364E+02,
            -0.122819730000000E+03, 0.484588430000000E+02,
            -0.122813100000000E+03, 0.484528560000000E+02,
            -0.122810699656295E+03, 0.484489394154454E+02,
            -0.122807708000000E+03, 0.484440580000000E+02,
            -0.122806716195520E+03, 0.484419671792455E+02,
            -0.122802509000000E+03, 0.484330980000000E+02,
            -0.122803004619265E+03, 0.484309676207466E+02,
            -0.122803326834080E+03, 0.484295826064757E+02,
            -0.122803521000000E+03, 0.484287480000000E+02,
            -0.122806510124273E+03, 0.484265382433425E+02,
            -0.122812173045928E+03, 0.484223518403418E+02,
            -0.122812208000000E+03, 0.484223260000000E+02,
            -0.122825307995692E+03, 0.484240652785748E+02,
            -0.122825803000000E+03, 0.484241310000000E+02,
            -0.122852542484940E+03, 0.484208474850799E+02,
            -0.122863360429977E+03, 0.484195190793902E+02,
            -0.122866475017092E+03, 0.484191366190217E+02,
            -0.122866865338919E+03, 0.484190886888710E+02,
            -0.122867939205531E+03, 0.484189568218194E+02,
            -0.122867996814560E+03, 0.484189497476327E+02,
            -0.122874135000000E+03, 0.484181960000000E+02,
            -0.122874527604161E+03, 0.484182203541858E+02,
            -0.122883759000000E+03, 0.484187930000000E+02,
            -0.122888474856222E+03, 0.484206350791676E+02,
            -0.122893646000000E+03, 0.484226550000000E+02,
            -0.122889016000000E+03, 0.484359470000000E+02,
            -0.122903214000000E+03, 0.484369790000000E+02,
            -0.122913888000000E+03, 0.484432310000000E+02,
            -0.122917771000000E+03, 0.484397810000000E+02,
            -0.122927683717092E+03, 0.484399602096806E+02,
            -0.122928004000000E+03, 0.484399660000000E+02,
            -0.122927964788402E+03, 0.484400111660268E+02,
            -0.122916460000000E+03, 0.484532630000000E+02,
            -0.122920099000000E+03, 0.484584280000000E+02,
            -0.122926901000000E+03, 0.484608740000000E+02,
            -0.122937881000000E+03, 0.484562210000000E+02,
            -0.122950555015899E+03, 0.484535630706048E+02,
            -0.122962009000000E+03, 0.484511610000000E+02,
            -0.123001288000000E+03, 0.484556628590224E+02,
            -0.123026267878952E+03, 0.484585258619868E+02,
            -0.123029013724265E+03, 0.484588405698076E+02,
            -0.123033178635901E+03, 0.484593179201736E+02,
            -0.123038888830654E+03, 0.484599723790898E+02,
            -0.123039156000000E+03, 0.484600030000000E+02,
            -0.123058154000000E+03, 0.484715220000000E+02,
            -0.123067675000000E+03, 0.484794970000000E+02,
            -0.123070563309358E+03, 0.484802266082759E+02,
            -0.123070833040893E+03, 0.484802947444538E+02,
            -0.123076030355349E+03, 0.484816076244710E+02,
            -0.123099948917862E+03, 0.484876496300162E+02,
            -0.123119451000000E+03, 0.484925760000000E+02,
            -0.123129287766067E+03, 0.484982542349179E+02,
            -0.123134915117353E+03, 0.485015026015408E+02,
            -0.123141478000000E+03, 0.485052910000000E+02,
            -0.123144822701730E+03, 0.485083136865329E+02,
            -0.123148640952156E+03, 0.485117643302889E+02,
            -0.123150537321568E+03, 0.485134781244465E+02,
            -0.123151065000000E+03, 0.485139550000000E+02,
            -0.123152147442285E+03, 0.485153416540207E+02,
            -0.123152738173008E+03, 0.485160984048828E+02,
            -0.123163234000000E+03, 0.485295440000000E+02,
            -0.123164057000000E+03, 0.485356220000000E+02,
            -0.123161853000000E+03, 0.485392550000000E+02,
            -0.123161750942688E+03, 0.485414834733667E+02,
            -0.123161470000000E+03, 0.485476180000000E+02,
            -0.123167656573858E+03, 0.485526319405021E+02,
            -0.123172412000000E+03, 0.485564860000000E+02,
            -0.123176266000000E+03, 0.485621310000000E+02,
            -0.123176243708612E+03, 0.485624730166654E+02,
            -0.123175852000000E+03, 0.485684830000000E+02,
            -0.123171958000000E+03, 0.485722550000000E+02,
            -0.123173061000000E+03, 0.485790860000000E+02,
            -0.123184941000000E+03, 0.485869700000000E+02,
            -0.123195467450664E+03, 0.485863505553890E+02,
            -0.123196697396232E+03, 0.485862781774168E+02,
            -0.123197754000000E+03, 0.485862160000000E+02,
            -0.123198045125962E+03, 0.485864522812820E+02,
            -0.123202680000000E+03, 0.485902140000000E+02,
            -0.123203026000000E+03, 0.485961780000000E+02,
            -0.123195725000000E+03, 0.486070550000000E+02,
            -0.123179996129946E+03, 0.486207472995959E+02,
            -0.123178425000000E+03, 0.486221150000000E+02,
            -0.123151643000000E+03, 0.486236860000000E+02,
            -0.123139705244805E+03, 0.486227863453916E+02,
            -0.123135644582323E+03, 0.486201710055677E+02,
            -0.123131377372596E+03, 0.486205150057834E+02,
            -0.123107362000000E+03, 0.486224510000000E+02,
            -0.123101647918870E+03, 0.486162765822218E+02,
            -0.123098626049648E+03, 0.486130112657825E+02,
            -0.123098462000000E+03, 0.486128340000000E+02,
            -0.123098254000000E+03, 0.486100920000000E+02,
            -0.123102074000000E+03, 0.486040350000000E+02,
            -0.123101552000000E+03, 0.485978200000000E+02,
            -0.123100979011718E+03, 0.485976923053470E+02,
            -0.123087991970190E+03, 0.485947980476233E+02,
            -0.123079379694944E+03, 0.485928787387418E+02,
            -0.123079368517922E+03, 0.485928762478604E+02,
            -0.123079334903513E+03, 0.485928687566419E+02,
            -0.123078354245955E+03, 0.485926502098924E+02,
            -0.123078029433353E+03, 0.485925778230151E+02,
            -0.123077161967252E+03, 0.485923845018144E+02,
            -0.123074727135920E+03, 0.485918418817440E+02,
            -0.123074611000000E+03, 0.485918160000000E+02,
            -0.123060040000000E+03, 0.485821050000000E+02,
            -0.123056818600556E+03, 0.485785370170628E+02,
            -0.123055637295811E+03, 0.485772286182611E+02,
            -0.123049834129163E+03, 0.485708011012966E+02,
            -0.123048403000000E+03, 0.485692160000000E+02,
            -0.123037067007025E+03, 0.485647482308128E+02,
            -0.123033669000000E+03, 0.485634090000000E+02,
            -0.123025587194622E+03, 0.485622858881320E+02,
            -0.123015046000000E+03, 0.485608210000000E+02,
            -0.123014957175286E+03, 0.485608244377565E+02,
            -0.123005887152572E+03, 0.485611754721491E+02,
            -0.122994982694655E+03, 0.485615975041420E+02,
            -0.122987296000000E+03, 0.485618950000000E+02,
            -0.122986110000000E+03, 0.485699840000000E+02,
            -0.122989649000000E+03, 0.485746680000000E+02,
            -0.122995026000000E+03, 0.485781620000000E+02,
            -0.123001291973563E+03, 0.485798454915671E+02,
            -0.123003071284953E+03, 0.485803235426935E+02,
            -0.123004800000000E+03, 0.485807880000000E+02,
            -0.123016647000000E+03, 0.485802440000000E+02,
            -0.123023206615953E+03, 0.485845746092944E+02,
            -0.123028982347697E+03, 0.485883877041088E+02,
            -0.123033093576815E+03, 0.485911019067628E+02,
            -0.123034101000000E+03, 0.485917670000000E+02,
            -0.123024902000000E+03, 0.485944840000000E+02,
            -0.123023433000000E+03, 0.485994770000000E+02,
            -0.123041189000000E+03, 0.486119470000000E+02,
            -0.123046453150843E+03, 0.486114965755598E+02,
            -0.123046530000000E+03, 0.486114900000000E+02,
            -0.123048652000000E+03, 0.486210020000000E+02,
            -0.123040078194353E+03, 0.486254322142389E+02,
            -0.123023495000000E+03, 0.486340010000000E+02,
            -0.123015592000000E+03, 0.486425670000000E+02,
            -0.123014829000000E+03, 0.486475030000000E+02,
            -0.123009977399319E+03, 0.486549816855750E+02,
            -0.123009924000000E+03, 0.486550640000000E+02,
            -0.123001295006939E+03, 0.486600601213626E+02,
            -0.122999889835340E+03, 0.486608737050326E+02,
            -0.122988884000000E+03, 0.486672460000000E+02,
            -0.122984853000000E+03, 0.486726860000000E+02,
            -0.122970595802002E+03, 0.486809490071059E+02,
            -0.122949116000000E+03, 0.486933980000000E+02,
            -0.122941316000000E+03, 0.487029040000000E+02,
            -0.122942367000000E+03, 0.487067230000000E+02,
            -0.122927109877131E+03, 0.487110138483226E+02,
            -0.122918837934242E+03, 0.487133402143466E+02,
            -0.122918252000000E+03, 0.487135050000000E+02,
            -0.122908658361456E+03, 0.487141235388230E+02,
            -0.122907454714043E+03, 0.487142011426071E+02,
            -0.122906298643435E+03, 0.487142756789313E+02,
        };
    protected static final double[] SAN_JUAN_COUNTY_3 =
        {
            -0.123172066000000E+03, 0.486798660000000E+02,
            -0.123147990000000E+03, 0.486680010000000E+02,
            -0.123133285875321E+03, 0.486583191438863E+02,
            -0.123130962000000E+03, 0.486567890000000E+02,
            -0.123122016049879E+03, 0.486470646274710E+02,
            -0.123106165000000E+03, 0.486334730000000E+02,
            -0.123119677246557E+03, 0.486329724058636E+02,
            -0.123134956336931E+03, 0.486372395392956E+02,
            -0.123170932260699E+03, 0.486515091289271E+02,
            -0.123197404105632E+03, 0.486620089945159E+02,
            -0.123215917000000E+03, 0.486693520000000E+02,
            -0.123229744313989E+03, 0.486785441581480E+02,
            -0.123233070031572E+03, 0.486807550374269E+02,
            -0.123237148000000E+03, 0.486834660000000E+02,
            -0.123237135383772E+03, 0.486835850832961E+02,
            -0.123236567000000E+03, 0.486889500000000E+02,
            -0.123213499856684E+03, 0.486896934099409E+02,
            -0.123212892000000E+03, 0.486897130000000E+02,
            -0.123212464368268E+03, 0.486895683569085E+02,
            -0.123202992056872E+03, 0.486863644216061E+02,
            -0.123197953000000E+03, 0.486846600000000E+02,
            -0.123186997647353E+03, 0.486848970569698E+02,
            -0.123186076000000E+03, 0.486849170000000E+02,
            -0.123184143390981E+03, 0.486842202399606E+02,
            -0.123183803136584E+03, 0.486840975686572E+02,
            -0.123172066000000E+03, 0.486798660000000E+02,
        };
}
