/*
 * Copyright (c) 2004, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package org.jboss.com.sun.corba.se.impl.orbutil;

/**
 * A convenience class for retrieving the string value of a system property as a privileged action. This class is
 * directly copied from sun.security.action.GetPropertyAction in order to avoid depending on the sun.security.action
 * package.
 * 
 * <p>
 * An instance of this class can be used as the argument of <code>AccessController.doPrivileged</code>.
 * 
 * <p>
 * The following code retrieves the value of the system property named <code>"prop"</code> as a privileged action:
 * <p>
 * 
 * <pre>
 * String s = (String) java.security.AccessController.doPrivileged(new GetPropertyAction(&quot;prop&quot;));
 * </pre>
 * 
 * @author Roland Schemers
 * @author Ken Cavanaugh
 * @see java.security.PrivilegedAction
 * @see java.security.AccessController
 */

public class GetPropertyAction implements java.security.PrivilegedAction<String>
{
    private String theProp;

    private String defaultVal;

    /**
     * Constructor that takes the name of the system property whose string value needs to be determined.
     * 
     * @param theProp
     *            the name of the system property.
     */
    public GetPropertyAction(String theProp)
    {
        this.theProp = theProp;
    }

    /**
     * Constructor that takes the name of the system property and the default value of that property.
     * 
     * @param theProp
     *            the name of the system property.
     * @param defaulVal
     *            the default value.
     */
    public GetPropertyAction(String theProp, String defaultVal)
    {
        this.theProp = theProp;
        this.defaultVal = defaultVal;
    }

    /**
     * Determines the string value of the system property whose name was specified in the constructor.
     * 
     * @return the string value of the system property, or the default value if there is no property with that key.
     */
    public String run()
    {
        String value = System.getProperty(theProp);
        return (value == null) ? defaultVal : value;
    }
}
